package ru.yandex.direct.grid.processing.service.group

import com.google.common.base.Preconditions.checkState
import io.leangen.graphql.annotations.GraphQLArgument
import io.leangen.graphql.annotations.GraphQLContext
import io.leangen.graphql.annotations.GraphQLMutation
import io.leangen.graphql.annotations.GraphQLNonNull
import io.leangen.graphql.annotations.GraphQLQuery
import io.leangen.graphql.annotations.GraphQLRootContext
import ru.yandex.direct.core.entity.adgroup.container.InternalAdGroupOperationContainer
import ru.yandex.direct.core.entity.adgroup.container.InternalAdGroupOperationContainer.RequestSource
import ru.yandex.direct.core.entity.adgroup.container.InternalAdGroupUpdateItem
import ru.yandex.direct.core.entity.adgroup.model.InternalAdGroup
import ru.yandex.direct.core.entity.adgroup.service.complex.internal.ComplexInternalAdGroupService
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.AdGroupAdditionalTargeting
import ru.yandex.direct.core.entity.internalads.service.InternalAdsProductService
import ru.yandex.direct.core.security.authorization.PreAuthorizeRead
import ru.yandex.direct.core.security.authorization.PreAuthorizeWrite
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.model.UidAndClientId
import ru.yandex.direct.grid.processing.annotations.EnableLoggingOnValidationIssues
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext
import ru.yandex.direct.grid.processing.model.client.GdClient
import ru.yandex.direct.grid.processing.model.group.GdInternalAdGroupFieldChange
import ru.yandex.direct.grid.processing.model.group.GdInternalAdGroupFieldChangeOperation
import ru.yandex.direct.grid.processing.model.group.GdInternalAdGroupFieldState
import ru.yandex.direct.grid.processing.model.group.GdInternalAdGroupFieldStateValue
import ru.yandex.direct.grid.processing.model.group.GdInternalAdGroupsAggregatedState
import ru.yandex.direct.grid.processing.model.group.GdInternalAdGroupsAggregatedStatePayload
import ru.yandex.direct.grid.processing.model.group.GdInternalAdGroupsMassUpdate
import ru.yandex.direct.grid.processing.model.group.GdInternalAdGroupsMassUpdatePayload
import ru.yandex.direct.grid.processing.service.group.internalad.InternalAdGroupMassUpdateFieldSupport
import ru.yandex.direct.grid.processing.service.group.internalad.InternalAdGroupWithTargeting
import ru.yandex.direct.grid.processing.service.group.internalad.InternalAdGroupsLoader
import ru.yandex.direct.grid.processing.service.group.internalad.getSingleValueFromUnion
import ru.yandex.direct.grid.processing.service.group.validation.AdGroupInternalMassUpdateValidationService
import ru.yandex.direct.grid.processing.service.internalad.MassUpdateFieldSupport
import ru.yandex.direct.grid.processing.service.internalad.groupObjectsByValue
import ru.yandex.direct.model.ModelChanges
import ru.yandex.direct.operation.Applicability
import ru.yandex.direct.result.MassResult

/**
 * Набор ручек для массового обновления параметров и таргетингов групп внутренней рекламы
 */
@GridGraphQLService
open class AdGroupInternalMassUpdateGraphQlService(
    private val internalAdGroupsLoader: InternalAdGroupsLoader,
    private val complexInternalAdGroupService: ComplexInternalAdGroupService,
    private val validationService: AdGroupInternalMassUpdateValidationService,
    private val massUpdateFieldSupports: List<InternalAdGroupMassUpdateFieldSupport<*>>,
    private val internalAdsProductService: InternalAdsProductService,
) {

    @PreAuthorizeRead
    @GraphQLQuery(name = "internalAdGroupsAggregatedState")
    open fun internalAdGroupsAggregatedState(
        @GraphQLRootContext context: GridGraphQLContext,
        @GraphQLContext client: GdClient,
        @GraphQLArgument(name = "input") input: GdInternalAdGroupsAggregatedState
    ): GdInternalAdGroupsAggregatedStatePayload {
        val clientInfo = context.queriedClient ?: throw IllegalStateException("No queriedClient in graphql context")
        val clientId = ClientId.fromLong(clientInfo.id)
        checkState(
            internalAdsProductService.clientCanHaveInternalAdCampaigns(clientId),
            "Internal ads are not supported for client"
        )
        validationService.validateInternalAdGroupsAggregatedState(input)

        val internalAdGroups = internalAdGroupsLoader.load(clientId, input.adGroupIds)

        // сбор агрегированного состояния
        return GdInternalAdGroupsAggregatedStatePayload().apply {
            adGroupIds = internalAdGroups.map { it.adGroup.id }
            aggregatedState = massUpdateFieldSupports.flatMap {
                aggregateAdGroupFieldState(internalAdGroups, it)
            }
        }
    }

    private fun <V, G : GdInternalAdGroupFieldStateValue> aggregateAdGroupFieldState(
        objects: Collection<InternalAdGroupWithTargeting>,
        fieldSupport: MassUpdateFieldSupport<InternalAdGroupWithTargeting, V, G, *>
    ): List<GdInternalAdGroupFieldState> {
        return groupObjectsByValue(objects, fieldSupport.operations::extract)
            .map { valueGroup ->
                GdInternalAdGroupFieldState().apply {
                    ids = valueGroup.objects.map { it.adGroup.id }
                    value = fieldSupport.convertToGdFieldState(valueGroup.value)
                }
            }
    }

    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "internalAdGroupsMassUpdate")
    open fun internalAdGroupsMassUpdate(
        @GraphQLRootContext context: GridGraphQLContext,
        @GraphQLArgument(name = "input") input: @GraphQLNonNull GdInternalAdGroupsMassUpdate
    ): @GraphQLNonNull GdInternalAdGroupsMassUpdatePayload {
        val clientId = context.subjectUser?.clientId
            ?: throw IllegalStateException("No subjectUser in graphql context")
        val subjectUid = context.subjectUser?.uid
            ?: throw IllegalStateException("No subjectUser in graphql context")

        val owner = UidAndClientId.of(subjectUid, clientId)
        checkState(
            internalAdsProductService.clientCanHaveInternalAdCampaigns(owner.clientId),
            "Internal ads are not supported for client"
        )
        validationService.validateInternalAdGroupsMassUpdate(input)

        val affectedAdGroupIds = input.changes.asSequence().flatMap { it.ids }.toSet()
        val internalAdGroupIndex = internalAdGroupsLoader.load(clientId, affectedAdGroupIds)
            .associateBy { it.adGroup.id }

        input.changes.forEach { change ->
            applySingleChange(change, massUpdateFieldSupports, internalAdGroupIndex)
        }

        val coreUpdateItems: List<InternalAdGroupUpdateItem> = internalAdGroupIndex.values.map {
            val adGroup = it.adGroup
            val targetings: List<AdGroupAdditionalTargeting> = it.targetings.toList()
            InternalAdGroupUpdateItem().apply {
                adGroupChanges = ModelChanges(adGroup.id, InternalAdGroup::class.java)
                    .process(adGroup.name, InternalAdGroup.NAME)
                    .process(adGroup.level, InternalAdGroup.LEVEL)
                    .process(adGroup.rf, InternalAdGroup.RF)
                    .process(adGroup.rfReset, InternalAdGroup.RF_RESET)
                    .process(adGroup.maxClicksCount, InternalAdGroup.MAX_CLICKS_COUNT)
                    .process(adGroup.maxClicksPeriod, InternalAdGroup.MAX_CLICKS_PERIOD)
                    .process(adGroup.maxStopsCount, InternalAdGroup.MAX_STOPS_COUNT)
                    .process(adGroup.maxStopsPeriod, InternalAdGroup.MAX_STOPS_PERIOD)
                    .process(adGroup.startTime, InternalAdGroup.START_TIME)
                    .process(adGroup.finishTime, InternalAdGroup.FINISH_TIME)
                    .process(adGroup.geo, InternalAdGroup.GEO)
                additionalTargetings = targetings
            }
        }
        val operationContainer = InternalAdGroupOperationContainer(
            Applicability.FULL,
            context.operator.uid,
            owner,
            true,
            RequestSource.FRONT
        )
        val result: MassResult<Long> =
            complexInternalAdGroupService.getUpdateOperation(operationContainer, coreUpdateItems).prepareAndApply()

        return GdInternalAdGroupsMassUpdatePayload().withValidationResult(
            validationService.getValidationResult(result.validationResult)
        )
    }

    private fun applySingleChange(
        change: GdInternalAdGroupFieldChange,
        massUpdateFieldSupports: List<InternalAdGroupMassUpdateFieldSupport<*>>,
        internalAdGroupIndex: Map<Long, InternalAdGroupWithTargeting>
    ) {
        val value = change.value.getSingleValueFromUnion()
        val adGroups = change.ids.mapNotNull { internalAdGroupIndex[it] }
        when (change.operation!!) {
            GdInternalAdGroupFieldChangeOperation.ADD -> massUpdateFieldSupports.forEach {
                it.applyAdd(adGroups, value)
            }
            GdInternalAdGroupFieldChangeOperation.REMOVE -> massUpdateFieldSupports.forEach {
                it.applyRemove(adGroups, value)
            }
        }
    }
}
