package ru.yandex.direct.grid.processing.service.group;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Sets;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.addition.callout.service.CalloutService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.grid.processing.model.api.GdValidationResult;
import ru.yandex.direct.grid.processing.model.group.mutation.GdChangeAdGroupsRelevanceMatch;
import ru.yandex.direct.grid.processing.model.group.mutation.GdChangeAdGroupsRelevanceMatchPayload;
import ru.yandex.direct.grid.processing.model.group.mutation.GdCopyAdGroups;
import ru.yandex.direct.grid.processing.model.group.mutation.GdCopyAdGroupsPayload;
import ru.yandex.direct.grid.processing.model.group.mutation.GdRemoderateAdsCallouts;
import ru.yandex.direct.grid.processing.model.group.mutation.GdRemoderateAdsCalloutsPayload;
import ru.yandex.direct.grid.processing.service.group.validation.AdGroupCopyValidationService;
import ru.yandex.direct.grid.processing.service.group.validation.AdGroupMassActionsValidationService;
import ru.yandex.direct.grid.processing.service.validation.GridValidationResultConversionService;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.function.Function.identity;
import static ru.yandex.direct.grid.processing.util.ResponseConverter.getSuccessfullyResults;
import static ru.yandex.direct.utils.FunctionalUtils.flatMapToSet;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.PathHelper.path;

@Service
@ParametersAreNonnullByDefault
public class AdGroupMassActionsMutationService {

    private final AdGroupMassActionsValidationService validationService;
    private final AdGroupCopyValidationService copyValidationService;
    private final CalloutService calloutService;
    private final CopyAdGroupsService copyAdGroupsService;
    private final AdGroupRelevanceMatchService adGroupRelevanceMatchService;
    private final GridValidationService gridValidationService;
    private final GridValidationResultConversionService validationResultConverter;

    @Autowired
    public AdGroupMassActionsMutationService(
            AdGroupMassActionsValidationService validationService,
            AdGroupCopyValidationService copyValidationService,
            CalloutService calloutService,
            CopyAdGroupsService copyAdGroupsService, AdGroupRelevanceMatchService adGroupRelevanceMatchService,
            GridValidationService gridValidationService,
            GridValidationResultConversionService validationResultConverter) {
        this.validationService = validationService;
        this.copyValidationService = copyValidationService;
        this.calloutService = calloutService;
        this.copyAdGroupsService = copyAdGroupsService;
        this.adGroupRelevanceMatchService = adGroupRelevanceMatchService;
        this.gridValidationService = gridValidationService;
        this.validationResultConverter = validationResultConverter;
    }

    GdRemoderateAdsCalloutsPayload remoderateAdsCallouts(User operator, ClientId clientId,
                                                         GdRemoderateAdsCallouts input) {
        validationService.validateRemoderateAdsCalloutsRequest(operator, input);

        Map<Long, List<Long>> existingCalloutIdsByAdGroupIds =
                calloutService.getExistingCalloutIdsByAdGroupIds(clientId, input.getAdGroupIds());
        flatMapToSet(existingCalloutIdsByAdGroupIds.values(), identity());
        Set<Long> allCalloutIds = flatMapToSet(existingCalloutIdsByAdGroupIds.values(), identity());

        calloutService.remoderateCallouts(clientId, allCalloutIds, input.getModerateAccept());

        Set<Long> processedAdGroupIds = existingCalloutIdsByAdGroupIds.keySet();
        Set<Long> skippedAdGroupIds = Sets.difference(input.getAdGroupIds(), processedAdGroupIds);
        return new GdRemoderateAdsCalloutsPayload()
                .withProcessedAdGroupIds(processedAdGroupIds)
                .withSkippedAdGroupIds(skippedAdGroupIds);
    }

    GdCopyAdGroupsPayload copyAdGroups(User operator, UidAndClientId uidAndClientId, GdCopyAdGroups input) {
        validationService.validateCopyAdGroupsRequest(operator, input);

        GdValidationResult preValidationResult =
                copyValidationService.validateCopyAdGroups(uidAndClientId.getClientId(), input);
        if (preValidationResult != null) {
            return new GdCopyAdGroupsPayload()
                    .withCopiedAdGroupIds(Collections.emptyList())
                    .withValidationResult(preValidationResult);
        }

        MassResult<Long> result = copyAdGroupsService.copyAdGroups(operator.getUid(), uidAndClientId,
                input.getAdGroupIds(), input.getDestinationCampaignId());

        GdValidationResult validationResult =
                copyValidationService.getValidationResult(result, path(field(GdCopyAdGroups.AD_GROUP_IDS)));
        List<Long> copiedAdGroupIds = getSuccessfullyResults(result, identity());

        return new GdCopyAdGroupsPayload()
                .withCopiedAdGroupIds(copiedAdGroupIds)
                .withValidationResult(validationResult);
    }

    GdChangeAdGroupsRelevanceMatchPayload changeAdGroupsRelevanceMatch(User operator, ClientId clientId,
                                                                       GdChangeAdGroupsRelevanceMatch input) {
        validationService.validateChangeAdGroupsRelevanceMatch(operator, input);

        ValidationResult<List<Long>, Defect> result =
                adGroupRelevanceMatchService.changeAdGroupsRelevanceMatch(operator.getUid(), clientId,
                        input.getAdGroupIds(), input.getEnableRelevanceMatch());

        GdValidationResult gdValidationResult = validationResultConverter.buildGridValidationResult(result);

        List<Long> validItems = ValidationResult.getValidItems(result);

        return new GdChangeAdGroupsRelevanceMatchPayload()
                .withValidationResult(gdValidationResult)
                .withUpdatedAdGroupIds(validItems)
                .withSuccessCount(validItems.size())
                .withTotalCount(result.getValue().size());
    }
}
