package ru.yandex.direct.grid.processing.service.group;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Multimap;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.relevancematch.model.RelevanceMatch;
import ru.yandex.direct.core.entity.relevancematch.repository.RelevanceMatchRepository;
import ru.yandex.direct.core.entity.relevancematch.service.RelevanceMatchAddOperation;
import ru.yandex.direct.core.entity.relevancematch.service.RelevanceMatchDeleteOperation;
import ru.yandex.direct.core.entity.relevancematch.service.RelevanceMatchService;
import ru.yandex.direct.core.entity.showcondition.container.ShowConditionAutoPriceParams;
import ru.yandex.direct.core.entity.showcondition.container.ShowConditionFixedAutoPrices;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.operation.AddedModelId;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.ytcore.entity.statistics.service.RecentStatisticsService;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Collections.emptyMap;
import static java.util.function.Predicate.not;
import static ru.yandex.direct.operation.tree.ItemSubOperationExecutor.extractSubList;
import static ru.yandex.direct.operation.tree.TreeOperationUtils.mergeSubListValidationResults;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class AdGroupRelevanceMatchService {

    private final RelevanceMatchService relevanceMatchService;
    private final RelevanceMatchRepository relevanceMatchRepository;
    private final RecentStatisticsService recentStatisticsService;
    private final ClientService clientService;
    private final ShardHelper shardHelper;

    @Autowired
    public AdGroupRelevanceMatchService(RelevanceMatchService relevanceMatchService,
                                        RelevanceMatchRepository relevanceMatchRepository,
                                        RecentStatisticsService recentStatisticsService, ClientService clientService,
                                        ShardHelper shardHelper) {
        this.relevanceMatchService = relevanceMatchService;
        this.relevanceMatchRepository = relevanceMatchRepository;
        this.recentStatisticsService = recentStatisticsService;
        this.clientService = clientService;
        this.shardHelper = shardHelper;
    }

    ValidationResult<List<Long>, Defect> changeAdGroupsRelevanceMatch(Long operatorUid, ClientId clientId,
                                                                      List<Long> adGroupIds,
                                                                      Boolean relevanceMatchIsEnable) {
        int shard = shardHelper.getShardByClientId(clientId);

        if (relevanceMatchIsEnable) {
            return enableRelevanceMatch(shard, operatorUid, clientId, adGroupIds);
        } else {
            return disableRelevanceMatch(shard, operatorUid, clientId, adGroupIds);
        }
    }

    private ValidationResult<List<Long>, Defect> enableRelevanceMatch(int shard,
                                                                      Long operatorUid, ClientId clientId,
                                                                      List<Long> adGroupIds) {
        Multimap<Long, Long> relevanceMatchesByAdGroupIds =
                relevanceMatchRepository.getRelevanceMatchIdsByAdGroupIds(shard, clientId,
                        adGroupIds);

        HashMap<Integer, Integer> indexMap = new HashMap<>();
        List<Long> affectedAdGroupIds = extractSubList(indexMap, adGroupIds,
                not(relevanceMatchesByAdGroupIds::containsKey));

        List<RelevanceMatch> relevanceMatchesToAdd = mapList(affectedAdGroupIds,
                id -> new RelevanceMatch().withAdGroupId(id));

        ShowConditionFixedAutoPrices fixedAutoPrices = ShowConditionFixedAutoPrices.ofPerAdGroupFixedPrices(emptyMap());
        ShowConditionAutoPriceParams showConditionAutoPriceParams = new ShowConditionAutoPriceParams(fixedAutoPrices,
                recentStatisticsService);

        Client client = checkNotNull(clientService.getClient(clientId));

        RelevanceMatchAddOperation addOperation =
                relevanceMatchService.createPartialAddOperationWithAutoPrices(
                        client.getWorkCurrency().getCurrency(), clientId, operatorUid,
                        relevanceMatchesToAdd, showConditionAutoPriceParams);

        MassResult<AddedModelId> operationResult = addOperation.prepareAndApply();
        return prepareValidationResult(adGroupIds, indexMap, operationResult);
    }

    private ValidationResult<List<Long>, Defect> disableRelevanceMatch(int shard,
                                                                       Long operatorUid, ClientId clientId,
                                                                       List<Long> adGroupIds) {
        Map<Long, RelevanceMatch> relevanceMatchesByAdGroupIds =
                relevanceMatchRepository.getRelevanceMatchesByAdGroupIds(shard, clientId,
                        adGroupIds);

        HashMap<Integer, Integer> indexMap = new HashMap<>();

        List<Long> affectedRelevanceMatchIds = extractSubList(indexMap, adGroupIds,
                relevanceMatchesByAdGroupIds::containsKey,
                adGroupId -> relevanceMatchesByAdGroupIds.get(adGroupId).getId());

        RelevanceMatchDeleteOperation deleteOperation =
                relevanceMatchService.createDeleteOperation(clientId, operatorUid,
                        affectedRelevanceMatchIds,
                        Applicability.PARTIAL);

        MassResult<Long> operationResult = deleteOperation.prepareAndApply();
        return prepareValidationResult(adGroupIds, indexMap, operationResult);
    }

    private ValidationResult<List<Long>, Defect> prepareValidationResult(List<Long> adGroupIds,
                                                                         HashMap<Integer, Integer> indexMap,
                                                                         MassResult<?> operationResult) {
        //noinspection unchecked
        ValidationResult<List<Long>, Defect> operationVr =
                (ValidationResult<List<Long>, Defect>) operationResult.getValidationResult();

        ValidationResult<List<Long>, Defect> vr = new ValidationResult<>(adGroupIds);
        mergeSubListValidationResults(vr, operationVr, indexMap);
        return vr;
    }
}
