package ru.yandex.direct.grid.processing.service.group;

import java.util.LinkedHashSet;
import java.util.Set;

import com.google.common.collect.ImmutableBiMap;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.ContentPromotionAdgroupType;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.grid.model.campaign.GdCampaignType;
import ru.yandex.direct.grid.model.entity.adgroup.GdAdGroupType;
import ru.yandex.direct.grid.processing.model.group.mutation.GdContentPromotionGroupType;
import ru.yandex.direct.grid.processing.model.group.mutation.GdCpmGroupType;

import static java.lang.String.format;
import static ru.yandex.direct.feature.FeatureName.CONTENT_PROMOTION_COLLECTIONS_ON_GRID;
import static ru.yandex.direct.feature.FeatureName.CONTENT_PROMOTION_EDA_INTERFACE;
import static ru.yandex.direct.feature.FeatureName.CONTENT_PROMOTION_SERVICES_ON_GRID;
import static ru.yandex.direct.feature.FeatureName.CONTENT_PROMOTION_VIDEO_ON_GRID;
import static ru.yandex.direct.feature.FeatureName.CPM_AUDIO_GROUPS_EDIT_FOR_DNA;
import static ru.yandex.direct.feature.FeatureName.CPM_GEOPRODUCT_ENABLED;
import static ru.yandex.direct.feature.FeatureName.CPM_GEO_PIN_PRODUCT_ENABLED;
import static ru.yandex.direct.feature.FeatureName.CPM_INDOOR_GROUPS_EDIT_FOR_DNA;
import static ru.yandex.direct.feature.FeatureName.CPM_OUTDOOR_GROUPS_EDIT_FOR_DNA;
import static ru.yandex.direct.feature.FeatureName.CPM_YNDX_FRONTPAGE_ON_GRID;
import static ru.yandex.direct.feature.FeatureName.SHOW_CPM_AUDIO_GROUPS_IN_GRID;
import static ru.yandex.direct.feature.FeatureName.SHOW_CPM_GEOPRODUCT_GROUPS_IN_GRID;
import static ru.yandex.direct.feature.FeatureName.SHOW_CPM_INDOOR_GROUPS_IN_GRID;
import static ru.yandex.direct.feature.FeatureName.SHOW_CPM_OUTDOOR_GROUPS_IN_GRID;
import static ru.yandex.direct.feature.FeatureName.SHOW_CPM_PRICE_CAMPAIGNS_IN_GRID;
import static ru.yandex.direct.feature.FeatureName.SHOW_CPM_VIDEO_GROUPS_IN_GRID;
import static ru.yandex.direct.grid.processing.model.group.mutation.GdContentPromotionGroupType.COLLECTION;
import static ru.yandex.direct.grid.processing.model.group.mutation.GdContentPromotionGroupType.VIDEO;

public class AdGroupTypeUtils {

    public static final ImmutableSet<AdGroupType> VISIBLE_GROUP_TYPES = ImmutableSet.<AdGroupType>builder()
            .add(AdGroupType.BASE)
            .add(AdGroupType.DYNAMIC)
            .add(AdGroupType.MOBILE_CONTENT)
            .add(AdGroupType.PERFORMANCE)
            .add(AdGroupType.MCBANNER)
            .add(AdGroupType.CPM_BANNER)
            .add(AdGroupType.INTERNAL)
            .build();

    public static final ImmutableMap<FeatureName, AdGroupType> VISIBLE_GROUP_TYPES_BY_FEATURES
            = ImmutableMap.<FeatureName, AdGroupType>builder()
            .put(SHOW_CPM_VIDEO_GROUPS_IN_GRID, AdGroupType.CPM_VIDEO)
            .put(SHOW_CPM_AUDIO_GROUPS_IN_GRID, AdGroupType.CPM_AUDIO)
            .put(SHOW_CPM_OUTDOOR_GROUPS_IN_GRID, AdGroupType.CPM_OUTDOOR)
            .put(SHOW_CPM_INDOOR_GROUPS_IN_GRID, AdGroupType.CPM_INDOOR)
            .put(SHOW_CPM_GEOPRODUCT_GROUPS_IN_GRID, AdGroupType.CPM_GEOPRODUCT)
            .put(CPM_GEO_PIN_PRODUCT_ENABLED, AdGroupType.CPM_GEO_PIN)
            .put(CONTENT_PROMOTION_VIDEO_ON_GRID, AdGroupType.CONTENT_PROMOTION)
            .put(CONTENT_PROMOTION_COLLECTIONS_ON_GRID, AdGroupType.CONTENT_PROMOTION)
            .put(CONTENT_PROMOTION_SERVICES_ON_GRID, AdGroupType.CONTENT_PROMOTION)
            .put(CONTENT_PROMOTION_EDA_INTERFACE, AdGroupType.CONTENT_PROMOTION)
            .put(CPM_YNDX_FRONTPAGE_ON_GRID, AdGroupType.CPM_YNDX_FRONTPAGE)
            .put(SHOW_CPM_PRICE_CAMPAIGNS_IN_GRID, AdGroupType.CPM_YNDX_FRONTPAGE)
            .build();

    public static final ImmutableMap<AdGroupType, FeatureName> EDITABLE_GROUP_FEATURES_BY_TYPE
            = ImmutableMap.<AdGroupType, FeatureName>builder()
            .put(AdGroupType.CPM_GEOPRODUCT, CPM_GEOPRODUCT_ENABLED)
            .put(AdGroupType.CPM_GEO_PIN, CPM_GEO_PIN_PRODUCT_ENABLED)
            .put(AdGroupType.CPM_INDOOR, CPM_INDOOR_GROUPS_EDIT_FOR_DNA)
            .put(AdGroupType.CPM_OUTDOOR, CPM_OUTDOOR_GROUPS_EDIT_FOR_DNA)
            .put(AdGroupType.CPM_AUDIO, CPM_AUDIO_GROUPS_EDIT_FOR_DNA)
            .build();

    public static final ImmutableSet<GdAdGroupType> VISIBLE_SYNTHETIC_GROUP_TYPES = ImmutableSet.<GdAdGroupType>builder()
            .build();

    public static final ImmutableMap<FeatureName, GdAdGroupType> VISIBLE_SYNTHETIC_GROUP_TYPE_BY_FEATURE
            = ImmutableMap.<FeatureName, GdAdGroupType>builder()
            // CONTENT_PROMOTION_COLLECTION должен обязательно должен быть первым (DIRECT-111469)
            .put(CONTENT_PROMOTION_COLLECTIONS_ON_GRID, GdAdGroupType.CONTENT_PROMOTION_COLLECTION)
            .put(CONTENT_PROMOTION_VIDEO_ON_GRID, GdAdGroupType.CONTENT_PROMOTION_VIDEO)
            .put(CONTENT_PROMOTION_SERVICES_ON_GRID, GdAdGroupType.CONTENT_PROMOTION_SERVICE)
            .put(CONTENT_PROMOTION_EDA_INTERFACE, GdAdGroupType.CONTENT_PROMOTION_EDA)

            .put(CPM_YNDX_FRONTPAGE_ON_GRID, GdAdGroupType.CPM_YNDX_FRONTPAGE)
            .put(SHOW_CPM_PRICE_CAMPAIGNS_IN_GRID, GdAdGroupType.CPM_PRICE)

            .build();

    public static final ImmutableSet<GdAdGroupType> ALL_SYNTHETIC_GROUP_TYPES = ImmutableSet.<GdAdGroupType>builder()
            .addAll(VISIBLE_SYNTHETIC_GROUP_TYPES)
            .addAll(VISIBLE_SYNTHETIC_GROUP_TYPE_BY_FEATURE.values())
            .build();

    public static final ImmutableMap<AdGroupType, GdCpmGroupType> CPM_GROUP_TYPE_BY_ADGROUP_TYPE
            = ImmutableMap.<AdGroupType, GdCpmGroupType>builder()
            .put(AdGroupType.CPM_VIDEO, GdCpmGroupType.CPM_VIDEO)
            .put(AdGroupType.CPM_OUTDOOR, GdCpmGroupType.CPM_OUTDOOR)
            .put(AdGroupType.CPM_INDOOR, GdCpmGroupType.CPM_INDOOR)
            .put(AdGroupType.CPM_GEOPRODUCT, GdCpmGroupType.CPM_GEOPRODUCT)
            .put(AdGroupType.CPM_GEO_PIN, GdCpmGroupType.CPM_GEO_PIN)
            .put(AdGroupType.CPM_AUDIO, GdCpmGroupType.CPM_AUDIO)
            .put(AdGroupType.CPM_BANNER, GdCpmGroupType.CPM_BANNER)
            .put(AdGroupType.CPM_YNDX_FRONTPAGE, GdCpmGroupType.CPM_YNDX_FRONTPAGE)
            .build();

    public static final ImmutableMap<GdCpmGroupType, AdGroupType> ADGROUP_TYPE_BY_CPM_GROUP_TYPE = ImmutableMap.<GdCpmGroupType, AdGroupType>builder()
            .putAll(ImmutableBiMap.copyOf(CPM_GROUP_TYPE_BY_ADGROUP_TYPE).inverse())
            .put(GdCpmGroupType.CPM_PRICE, AdGroupType.CPM_YNDX_FRONTPAGE)
            .build();

    public static final ImmutableMap<GdContentPromotionGroupType, ContentPromotionAdgroupType>
            CONTENT_PROMOTION_ADGROUP_CORE_TYPE_BY_GD_TYPE =
            ImmutableMap.<GdContentPromotionGroupType, ContentPromotionAdgroupType>builder()
                    .put(COLLECTION, ContentPromotionAdgroupType.COLLECTION)
                    .put(VIDEO, ContentPromotionAdgroupType.VIDEO)
                    .build();

    public static Set<AdGroupType> getVisibleAdGroupTypes(Set<String> enabledFeatures) {
        Set<AdGroupType> allowedTypes = EntryStream.of(VISIBLE_GROUP_TYPES_BY_FEATURES)
                .mapKeys(FeatureName::getName)
                .filterKeys(enabledFeatures::contains)
                .values()
                .toSet();
        return ImmutableSet.<AdGroupType>builder()
                .addAll(VISIBLE_GROUP_TYPES)
                .addAll(allowedTypes)
                .build();
    }

    public static boolean isSynthetic(GdAdGroupType adGroupType) {
        return ALL_SYNTHETIC_GROUP_TYPES.contains(adGroupType);
    }

    public static ImmutableSet<GdAdGroupType> getVisibleSyntheticAdGroupTypes(Set<String> enabledFeatures) {
        LinkedHashSet<GdAdGroupType> allowedTypes = EntryStream.of(VISIBLE_SYNTHETIC_GROUP_TYPE_BY_FEATURE)
                .mapKeys(FeatureName::getName)
                .filterKeys(enabledFeatures::contains)
                .values()
                .toCollection(LinkedHashSet::new);

        return ImmutableSet.<GdAdGroupType>builder()
                .addAll(VISIBLE_SYNTHETIC_GROUP_TYPES)
                .addAll(allowedTypes)
                .build();
    }

    public static Set<GdCpmGroupType> getEditableCpmAdGroupTypes(Set<String> enabledFeatures) {
        var blockedTypes = EntryStream.of(EDITABLE_GROUP_FEATURES_BY_TYPE)
                .mapValues(FeatureName::getName)
                .removeValues(enabledFeatures::contains)
                .keys()
                .toSet();

        return StreamEx.of(getVisibleAdGroupTypes(enabledFeatures))
                .remove(blockedTypes::contains)
                .map(CPM_GROUP_TYPE_BY_ADGROUP_TYPE::get)
                .nonNull()
                .toImmutableSet();
    }

    public static GdAdGroupType getContentPromotionAdGroupType(Enum contentPromotionType) {
        return GdAdGroupType.valueOf(format("%s_%s", GdCampaignType.CONTENT_PROMOTION.name(),
                contentPromotionType.name()));
    }
}
