package ru.yandex.direct.grid.processing.service.group;

import java.util.Collections;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.ContentPromotionAdgroupType;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackage;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.grid.model.campaign.GdCampaign;
import ru.yandex.direct.grid.model.campaign.GdCampaignType;
import ru.yandex.direct.grid.model.entity.adgroup.AdGroupTypeConverter;
import ru.yandex.direct.grid.model.entity.adgroup.GdAdGroupType;

import static ru.yandex.direct.core.entity.campaign.CampaignUtils.CPM_TYPES;
import static ru.yandex.direct.feature.FeatureName.CPM_AUDIO_GROUPS_EDIT_FOR_DNA;
import static ru.yandex.direct.feature.FeatureName.CPM_INDOOR_GROUPS_EDIT_FOR_DNA;
import static ru.yandex.direct.feature.FeatureName.CPM_OUTDOOR_GROUPS_EDIT_FOR_DNA;
import static ru.yandex.direct.feature.FeatureName.EDIT_CPM_YNDX_FRONTPAGE_IN_DNA;
import static ru.yandex.direct.grid.model.entity.adgroup.AdGroupTypeConverter.toGdAdGroupType;
import static ru.yandex.direct.grid.model.entity.campaign.converter.CampaignDataConverter.toGdCampaignType;
import static ru.yandex.direct.grid.processing.service.group.AdGroupTypeUtils.getContentPromotionAdGroupType;
import static ru.yandex.direct.grid.processing.service.group.AdGroupTypeUtils.getVisibleSyntheticAdGroupTypes;
import static ru.yandex.direct.grid.processing.service.pricepackage.converter.PricePackageDataConverter.toGdAdGroupTypes;

@ParametersAreNonnullByDefault
public class AvailableAdGroupTypesCalculator {
    private static final ImmutableMap<GdCampaignType, Set<GdAdGroupType>> AVAILABLE_AD_GROUP_TYPES_FOR_CAMPAIGN_TYPE =
            ImmutableMap.<GdCampaignType, Set<GdAdGroupType>>builder()
                    .put(GdCampaignType.TEXT, Set.of(GdAdGroupType.TEXT))
                    .put(GdCampaignType.PERFORMANCE, Set.of(GdAdGroupType.PERFORMANCE))
                    .put(GdCampaignType.MOBILE_CONTENT, Set.of(GdAdGroupType.MOBILE_CONTENT))
                    .put(GdCampaignType.DYNAMIC, Set.of(GdAdGroupType.DYNAMIC))
                    .put(GdCampaignType.MCBANNER, Set.of(GdAdGroupType.MCBANNER))
                    .put(GdCampaignType.CPM_BANNER,
                            Set.of(GdAdGroupType.CPM_BANNER, GdAdGroupType.CPM_VIDEO, GdAdGroupType.CPM_AUDIO))
                    .put(GdCampaignType.CPM_DEALS, Set.of(GdAdGroupType.CPM_BANNER))
                    .put(GdCampaignType.INTERNAL_AUTOBUDGET, Set.of(GdAdGroupType.INTERNAL))
                    .put(GdCampaignType.INTERNAL_DISTRIB, Set.of(GdAdGroupType.INTERNAL))
                    .put(GdCampaignType.INTERNAL_FREE, Set.of(GdAdGroupType.INTERNAL))
                    .put(GdCampaignType.CPM_YNDX_FRONTPAGE, Set.of(GdAdGroupType.CPM_YNDX_FRONTPAGE))
                    .put(GdCampaignType.CPM_PRICE, Set.of(GdAdGroupType.CPM_PRICE, GdAdGroupType.CPM_PRICE_VIDEO,
                            GdAdGroupType.CPM_PRICE_AUDIO, GdAdGroupType.CPM_PRICE_BANNER,
                            GdAdGroupType.CPM_PRICE_FRONTPAGE_VIDEO))
                    .put(GdCampaignType.CONTENT_PROMOTION,
                            Set.of(GdAdGroupType.CONTENT_PROMOTION_COLLECTION,
                                    GdAdGroupType.CONTENT_PROMOTION_VIDEO,
                                    GdAdGroupType.CONTENT_PROMOTION_SERVICE,
                                    GdAdGroupType.CONTENT_PROMOTION_EDA))
                    .build();

    public static final Set<AdGroupType> AVAILABLE_AD_GROUP_TYPES_FOR_REMODERATION =
            Set.of(AdGroupType.BASE, AdGroupType.MOBILE_CONTENT, AdGroupType.DYNAMIC,
                    AdGroupType.CPM_YNDX_FRONTPAGE);

    public static final Set<AdGroupType> AVAILABLE_AD_GROUP_TYPES_FOR_ACCEPT_MODERATION =
            Set.of(AdGroupType.BASE, AdGroupType.MOBILE_CONTENT, AdGroupType.DYNAMIC);

    private static final Set<GdAdGroupType> AVAILABLE_AD_GROUP_TYPES_FOR_COPY =
            Set.of(GdAdGroupType.TEXT, GdAdGroupType.CPM_BANNER, GdAdGroupType.CPM_VIDEO, GdAdGroupType.CPM_AUDIO,
                    GdAdGroupType.CPM_INDOOR, GdAdGroupType.CPM_OUTDOOR, GdAdGroupType.CPM_GEOPRODUCT,
                    GdAdGroupType.CPM_GEO_PIN, GdAdGroupType.MCBANNER, GdAdGroupType.DYNAMIC,
                    GdAdGroupType.MOBILE_CONTENT,
                    GdAdGroupType.CONTENT_PROMOTION, GdAdGroupType.CPM_YNDX_FRONTPAGE, GdAdGroupType.PERFORMANCE,
                    GdAdGroupType.INTERNAL);

    private static final Set<AdGroupType> NOT_AVAILABLE_AD_GROUP_TYPES_FOR_EDIT_ADS =
            Set.of(AdGroupType.PERFORMANCE);

    private static final Set<AdGroupType> NOT_AVAILABLE_AD_GROUP_TYPES_FOR_EDIT_KEYWORDS =
            Set.of(AdGroupType.PERFORMANCE);

    private static final Set<AdGroupType> NOT_AVAILABLE_AD_GROUP_TYPES_FOR_EDIT_REGIONS =
            Set.of(AdGroupType.CPM_OUTDOOR, AdGroupType.CPM_INDOOR);

    public static final Map<AdGroupType, FeatureName> EDITABLE_GROUP_FEATURES_BY_AD_GROUP_TYPE = Map.of(
            AdGroupType.CPM_INDOOR, CPM_INDOOR_GROUPS_EDIT_FOR_DNA,
            AdGroupType.CPM_OUTDOOR, CPM_OUTDOOR_GROUPS_EDIT_FOR_DNA,
            AdGroupType.CPM_AUDIO, CPM_AUDIO_GROUPS_EDIT_FOR_DNA
    );

    public static final Map<AdGroupType, FeatureName> COPYABLE_GROUP_FEATURES_BY_AD_GROUP_TYPE = Map.of(
            AdGroupType.CPM_INDOOR, CPM_INDOOR_GROUPS_EDIT_FOR_DNA,
            AdGroupType.CPM_OUTDOOR, CPM_OUTDOOR_GROUPS_EDIT_FOR_DNA,
            AdGroupType.CPM_AUDIO, CPM_AUDIO_GROUPS_EDIT_FOR_DNA,
            AdGroupType.CPM_YNDX_FRONTPAGE, EDIT_CPM_YNDX_FRONTPAGE_IN_DNA
    );

    /**
     * проверяет доступно ли редактирование регионов по переданному типу группы
     */
    public static boolean canEditAdGroupRegions(AdGroupType adGroupType) {
        return !NOT_AVAILABLE_AD_GROUP_TYPES_FOR_EDIT_REGIONS.contains(adGroupType);
    }

    /**
     * Проверяет, доступно ли редактирование объявлений для групп переданного типа
     */
    public static boolean canEditAdGroupAds(AdGroupType adGroupType, Set<String> enabledFeatures) {
        if (adGroupType == AdGroupType.PERFORMANCE && !enabledFeatures.contains(FeatureName.CREATIVE_FREE_INTERFACE.getName())) {
            return true;
        }
        return !NOT_AVAILABLE_AD_GROUP_TYPES_FOR_EDIT_ADS.contains(adGroupType);
    }

    /**
     * Проверяет, доступно ли редактирование ключевых фраз для групп переданного типа
     */
    public static boolean canEditAdGroupKeywords(AdGroupType adGroupType) {
        return !NOT_AVAILABLE_AD_GROUP_TYPES_FOR_EDIT_KEYWORDS.contains(adGroupType);
    }

    /**
     * возвращает доступые для кампании типы групп
     */
    public static Set<GdAdGroupType> getAvailableAdGroupTypesForCampaign(GdCampaign campaign,
                                                                         Set<String> enabledFeatures,
                                                                         Map<Long, ContentPromotionAdgroupType> contentPromotionTypeByCampaignId,
                                                                         Map<Long, PricePackage> pricePackageByCampaignId) {
        Set<GdAdGroupType> defaultAvailableTypes = AVAILABLE_AD_GROUP_TYPES_FOR_CAMPAIGN_TYPE.get(campaign.getType());

        if (campaign.getType().equals(GdCampaignType.CONTENT_PROMOTION)) {
            ContentPromotionAdgroupType contentPromotionType = contentPromotionTypeByCampaignId.get(campaign.getId());
            ImmutableSet<GdAdGroupType> visibleAdGroupTypes = getVisibleSyntheticAdGroupTypes(enabledFeatures);

            // Если в кампании продвижения контента есть хоть одна группа, то в ней можно создавать только группы этого
            // типа, в противном случае выбираем доступные группы в зависимости от включенных фич.
            return Sets.intersection(visibleAdGroupTypes, contentPromotionType == null ? defaultAvailableTypes
                    : Set.of(getContentPromotionAdGroupType(contentPromotionType)));
        }

        if (campaign.getType().equals(GdCampaignType.CPM_PRICE)
                && pricePackageByCampaignId.containsKey(campaign.getId())) {
            var pricePackage = pricePackageByCampaignId.get(campaign.getId());
            return toGdAdGroupTypes(pricePackage.getAvailableAdGroupTypes());
        }

        return defaultAvailableTypes;
    }

    /**
     * проверяет доступно ли редактирование по переданному типу группы
     */
    public static boolean canEditAdGroup(CampaignType campaignType,
                                         AdGroupType adGroupType,
                                         Set<String> enabledFeatures) {
        if (enabledFeatures.contains(FeatureName.IS_CPM_BANNER_CAMPAIGN_DISABLED.getName())
                && CPM_TYPES.contains(campaignType)) {
            return false;
        }
        // не ограничиваем по фиче cpm_audio на прайсовых кампаниях
        if (campaignType == CampaignType.CPM_PRICE && adGroupType == AdGroupType.CPM_AUDIO) {
            return true;
        }
        var uneditableTypes = EntryStream.of(EDITABLE_GROUP_FEATURES_BY_AD_GROUP_TYPE)
                .mapValues(FeatureName::getName)
                .removeValues(enabledFeatures::contains)
                .keys()
                .toSet();

        return !uneditableTypes.contains(adGroupType);
    }

    /**
     * проверяет доступно ли копирование по переданному типу группы и типу кампании
     */
    public static boolean canCopyAdGroup(AdGroupType adGroupType, CampaignType campaignType,
                                         Set<String> enabledFeatures) {
        var uncopyableTypes = EntryStream.of(COPYABLE_GROUP_FEATURES_BY_AD_GROUP_TYPE)
                .mapValues(FeatureName::getName)
                .removeValues(enabledFeatures::contains)
                .keys()
                .toSet();
        GdAdGroupType gdAdGroupType = toGdAdGroupType(adGroupType, toGdCampaignType(campaignType));

        return AVAILABLE_AD_GROUP_TYPES_FOR_COPY.contains(gdAdGroupType)
                && !uncopyableTypes.contains(adGroupType);
    }

    /**
     * проверяет доступно ли копирование группы для кампании по переданному типу кампании
     */
    public static boolean canCopyAdGroupForCampaign(CampaignType campaignType, Set<String> enabledFeatures) {
        GdCampaignType gdCampaignType = toGdCampaignType(campaignType);
        Set<GdAdGroupType> gdAdGroupTypes =
                AVAILABLE_AD_GROUP_TYPES_FOR_CAMPAIGN_TYPE.getOrDefault(gdCampaignType, Collections.emptySet());

        return StreamEx.of(gdAdGroupTypes)
                .map(AdGroupTypeConverter::toInternalAdGroupType)
                .anyMatch(adGroupType -> canCopyAdGroup(adGroupType, campaignType, enabledFeatures));
    }

    static boolean canBeSentToRemoderationAdGroupType(AdGroupType adGroupType) {
        return AVAILABLE_AD_GROUP_TYPES_FOR_REMODERATION.contains(adGroupType);
    }

    static boolean canAcceptModerationAdGroupType(AdGroupType adGroupType) {
        return AVAILABLE_AD_GROUP_TYPES_FOR_ACCEPT_MODERATION.contains(adGroupType);
    }

}
