package ru.yandex.direct.grid.processing.service.group;

import java.util.Collection;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.copyentity.CopyConfigBuilder;
import ru.yandex.direct.core.copyentity.CopyOperation;
import ru.yandex.direct.core.copyentity.CopyOperationFactory;
import ru.yandex.direct.core.copyentity.CopyResult;
import ru.yandex.direct.core.copyentity.model.CopyCampaignFlags;
import ru.yandex.direct.core.copyentity.translations.RenameProcessor;
import ru.yandex.direct.core.entity.adgroup.container.ComplexAdGroup;
import ru.yandex.direct.core.entity.adgroup.container.ComplexCpmAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.service.AdGroupService;
import ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupAddOperationFactory;
import ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupService;
import ru.yandex.direct.core.entity.campaign.model.BaseCampaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.service.CampaignService;
import ru.yandex.direct.core.entity.client.service.ClientGeoService;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.currency.model.cpmyndxfrontpage.CpmYndxFrontpageAdGroupPriceRestrictions;
import ru.yandex.direct.core.entity.currency.service.CpmYndxFrontpageCurrencyService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.showcondition.container.ShowConditionAutoPriceParams;
import ru.yandex.direct.core.entity.showcondition.container.ShowConditionFixedAutoPrices;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.result.ResultState;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.ytcore.entity.statistics.service.RecentStatisticsService;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.core.entity.adgroup.model.AdGroupType.CPM_AUDIO;
import static ru.yandex.direct.core.entity.adgroup.model.AdGroupType.CPM_BANNER;
import static ru.yandex.direct.core.entity.adgroup.model.AdGroupType.CPM_GEOPRODUCT;
import static ru.yandex.direct.core.entity.adgroup.model.AdGroupType.CPM_GEO_PIN;
import static ru.yandex.direct.core.entity.adgroup.model.AdGroupType.CPM_INDOOR;
import static ru.yandex.direct.core.entity.adgroup.model.AdGroupType.CPM_OUTDOOR;
import static ru.yandex.direct.core.entity.adgroup.model.AdGroupType.CPM_VIDEO;
import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupModelUtils.cloneComplexCpmAdGroup;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.errorsWhileCopyingAdGroups;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class CopyAdGroupsService {
    private static final Set<AdGroupType> CPM_GROUP_TYPES_FOR_COPY = ImmutableSet.of(
            CPM_BANNER, CPM_VIDEO, CPM_AUDIO, CPM_OUTDOOR, CPM_INDOOR, CPM_GEOPRODUCT, CPM_GEO_PIN);

    private final ClientGeoService clientGeoService;
    private final AdGroupService adGroupService;
    private final ComplexAdGroupService complexAdGroupService;
    private final ComplexAdGroupAddOperationFactory complexAdGroupAddOperationFactory;
    private final RecentStatisticsService recentStatisticsService;
    private final ClientService clientService;
    private final UserService userService;
    private final ShardHelper shardHelper;
    private final CopyOperationFactory copyOperationFactory;
    private final CpmYndxFrontpageCurrencyService cpmYndxFrontpageCurrencyService;
    private final CampaignService campaignService;
    private final FeatureService featureService;
    private final RenameProcessor renameProcessor;

    @Autowired
    public CopyAdGroupsService(
            ClientGeoService clientGeoService,
            AdGroupService adGroupService,
            ComplexAdGroupService complexAdGroupService,
            ComplexAdGroupAddOperationFactory complexAdGroupAddOperationFactory,
            RecentStatisticsService recentStatisticsService,
            ClientService clientService,
            UserService userService,
            ShardHelper shardHelper,
            CopyOperationFactory copyOperationFactory,
            CpmYndxFrontpageCurrencyService cpmYndxFrontpageCurrencyService,
            CampaignService campaignService,
            FeatureService featureService,
            RenameProcessor renameProcessor) {
        this.clientGeoService = clientGeoService;
        this.adGroupService = adGroupService;
        this.complexAdGroupService = complexAdGroupService;
        this.complexAdGroupAddOperationFactory = complexAdGroupAddOperationFactory;
        this.recentStatisticsService = recentStatisticsService;
        this.clientService = clientService;
        this.userService = userService;
        this.shardHelper = shardHelper;
        this.copyOperationFactory = copyOperationFactory;
        this.cpmYndxFrontpageCurrencyService = cpmYndxFrontpageCurrencyService;
        this.campaignService = campaignService;
        this.featureService = featureService;
        this.renameProcessor = renameProcessor;
    }

    /**
     * Копирования групп объявлений.
     * Если destinationCampaignId == null, значит копируем в те же кампании групп
     * <p>
     * Поддерживаемые типы кампаний: ТГО, CPM_BANNER, CPM_VIDEO, CPM_AUDIO, CPM_OUTDOOR, CPM_INDOOR, CPM_GEOPRODUCT
     *
     * @param operatorUid           идентификатор пользователя оператора
     * @param uidAndClientId        uid и clientId клиента, которому принадлежат группы
     * @param adGroupIds            список id групп
     * @param destinationCampaignId id кампании, в которую копируем группы
     * @return результат копирования для всех групп (aka FULL operation), иначе возращаем результат валидации
     */
    MassResult<Long> copyAdGroups(
            Long operatorUid,
            UidAndClientId uidAndClientId,
            List<Long> adGroupIds,
            @Nullable Long destinationCampaignId
    ) {
        var adGroupTypes = adGroupService.getAdGroupTypes(uidAndClientId.getClientId(), adGroupIds)
                .values()
                .stream()
                //объединяем все охватные группы в один тип CPM_BANNER
                .map(adGroupType -> CPM_GROUP_TYPES_FOR_COPY.contains(adGroupType) ? CPM_BANNER : adGroupType)
                .distinct()
                .collect(toList());

        // проверяем, что пришли группы только одного типа (все охватные группы считаем за один тип)
        checkArgument(adGroupTypes.size() == 1);

        var adGroupType = adGroupTypes.get(0);
        switch (adGroupType) {
            case BASE:
            case INTERNAL:
            case PERFORMANCE:
            case DYNAMIC:
            case MOBILE_CONTENT:
            case MCBANNER:
            case CPM_BANNER:
            case CPM_YNDX_FRONTPAGE:
                return copyAdGroupsEntityGraph(operatorUid, uidAndClientId, adGroupIds, destinationCampaignId);
            default:
                throw new UnsupportedOperationException("Cannot copy this AdGroup type: " + adGroupType);
        }
    }

    private MassResult<Long> copyCpmAdGroups(
            Long operatorUid,
            UidAndClientId uidAndClientId,
            List<Long> adGroupIds,
            @Nullable Long destinationCampaignId
    ) {
        ClientId clientId = uidAndClientId.getClientId();
        List<ComplexCpmAdGroup> complexCpmAdGroups =
                complexAdGroupService.getComplexCpmAdGroups(operatorUid, uidAndClientId, adGroupIds);

        List<ComplexCpmAdGroup> newComplexCpmAdGroups = StreamEx.of(complexCpmAdGroups)
                .map(complexCpmAdGroup -> cloneComplexCpmAdGroup(complexCpmAdGroup,
                        nvl(destinationCampaignId, complexCpmAdGroup.getAdGroup().getCampaignId())))
                .toList();

        List<AdGroup> adGroups = mapList(newComplexCpmAdGroups, ComplexAdGroup::getAdGroup);
        fillAdGroupsCopyName(adGroups, uidAndClientId.getClientId());

        var geoTree = clientGeoService.getClientTranslocalGeoTree(clientId);

        var clientCurrency = clientService.getWorkCurrency(clientId);
        var fixedPrice = clientCurrency.getMinCpmPrice();
        Long campaignId = newComplexCpmAdGroups.get(0).getAdGroup().getCampaignId();
        CampaignType campaignType = campaignService.getCampaignsTypes(ImmutableList.of(campaignId)).get(campaignId);

        if (campaignType == CampaignType.CPM_YNDX_FRONTPAGE) {
            int shard = shardHelper.getShardByClientIdStrictly(clientId);
            Map<Integer, CpmYndxFrontpageAdGroupPriceRestrictions> indexesToPriceDataMapByAdGroups =
                    cpmYndxFrontpageCurrencyService.getAdGroupIndexesToPriceDataMapByAdGroups(mapList(newComplexCpmAdGroups, ComplexAdGroup::getAdGroup),
                            shard, clientId);
            fixedPrice = EntryStream.of(indexesToPriceDataMapByAdGroups)
                    .filterValues(Objects::nonNull)
                    .minByInt(Map.Entry::getKey)
                    .map(e -> e.getValue().getCpmYndxFrontpageMinPrice())
                    .orElse(fixedPrice);
        } else if (newComplexCpmAdGroups.get(0).getKeywords() != null
                && !newComplexCpmAdGroups.get(0).getKeywords().isEmpty()
                && newComplexCpmAdGroups.get(0).getKeywords().get(0).getPrice() != null) {
            fixedPrice = newComplexCpmAdGroups.get(0).getKeywords().get(0).getPrice();
        }

        var fixedAdGroupAutoPrices = ShowConditionFixedAutoPrices.ofGlobalFixedPrice(fixedPrice);
        var autoPriceParams = new ShowConditionAutoPriceParams(fixedAdGroupAutoPrices, recentStatisticsService);

        var addOperation = complexAdGroupAddOperationFactory.createCpmAdGroupAddOperation(
                true,
                newComplexCpmAdGroups,
                geoTree,
                true,
                autoPriceParams,
                operatorUid,
                clientId,
                uidAndClientId.getUid(),
                false);

        return addOperation.prepareAndApply();
    }

    private MassResult<Long> copyAdGroupsEntityGraph(
            Long operatorUid,
            UidAndClientId uidAndClientId,
            List<Long> adGroupIds,
            @Nullable Long destinationCampaignId) {
        ClientId clientId = uidAndClientId.getClientId();

        CopyConfigBuilder<AdGroup, Long> builder =
                new CopyConfigBuilder<>(clientId, clientId, operatorUid, AdGroup.class, adGroupIds);

        builder.withFlags(new CopyCampaignFlags.Builder().withCopyStopped(true).build());

        // Проставляем, куда копировать
        var groups = adGroupService.getSimpleAdGroups(clientId, adGroupIds);
        for (var group : groups.values()) {
            builder.withParentIdMapping(BaseCampaign.class,
                    group.getCampaignId(), nvl(destinationCampaignId, group.getCampaignId()));
        }

        CopyOperation<AdGroup, Long> operation = copyOperationFactory.build(builder.build());

        CopyResult<Long> copyResult = operation.copy();
        copyResult.logFailedResultsForMonitoring();

        // если есть ошибки - добавляем специальную ошибку "При копировании групп были ошибки" для фронта, остальные
        // оставляем для дебага
        ValidationResult<?, Defect> vr = copyResult.getMassResult().getValidationResult();
        if (vr.hasAnyErrors()) {
            vr.addWarning(errorsWhileCopyingAdGroups());
        }
        // соответствуем исходному контракту - в результате id скопированных групп, а не копируемых
        List<Result<Long>> copiedEntityResults = copyResult.getCopiedEntityResults(AdGroup.class);
        return new MassResult<>(copiedEntityResults, vr, ResultState.SUCCESSFUL);
    }

    private void fillAdGroupsCopyName(Collection<AdGroup> adGroups, ClientId clientIdTo) {
        User chiefUserTo = userService.getChiefUserByClientIdMap(List.of(clientIdTo)).get(clientIdTo);
        Locale locale = RenameProcessor.getCopyLocaleByChiefUser(chiefUserTo);

        adGroups.forEach(adGroup -> {
            String newName = renameProcessor.generateAdGroupCopyName(adGroup.getName(), adGroup.getId(), locale);
            adGroup.withName(newName);
        });
    }
}
