package ru.yandex.direct.grid.processing.service.group;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import io.leangen.graphql.util.Utils;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.aggregatedstatuses.AggregatedStatusesViewService;
import ru.yandex.direct.core.entity.adgroup.generation.AdGroupKeywordRecommendationService;
import ru.yandex.direct.core.entity.adgroup.generation.AdGroupKeywordRecommendationService.Companion.AdGroupInfo;
import ru.yandex.direct.core.entity.adgroup.generation.AdGroupKeywordRecommendationService.Companion.AdGroupKeywordRecommendationInput;
import ru.yandex.direct.core.entity.adgroup.generation.AdGroupKeywordRecommendationService.Companion.AdditionalInfo;
import ru.yandex.direct.core.entity.adgroup.generation.AdGroupKeywordRecommendationService.Companion.BannerInfo;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.adgroup.repository.internal.AdGroupTagsRepository;
import ru.yandex.direct.core.entity.adgroup.service.AdGroupService;
import ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum;
import ru.yandex.direct.core.entity.aggregatedstatuses.adgroup.AggregatedStatusAdGroupData;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.feed.model.Feed;
import ru.yandex.direct.core.entity.feed.service.FeedService;
import ru.yandex.direct.core.entity.geo.service.CurrentGeoService;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackage;
import ru.yandex.direct.core.entity.pricepackage.service.PricePackageService;
import ru.yandex.direct.core.entity.relevancematch.model.RelevanceMatchCategory;
import ru.yandex.direct.core.entity.tag.model.CampaignTag;
import ru.yandex.direct.core.entity.tag.model.Tag;
import ru.yandex.direct.core.entity.tag.repository.TagRepository;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.grid.core.entity.group.model.GdiBaseGroup;
import ru.yandex.direct.grid.core.entity.group.model.GdiGroup;
import ru.yandex.direct.grid.core.entity.group.model.GdiGroupFilter;
import ru.yandex.direct.grid.core.entity.group.model.GdiGroupOrderBy;
import ru.yandex.direct.grid.core.entity.group.model.GdiGroupPrimaryStatus;
import ru.yandex.direct.grid.core.entity.group.model.GdiGroupsWithTotals;
import ru.yandex.direct.grid.core.entity.group.service.GridAdGroupService;
import ru.yandex.direct.grid.model.GdEntityStats;
import ru.yandex.direct.grid.model.aggregatedstatuses.GdAdgroupAggregatedStatusCountersInfo;
import ru.yandex.direct.grid.model.aggregatedstatuses.GdAdgroupAggregatedStatusInfo;
import ru.yandex.direct.grid.model.campaign.GdCampaignTruncated;
import ru.yandex.direct.grid.model.campaign.GdPriceCampaign;
import ru.yandex.direct.grid.model.campaign.GdiBaseCampaign;
import ru.yandex.direct.grid.model.entity.adgroup.GdAdGroupType;
import ru.yandex.direct.grid.model.feed.GdFeed;
import ru.yandex.direct.grid.model.utils.GridModerationUtils;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.client.GdClientInfo;
import ru.yandex.direct.grid.processing.model.client.GdOperatorAction;
import ru.yandex.direct.grid.processing.model.group.GdAdGroup;
import ru.yandex.direct.grid.processing.model.group.GdAdGroupAccess;
import ru.yandex.direct.grid.processing.model.group.GdAdGroupFilter;
import ru.yandex.direct.grid.processing.model.group.GdAdGroupTruncated;
import ru.yandex.direct.grid.processing.model.group.GdAdGroupWithTotals;
import ru.yandex.direct.grid.processing.model.group.GdAdGroupsContainer;
import ru.yandex.direct.grid.processing.model.group.mutation.GdRelevanceMatchCategory;
import ru.yandex.direct.grid.processing.model.showcondition.GdAdGroupGetKeywordRecommendationInput;
import ru.yandex.direct.grid.processing.model.showcondition.GdKeywordsByCategory;
import ru.yandex.direct.grid.processing.model.tag.GdTag;
import ru.yandex.direct.grid.processing.service.aggregatedstatuses.AdGroupPrimaryStatusCalculator;
import ru.yandex.direct.grid.processing.service.campaign.CampaignInfoService;
import ru.yandex.direct.grid.processing.service.dataloader.GridContextProvider;
import ru.yandex.direct.grid.processing.service.feed.FeedConverter;
import ru.yandex.direct.grid.processing.service.group.converter.AdGroupDataConverter;
import ru.yandex.direct.grid.processing.service.group.loader.AdGroupsHasCalloutsDataLoader;
import ru.yandex.direct.grid.processing.service.group.loader.AdGroupsHasShowConditionsDataLoader;
import ru.yandex.direct.grid.processing.service.group.loader.AdsCountDataLoader;
import ru.yandex.direct.grid.processing.service.group.loader.CanBeDeletedAdGroupsDataLoader;
import ru.yandex.direct.grid.processing.service.group.loader.KeywordsCountDataLoader;
import ru.yandex.direct.grid.processing.service.group.type.GroupTypeFacade;
import ru.yandex.direct.grid.processing.service.group.validation.AdGroupKeywordRecommendationValidationService;
import ru.yandex.direct.grid.processing.service.operator.OperatorAllowedActionsUtils;
import ru.yandex.direct.grid.processing.service.tags.TagConverter;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;
import ru.yandex.direct.grid.processing.util.GoalHelper;
import ru.yandex.direct.grid.processing.util.ReasonsFilterUtils;
import ru.yandex.direct.grid.processing.util.ResultConverterHelper;
import ru.yandex.direct.grid.processing.util.StatHelper;
import ru.yandex.direct.utils.CollectionUtils;
import ru.yandex.direct.validation.result.Path;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.emptySet;
import static java.util.Collections.singleton;
import static java.util.function.Function.identity;
import static org.apache.commons.collections4.CollectionUtils.emptyIfNull;
import static ru.yandex.direct.core.validation.ValidationUtils.hasValidationIssues;
import static ru.yandex.direct.feature.FeatureName.CPC_AND_CPM_ON_ONE_GRID_ENABLED;
import static ru.yandex.direct.feature.FeatureName.HIDE_OLD_SHOW_CAMPS_FOR_DNA;
import static ru.yandex.direct.feature.FeatureName.INTERFACE_CONTROL_GROUP;
import static ru.yandex.direct.feature.FeatureName.NEW_GEO_DEFAULT_AND_SUGGEST;
import static ru.yandex.direct.feature.FeatureName.SHOW_AGGREGATED_STATUS_OPEN_BETA;
import static ru.yandex.direct.feature.FeatureName.SHOW_DNA_BY_DEFAULT;
import static ru.yandex.direct.grid.model.campaign.GdCampaignType.CPM_BANNER;
import static ru.yandex.direct.grid.model.campaign.GdCampaignType.CPM_YNDX_FRONTPAGE;
import static ru.yandex.direct.grid.model.entity.campaign.converter.CampaignDataConverter.toCampaignType;
import static ru.yandex.direct.grid.processing.service.group.AdGroupActionConditionsUtil.canAcceptModerationConditions;
import static ru.yandex.direct.grid.processing.service.group.AdGroupActionConditionsUtil.canBeSentToRemoderationConditions;
import static ru.yandex.direct.grid.processing.service.group.AdGroupActionConditionsUtil.isAdGroupInDraftAndNotArchived;
import static ru.yandex.direct.grid.processing.service.group.AdGroupActionConditionsUtil.isAdGroupNotDraftAndNotArchived;
import static ru.yandex.direct.grid.processing.service.group.AdGroupActionConditionsUtil.isCpmPriceDefaultAdGroupReadyToStart;
import static ru.yandex.direct.grid.processing.service.group.AdGroupTypeUtils.getVisibleAdGroupTypes;
import static ru.yandex.direct.grid.processing.service.group.AdGroupTypeUtils.getVisibleSyntheticAdGroupTypes;
import static ru.yandex.direct.grid.processing.service.group.AdGroupTypeUtils.isSynthetic;
import static ru.yandex.direct.grid.processing.service.group.converter.AdGroupDataConverter.toInternalFilter;
import static ru.yandex.direct.regions.Region.MOSCOW_REGION_ID;
import static ru.yandex.direct.regions.Region.RUSSIA_REGION_ID;
import static ru.yandex.direct.regions.Region.SAINT_PETERSBURG_REGION_ID;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Сервис, возвращающий данные о группах баннеров клиента
 */
@Service
@ParametersAreNonnullByDefault
public class GroupDataService {

    private static final int MAX_RECOMMENDED_KEYWORDS_COUNT = 5;

    private final AdGroupsHasCalloutsDataLoader adGroupsHasCalloutsDataLoader;
    private final AdGroupsHasShowConditionsDataLoader adGroupsHasShowConditionsDataLoader;
    private final AdGroupRepository adGroupRepository;
    private final AdsCountDataLoader adsCountDataLoader;
    private final KeywordsCountDataLoader keywordsCountDataLoader;
    private final CanBeDeletedAdGroupsDataLoader canBeDeletedAdGroupsDataLoader;
    private final AdGroupService adGroupService;
    private final AdGroupTagsRepository adGroupTagsRepository;
    private final TagRepository tagRepository;
    private final GridAdGroupService gridAdGroupService;
    private final GridContextProvider gridContextProvider;
    private final GroupTypeFacade adGroupTypeFacade;
    private final CampaignInfoService campaignInfoService;
    private final CurrentGeoService currentGeoService;
    private final FeatureService featureService;
    private final AggregatedStatusesViewService aggregatedStatusesViewService;
    private final FeedService feedService;
    private final AdGroupKeywordRecommendationService adGroupKeywordRecommendationService;
    private final AdGroupKeywordRecommendationValidationService adGroupKeywordRecommendationValidationService;
    private final GridValidationService gridValidationService;
    private final PricePackageService pricePackageService;

    @SuppressWarnings("checkstyle:parameternumber")
    @Autowired
    public GroupDataService(AdGroupsHasCalloutsDataLoader adGroupsHasCalloutsDataLoader,
                            AdGroupsHasShowConditionsDataLoader adGroupsHasShowConditionsDataLoader,
                            AdGroupRepository adGroupRepository, AdsCountDataLoader adsCountDataLoader,
                            KeywordsCountDataLoader keywordsCountDataLoader,
                            CanBeDeletedAdGroupsDataLoader canBeDeletedAdGroupsDataLoader,
                            AdGroupService adGroupService, AdGroupTagsRepository adGroupTagsRepository,
                            TagRepository tagRepository, GridAdGroupService gridAdGroupService,
                            GroupTypeFacade adGroupTypeFacade,
                            CampaignInfoService campaignInfoService, CurrentGeoService currentGeoService,
                            GridContextProvider gridContextProvider,
                            FeatureService featureService,
                            AggregatedStatusesViewService aggregatedStatusesViewService, FeedService feedService,
                            AdGroupKeywordRecommendationService adGroupKeywordRecommendationService,
                            AdGroupKeywordRecommendationValidationService adGroupKeywordRecommendationValidationService,
                            GridValidationService gridValidationService, PricePackageService pricePackageService) {
        this.adGroupsHasCalloutsDataLoader = adGroupsHasCalloutsDataLoader;
        this.adGroupsHasShowConditionsDataLoader = adGroupsHasShowConditionsDataLoader;
        this.adGroupRepository = adGroupRepository;
        this.adsCountDataLoader = adsCountDataLoader;
        this.keywordsCountDataLoader = keywordsCountDataLoader;
        this.canBeDeletedAdGroupsDataLoader = canBeDeletedAdGroupsDataLoader;
        this.adGroupService = adGroupService;
        this.adGroupTagsRepository = adGroupTagsRepository;
        this.tagRepository = tagRepository;
        this.gridAdGroupService = gridAdGroupService;
        this.gridContextProvider = gridContextProvider;
        this.adGroupTypeFacade = adGroupTypeFacade;
        this.campaignInfoService = campaignInfoService;
        this.currentGeoService = currentGeoService;
        this.featureService = featureService;
        this.aggregatedStatusesViewService = aggregatedStatusesViewService;
        this.feedService = feedService;
        this.adGroupKeywordRecommendationService = adGroupKeywordRecommendationService;
        this.adGroupKeywordRecommendationValidationService = adGroupKeywordRecommendationValidationService;
        this.gridValidationService = gridValidationService;
        this.pricePackageService = pricePackageService;
    }

    /**
     * Получить список групп клиента, в формате пригодном для возвращения оператору
     * <p>
     * При добавлении новых фильтров в коде нужно так же их учесть в методе:
     * {@link ru.yandex.direct.grid.processing.service.group.converter.AdGroupDataConverter#hasAnyCodeFilter}
     *
     * @param client  параметры клиента
     * @param input   входные параметры для получения групп
     * @param context контекст исполнения GraphQL запроса
     */
    public GdAdGroupWithTotals getAdGroups(GdClientInfo client, GdAdGroupsContainer input, GridGraphQLContext context) {
        User operator = context.getOperator();
        ClientId clientId = ClientId.fromLong(client.getId());
        List<GdiGroupOrderBy> internalOrderByList =
                mapList(input.getOrderBy(), AdGroupDataConverter::toInternalOrderBy);

        GdAdGroupFilter inputFilter = input.getFilter();
        Set<Long> goalIds =
                GoalHelper.combineGoalIds(input.getStatRequirements().getGoalIds(), inputFilter.getGoalStats());

        Set<String> enabledFeatures = featureService.getEnabledForClientId(clientId);
        boolean withFilterByAggrStatus = enabledFeatures.contains(SHOW_DNA_BY_DEFAULT.getName())
                || enabledFeatures.contains(HIDE_OLD_SHOW_CAMPS_FOR_DNA.getName())
                || enabledFeatures.contains(SHOW_AGGREGATED_STATUS_OPEN_BETA.getName());
        GdiGroupFilter filter = toInternalFilter(inputFilter);
        int shard = client.getShard();
        GdiGroupsWithTotals gdiGroupsWithTotals = gridAdGroupService.getGroups(shard, client.getCountryRegionId(),
                clientId, operator,
                filter, nvl(inputFilter.getRecommendations(), emptySet()), internalOrderByList,
                input.getStatRequirements(), withFilterByAggrStatus,
                goalIds, context.getFetchedFieldsReslover().getAdGroup());
        List<GdiGroup> groups = gdiGroupsWithTotals.getGdiGroups();
        GdEntityStats gdEntityStats = gdiGroupsWithTotals.getTotalStats() != null
                ? StatHelper.convertInternalStatsToOuter(gdiGroupsWithTotals.getTotalStats()) : null;

        Map<Long, AggregatedStatusAdGroupData> adGroupStatusesByIds =
                aggregatedStatusesViewService.getAdGroupStatusesByIds(shard, listToSet(groups, GdiGroup::getId));
        if (withFilterByAggrStatus) {
            groups = filterByAggregatedStatuses(adGroupStatusesByIds, groups, filter);
        }

        Set<AdGroupType> visibleTypes = getVisibleAdGroupTypes(enabledFeatures);
        groups = filterList(groups, gdiGroup -> visibleTypes.contains(gdiGroup.getType()));

        if (groups.isEmpty()) {
            return new GdAdGroupWithTotals()
                    .withGdAdGroups(emptyList())
                    .withTotalStats(gdEntityStats);
        }

        Set<Long> campaignIds = listToSet(groups, GdiGroup::getCampaignId);
        Map<Long, GdCampaignTruncated> campaignsById = campaignInfoService.getTruncatedCampaigns(clientId, campaignIds);
        groups = filterList(groups, gdiGroup -> campaignsById.containsKey(gdiGroup.getCampaignId()));
        if (groups.isEmpty()) {
            return new GdAdGroupWithTotals()
                    .withGdAdGroups(emptyList())
                    .withTotalStats(gdEntityStats);
        }

        //Извлекаем из базы метки групп, если по ним надо фильтровать или они есть в запросе
        Set<Long> tagIdIn = inputFilter.getTagIdIn();
        boolean tagsFieldRequired = context.getFetchedFieldsReslover().getAdGroup().getTags();
        Map<Long, List<GdTag>> gdTagsByAdGroupId = null;
        if (tagIdIn != null || tagsFieldRequired) {
            Set<Long> adGroupIds = listToSet(groups, GdiGroup::getId);
            Map<Long, List<Long>> tagIdsByAdGroupId = adGroupTagsRepository.getAdGroupsTags(shard, adGroupIds);
            if (tagIdIn != null) {
                groups = filterList(groups, group -> {
                    Long adGroupId = group.getId();
                    List<Long> groupTagIds = tagIdsByAdGroupId.getOrDefault(adGroupId, emptyList());
                    return groupTagIds.stream().anyMatch(tagIdIn::contains);
                });
            }
            if (tagsFieldRequired) {
                List<CampaignTag> tags = tagRepository.getCampaignTagsWithUseCount(shard, campaignIds);
                Map<Long, CampaignTag> tagById = listToMap(tags, Tag::getId);
                Function<Long, GdTag> tagIdToGdTag = tagId -> TagConverter.toGdTag(tagById.get(tagId));
                gdTagsByAdGroupId = EntryStream.of(tagIdsByAdGroupId)
                        .mapValues(tagIds -> mapList(tagIds, tagIdToGdTag))
                        .toMap();
            }
        }

        //Сохраняем в контекст главные объявления групп. Нужно для подсчета флага удаления групп
        Map<Long, BannerWithSystemFields> mainAdsByAdGroupId = StreamEx.of(groups)
                .map(GdiGroup::getMainAd)
                .nonNull()
                .toMap(BannerWithSystemFields::getAdGroupId, identity());
        context.setMainAdsByAdGroupId(mainAdsByAdGroupId);

        boolean showAggregatedStatusesDebug = featureService
                .isEnabledForClientId(operator.getClientId(), FeatureName.SHOW_AGGREGATED_STATUS_DEBUG);
        enrichAdGroupsAggregatedStatuses(groups, adGroupStatusesByIds, showAggregatedStatusesDebug);

        List<GdAdGroup> gdAdGroups = toGdAdGroups(shard, clientId, operator, groups, campaignsById, gdTagsByAdGroupId);

        // Фильтрация синтетических типов групп
        Set<GdAdGroupType> visibleSyntheticAdGroupTypes = getVisibleSyntheticAdGroupTypes(enabledFeatures);
        gdAdGroups = filterList(gdAdGroups,
                gdGroup -> !isSynthetic(gdGroup.getType()) || visibleSyntheticAdGroupTypes.contains(gdGroup.getType()));

        fillProductRestrictionId(shard, gdAdGroups);

        return new GdAdGroupWithTotals()
                .withGdAdGroups(gdAdGroups)
                .withTotalStats(gdEntityStats);
    }

    private void enrichAdGroupsAggregatedStatuses(List<GdiGroup> groups,
                                                  Map<Long, AggregatedStatusAdGroupData> adGroupStatusesByIds,
                                                  boolean debug) {
        for (GdiGroup g : groups) {
            AggregatedStatusAdGroupData status = adGroupStatusesByIds.get(g.getId());
            if (status != null && status.getStatus().isPresent()) {
                var statusInfo = new GdAdgroupAggregatedStatusInfo()
                        .withStatus(status.getStatus().get())
                        .withReasons(status.getReasons())
                        .withRejectReasons(GridModerationUtils.toGdRejectReasons(status.getRejectReasons()))
                        .withMightHaveRejectReasons(aggregatedStatusesViewService.adgroupCouldHaveRejectReasons(status))
                        .withIsObsolete(status.getIsObsolete());
                if (status.getCounters() != null) {
                    statusInfo.withCounters(new GdAdgroupAggregatedStatusCountersInfo()
                            .withAds(status.getCounters().getAdStatuses())
                            .withKeywords(status.getCounters().getKeywordStatuses()));
                }
                g.setAggregatedStatusInfo(statusInfo);
                if (debug) {
                    g.setAggregatedStatus(aggregatedStatusesViewService.toJson(status));
                }
            }
        }
    }

    private void fillProductRestrictionId(int shard, List<GdAdGroup> groups) {
        List<GdAdGroup> cpmGroups = filterList(groups,
                g -> g.getCampaign().getType() == CPM_BANNER || g.getCampaign().getType() == CPM_YNDX_FRONTPAGE);
        if (!cpmGroups.isEmpty()) {
            List<AdGroup> coreGroups = mapList(cpmGroups, AdGroupDataConverter::toInternalGroup);
            Map<Long, Long> productRestrictionIdsByAdGroupIds =
                    adGroupService.getProductRestrictionIdsByAdgroupIds(shard, coreGroups);
            cpmGroups.forEach(g -> g.setProductRestrictionId(productRestrictionIdsByAdGroupIds.get(g.getId())));
        }
    }

    /**
     * Получить ограниченное представление групп
     */
    public List<GdAdGroupTruncated> getTruncatedAdGroups(int shard, long clientCountryRegionId, ClientId clientId,
                                                         User operator, Collection<Long> adGroupIds,
                                                         Map<Long, GdCampaignTruncated> campaignsById) {
        Map<Long, CampaignType> campaignTypes = EntryStream.of(campaignsById)
                .mapValues(gdCampaign -> toCampaignType(gdCampaign.getType()))
                .toMap();
        List<GdiGroup> groups = gridAdGroupService.getTruncatedAdGroupsFromMysql(shard, clientCountryRegionId,
                adGroupIds, campaignTypes);
        Set<String> enabledFeatures = featureService.getEnabledForClientId(clientId);
        Set<AdGroupType> visibleTypes = getVisibleAdGroupTypes(enabledFeatures);
        List<GdiGroup> filteredGroups = filterList(groups,
                g -> campaignsById.containsKey(g.getCampaignId()) && visibleTypes.contains(g.getType()));
        List<GdAdGroup> gdAdGroups = toGdAdGroups(shard, clientId, operator, filteredGroups, campaignsById, null);

        // Фильтрация синтетических типов групп
        Set<GdAdGroupType> visibleSyntheticAdGroupTypes = getVisibleSyntheticAdGroupTypes(enabledFeatures);
        gdAdGroups = filterList(gdAdGroups,
                gdGroup -> !isSynthetic(gdGroup.getType()) || visibleSyntheticAdGroupTypes.contains(gdGroup.getType()));

        return mapList(gdAdGroups, GdAdGroupTruncated.class::cast);
    }

    private List<GdAdGroup> toGdAdGroups(int shard, ClientId clientId, User operator,
                                         List<GdiGroup> internalGroups,
                                         Map<Long, GdCampaignTruncated> campaignsById,
                                         @Nullable Map<Long, List<GdTag>> gdTagsByGroupId) {
        Map<Long, GdiBaseCampaign> gdiBaseCampaignMap = campaignInfoService.getAllBaseCampaignsMap(clientId);
        var campaignIdToPricePackageMap = getCampaignIdToPricePackageMap(campaignsById);
        Set<String> enabledFeatures = featureService.getEnabledForClientId(clientId);

        Map<Long, GdFeed> gdFeedsById = getGdFeedsById(clientId, internalGroups);

        boolean cpcAndCpmOnOneGridEnabled = enabledFeatures.contains(CPC_AND_CPM_ON_ONE_GRID_ENABLED.getName());

        List<GdAdGroup> gdAdGroups = new ArrayList<>(internalGroups.size());
        for (int i = 0; i < internalGroups.size(); i++) {
            GdiGroup group = internalGroups.get(i);
            GdiBaseCampaign gdiBaseCampaign = gdiBaseCampaignMap.get(group.getCampaignId());
            GdCampaignTruncated gdCampaign = campaignsById.get(group.getCampaignId());
            List<GdTag> gdTags = ifNotNull(gdTagsByGroupId, m -> m.getOrDefault(group.getId(), emptyList()));
            GdAdGroup gdAdGroup = AdGroupDataConverter.toOuter(i, group, gdiBaseCampaign, gdCampaign, gdTags, operator,
                    enabledFeatures, gdFeedsById, cpcAndCpmOnOneGridEnabled,
                    campaignIdToPricePackageMap.get(group.getCampaignId()));
            gdAdGroups.add(gdAdGroup);
        }
        adGroupTypeFacade.setAvailableEntityTypes(shard, clientId, operator.getUid(), gdAdGroups);
        return gdAdGroups;
    }

    private Map<Long, PricePackage> getCampaignIdToPricePackageMap(Map<Long, GdCampaignTruncated> campaignsById) {
        Map<Long, Long> campaignIdToPricePackageIdMap = StreamEx.of(campaignsById.values())
                .select(GdPriceCampaign.class)
                .toMap(GdPriceCampaign::getId, GdPriceCampaign::getPricePackageId);
        if (campaignIdToPricePackageIdMap.isEmpty()) {
            return emptyMap();
        }
        var pricePackages = pricePackageService.getPricePackages(campaignIdToPricePackageIdMap.values());
        return StreamEx.of(campaignIdToPricePackageIdMap.keySet())
                .toMap(identity(), cid -> pricePackages.get(campaignIdToPricePackageIdMap.get(cid)));
    }

    private Map<Long, GdFeed> getGdFeedsById(ClientId clientId, List<GdiGroup> internalGroups) {
        List<Long> feedIds = StreamEx.of(internalGroups)
                .filter(gdiGroup -> gdiGroup instanceof GdiBaseGroup && ((GdiBaseGroup) gdiGroup).getFeedId() != null)
                .map(gdiGroup -> ((GdiBaseGroup) gdiGroup).getFeedId())
                .toList();
        if (CollectionUtils.isEmpty(feedIds)) {
            return emptyMap();
        }
        var feeds = feedService.getFeeds(clientId, feedIds);
        var historyItemsByFeedId = feedService.getLatestFeedHistoryItems(clientId, listToSet(feeds, Feed::getId));
        //передаем operatorCanWrite false, так как тут проверка доступа не нужна
        return StreamEx.of(feeds)
                .map(feed -> FeedConverter.convertFeedToGd(feed, historyItemsByFeedId.get(feed.getId()), emptyList(),
                        false))
                .mapToEntry(GdFeed::getId, identity())
                .toMap();
    }

    CompletableFuture<Boolean> getCanBeDeletedAdGroup(GdAdGroupAccess gdAdGroupAccess) {
        if (gdAdGroupAccess.getCanEdit()) {
            return canBeDeletedAdGroupsDataLoader.get().load(gdAdGroupAccess.getAdGroupId());
        }

        return CompletableFuture.completedFuture(false);
    }

    CompletableFuture<Integer> getCanBeDeletedAdGroupsCount(List<GdAdGroupAccess> gdAdGroupAccesses) {
        List<Long> editableAdGroupIds =
                filterAndMapList(gdAdGroupAccesses, GdAdGroupAccess::getCanEdit, GdAdGroupAccess::getAdGroupId);
        if (editableAdGroupIds.isEmpty()) {
            return CompletableFuture.completedFuture(0);
        }

        return canBeDeletedAdGroupsDataLoader.get().loadMany(editableAdGroupIds)
                .thenApply(ResultConverterHelper::getCountOfTrueBooleanValues);
    }

    CompletableFuture<Boolean> getCanBeSentToBSAdGroup(GdAdGroupAccess gdAdGroupAccess) {
        if (hasOperatorAction(GdOperatorAction.SEND_TO_BS) && isAdGroupNotDraftAndNotArchived(gdAdGroupAccess)) {
            return adGroupsHasShowConditionsDataLoader.get().load(gdAdGroupAccess.getAdGroupId());
        }

        return CompletableFuture.completedFuture(false);
    }

    CompletableFuture<Integer> getCanBeSentToBSAdGroupsCount(List<GdAdGroupAccess> gdAdGroupAccesses) {
        List<Long> adGroupIds = filterAndMapList(gdAdGroupAccesses,
                AdGroupActionConditionsUtil::isAdGroupNotDraftAndNotArchived, GdAdGroupAccess::getAdGroupId);

        if (hasOperatorAction(GdOperatorAction.SEND_TO_BS) && !adGroupIds.isEmpty()) {
            return adGroupsHasShowConditionsDataLoader.get().loadMany(adGroupIds)
                    .thenApply(ResultConverterHelper::getCountOfTrueBooleanValues);
        }

        return CompletableFuture.completedFuture(0);
    }

    CompletableFuture<Boolean> getCanBeSentToModerationAdGroup(GdAdGroupAccess gdAdGroupAccess) {
        if (canBeSentToModeration(gdAdGroupAccess)) {
            return adGroupsHasShowConditionsDataLoader.get().load(gdAdGroupAccess.getAdGroupId());
        }
        return CompletableFuture.completedFuture(false);
    }

    CompletableFuture<Integer> getCanBeSentToModerationAdGroupsCount(List<GdAdGroupAccess> gdAdGroupAccesses) {
        List<Long> adGroupIds = filterAndMapList(gdAdGroupAccesses,
                this::canBeSentToModeration,
                GdAdGroupAccess::getAdGroupId);

        if (!adGroupIds.isEmpty()) {
            return adGroupsHasShowConditionsDataLoader.get().loadMany(adGroupIds)
                    .thenApply(ResultConverterHelper::getCountOfTrueBooleanValues);
        }

        return CompletableFuture.completedFuture(0);
    }

    private boolean canBeSentToModeration(GdAdGroupAccess gdAdGroupAccess) {
        GridGraphQLContext context = gridContextProvider.getGridContext();
        ClientId clientId = ClientId.fromLong(context.getQueriedClient().getId());
        GdAdGroup adGroup = context.getGdAdGroupsMap().get(gdAdGroupAccess.getAdGroupId());

        GdiBaseCampaign gdiBaseCampaign = campaignInfoService.getAllBaseCampaignsMap(clientId)
                .get(adGroup.getCampaignId());
        return hasOperatorAction(GdOperatorAction.SEND_TO_MODERATION)
                && isAdGroupInDraftAndNotArchived(gdAdGroupAccess)
                && !isCpmPriceDefaultAdGroupReadyToStart(adGroup, gdiBaseCampaign);
    }

    CompletableFuture<Boolean> getCanBeSentToRemoderationAdGroup(GdAdGroupAccess gdAdGroupAccess) {
        if (canBeSentToRemoderation(gdAdGroupAccess)) {
            return adGroupsHasShowConditionsDataLoader.get().load(gdAdGroupAccess.getAdGroupId());
        }
        return CompletableFuture.completedFuture(false);
    }

    CompletableFuture<Integer> getCanBeSentToRemoderationAdGroupsCount(List<GdAdGroupAccess> gdAdGroupAccesses) {
        List<Long> adGroupIds = filterAndMapList(gdAdGroupAccesses,
                this::canBeSentToRemoderation,
                GdAdGroupAccess::getAdGroupId);

        if (!adGroupIds.isEmpty()) {
            return adGroupsHasShowConditionsDataLoader.get().loadMany(adGroupIds)
                    .thenApply(ResultConverterHelper::getCountOfTrueBooleanValues);
        }

        return CompletableFuture.completedFuture(0);
    }

    private boolean canBeSentToRemoderation(GdAdGroupAccess gdAdGroupAccess) {
        GridGraphQLContext context = gridContextProvider.getGridContext();
        ClientId clientId = ClientId.fromLong(context.getQueriedClient().getId());
        GdAdGroup adGroup = context.getGdAdGroupsMap().get(gdAdGroupAccess.getAdGroupId());

        GdiBaseCampaign gdiBaseCampaign = campaignInfoService.getAllBaseCampaignsMap(clientId)
                .get(adGroup.getCampaignId());
        return hasOperatorAction(GdOperatorAction.SEND_TO_REMODERATION)
                && canBeSentToRemoderationConditions(gdAdGroupAccess)
                && !isCpmPriceDefaultAdGroupReadyToStart(adGroup, gdiBaseCampaign);
    }

    CompletableFuture<Boolean> getCanAcceptModerationAdGroup(GdAdGroupAccess gdAdGroupAccess) {
        if (hasOperatorAction(GdOperatorAction.ACCEPT_MODERATION) && canAcceptModerationConditions(gdAdGroupAccess)) {
            return adGroupsHasShowConditionsDataLoader.get().load(gdAdGroupAccess.getAdGroupId());
        }

        return CompletableFuture.completedFuture(false);
    }

    CompletableFuture<Integer> getCanAcceptModerationAdGroupsCount(List<GdAdGroupAccess> gdAdGroupAccesses) {
        List<Long> adGroupIds = filterAndMapList(gdAdGroupAccesses,
                AdGroupActionConditionsUtil::canAcceptModerationConditions, GdAdGroupAccess::getAdGroupId);

        if (hasOperatorAction(GdOperatorAction.ACCEPT_MODERATION) && !adGroupIds.isEmpty()) {
            return adGroupsHasShowConditionsDataLoader.get().loadMany(adGroupIds)
                    .thenApply(ResultConverterHelper::getCountOfTrueBooleanValues);
        }

        return CompletableFuture.completedFuture(0);
    }

    CompletableFuture<Boolean> getCanRemoderateAdsCallouts(Long adGroupId) {
        if (hasOperatorAction(GdOperatorAction.REMODERATE_ADS_CALLOUTS)) {
            return adGroupsHasCalloutsDataLoader.get().load(adGroupId);
        }

        return CompletableFuture.completedFuture(false);
    }

    CompletableFuture<Integer> getCanRemoderateAdsCalloutsCount(List<Long> adGroupIds) {
        if (hasOperatorAction(GdOperatorAction.REMODERATE_ADS_CALLOUTS)) {
            return adGroupsHasCalloutsDataLoader.get().loadMany(adGroupIds)
                    .thenApply(ResultConverterHelper::getCountOfTrueBooleanValues);
        }

        return CompletableFuture.completedFuture(0);
    }

    CompletableFuture<Boolean> getCanAcceptAdsCalloutsModeration(Long adGroupId) {
        if (hasOperatorAction(GdOperatorAction.ACCEPT_ADS_CALLOUTS_MODERATION)) {
            return adGroupsHasCalloutsDataLoader.get().load(adGroupId);
        }

        return CompletableFuture.completedFuture(false);
    }

    CompletableFuture<Integer> getCanAcceptAdsCalloutsModerationCount(List<Long> adGroupIds) {
        if (hasOperatorAction(GdOperatorAction.ACCEPT_ADS_CALLOUTS_MODERATION)) {
            return adGroupsHasCalloutsDataLoader.get().loadMany(adGroupIds)
                    .thenApply(ResultConverterHelper::getCountOfTrueBooleanValues);
        }

        return CompletableFuture.completedFuture(0);
    }

    public CompletableFuture<Integer> getAdsCount(Long adGroupId) {
        return adsCountDataLoader.get().load(adGroupId);
    }

    public CompletableFuture<Integer> getKeywordsCount(Long campaignId, Long adGroupId) {
        return keywordsCountDataLoader.get().load(adGroupId, campaignId);
    }

    private boolean hasOperatorAction(GdOperatorAction action) {
        User operator = gridContextProvider.getGridContext().getOperator();
        return OperatorAllowedActionsUtils.hasAction(action, operator);
    }

    List<Long> getDefaultRegionIds(GridGraphQLContext context, Long campaignId) {
        ClientId clientId = ClientId.fromLong(context.getQueriedClient().getId());
        boolean isControlGroup = featureService.isEnabledForClientId(clientId, INTERFACE_CONTROL_GROUP);
        boolean newGeoDefaultEnabled = featureService.isEnabledForClientId(clientId, NEW_GEO_DEFAULT_AND_SUGGEST);
        boolean useNewGeoDefault = newGeoDefaultEnabled && !isControlGroup;

        Map<Long, List<Long>> geoByCampaignId = adGroupService.getDefaultGeoByCampaignId(clientId,
                singleton(campaignId), !useNewGeoDefault);

        return geoByCampaignId.get(campaignId);
    }

    public List<Long> getGeoSuggest() {
        Long currentClientRegionId = currentGeoService.getCurrentDirectRegionId().orElse(null);

        return StreamEx.of(currentClientRegionId, MOSCOW_REGION_ID, SAINT_PETERSBURG_REGION_ID, RUSSIA_REGION_ID)
                .nonNull()
                .distinct()
                .toList();
    }

    public GdKeywordsByCategory recommendedKeywordsByCategory(int shard, ClientId clientId,
                                                              GdAdGroupGetKeywordRecommendationInput input) {
        var result = new GdKeywordsByCategory();

        var campaignTypeValidationResult =
                adGroupKeywordRecommendationValidationService.validateCampaignType(
                        clientId, input.getAdGroupId(), input.getCampaignId());
        if (hasValidationIssues(campaignTypeValidationResult)) {
            return getGdKeywordsByCategoryValidationResult(result, campaignTypeValidationResult);
        }
        Long adGroupId = input.getAdGroupId();
        if (adGroupId == null) {
            // Если не передали adGroupId, значит должны были передать campaignId
            var clientIdValidationResult =
                    adGroupKeywordRecommendationValidationService.validateClientIdWithEnabledFeature(clientId);
            if (hasValidationIssues(clientIdValidationResult)) {
                return getGdKeywordsByCategoryValidationResult(result, clientIdValidationResult);
            }

            Long campaignId = input.getCampaignId();
            List<Long> adGroupIds =
                    adGroupRepository
                            .getAdGroupIdsByCampaignIds(shard, Set.of(campaignId))
                            .getOrDefault(campaignId, null);
            var campaignIdValidationResult =
                    adGroupKeywordRecommendationValidationService.validateCampaignId(adGroupIds);
            if (hasValidationIssues(campaignIdValidationResult)) {
                return getGdKeywordsByCategoryValidationResult(result, campaignIdValidationResult);
            }

            adGroupId = adGroupIds.get(0);
        }
        AdditionalInfo info = getAdditionalInfo(input);
        var internalInput = new AdGroupKeywordRecommendationInput(adGroupId, info);
        var recommendations =
                adGroupKeywordRecommendationService.recommendedKeywords(clientId, internalInput);
        Map<RelevanceMatchCategory, List<String>> keywords = recommendations.isSuccessful() ?
                recommendations.getResult() : emptyMap();
        var gdKeywords = EntryStream.of(keywords)
                .mapKeys(k -> GdRelevanceMatchCategory.fromTypedValue(k.name()))
                .toMap();
        if (!featureService.isEnabledForClientId(clientId, FeatureName.RELEVANCE_MATCH_CATEGORIES_ALLOWED_IN_UC)) {
            return getGdKeywordsByCategory(result, campaignTypeValidationResult, gdKeywords);
        }
        if (Utils.isNotEmpty(input.getRelevanceMatchCategory())) {
            Set<GdRelevanceMatchCategory> filterRelevanceMatchCategories =
                    StreamEx.of(input.getRelevanceMatchCategory()).toSet();
            gdKeywords = EntryStream.of(gdKeywords)
                    .filterKeys(filterRelevanceMatchCategories::contains)
                    .toMap();
        }
        var filteredGdKeywords = deleteDuplicatedStringsAndCut(gdKeywords);
        return getGdKeywordsByCategory(result, campaignTypeValidationResult, filteredGdKeywords);
    }

    private GdKeywordsByCategory getGdKeywordsByCategoryValidationResult(
            GdKeywordsByCategory result,
            ValidationResult validationResult
    ) {
        return getGdKeywordsByCategory(result, validationResult, emptyMap());
    }

    private GdKeywordsByCategory getGdKeywordsByCategory(
            GdKeywordsByCategory result,
            ValidationResult validationResult,
            Map<GdRelevanceMatchCategory, List<String>> gdKeywords
    ) {
        var gdValidationResult =
                gridValidationService.toGdValidationResult(validationResult, new Path(emptyList()));
        return result.withKeywordByCategory(gdKeywords)
                .withValidationResult(gdValidationResult);
    }

    private AdditionalInfo getAdditionalInfo(GdAdGroupGetKeywordRecommendationInput input) {
        AdditionalInfo info = null;
        if (input.getKeywordRecommendationData() != null) {
            info = new AdGroupInfo(
                    input.getKeywordRecommendationData().getRegionIds().stream().map(Integer::longValue).collect(Collectors.toList()),
                    input.getKeywordRecommendationData().getMinusKeywords(),
                    input.getKeywordRecommendationData().getKeywords());
        } else if (input.getKeywordRecommendationByBannerData() != null) {
            info = new BannerInfo(
                    input.getKeywordRecommendationByBannerData().getTitle(),
                    input.getKeywordRecommendationByBannerData().getBody());
        }
        return info;
    }

    private Map<GdRelevanceMatchCategory, List<String>> deleteDuplicatedStringsAndCut(
            Map<GdRelevanceMatchCategory, List<String>> keywordsByCategory) {
        // список категорий по фразам, связанных с этими категориями
        Map<String, Set<GdRelevanceMatchCategory>> categoriesByKeyword = new LinkedHashMap<>();
        keywordsByCategory.forEach((category, keywords) ->
                keywords.forEach(keyword -> {
                    var categories = categoriesByKeyword.getOrDefault(keyword, new LinkedHashSet<>());
                    categories.add(category);
                    categoriesByKeyword.put(keyword, categories);
                }));
        // список отсортированных по кол-ву категорий, привязанных к этой фразе
        var keywordsCategories =
                StreamEx.of(categoriesByKeyword.entrySet())
                        .sorted(Comparator.comparingInt(o -> o.getValue().size()))
                        .toList();

        Map<GdRelevanceMatchCategory, List<String>> result = new LinkedHashMap<>();

        for (Map.Entry<String, Set<GdRelevanceMatchCategory>> entry : keywordsCategories) {
            for (GdRelevanceMatchCategory category : entry.getValue()) {
                List<String> keywords = result.getOrDefault(category, new ArrayList<>());
                if (keywords.size() < MAX_RECOMMENDED_KEYWORDS_COUNT) {
                    keywords.add(entry.getKey());
                    result.put(category, keywords);
                    break;
                }
            }
        }
        return result;
    }

    private List<GdiGroup> filterByAggregatedStatuses(Map<Long, AggregatedStatusAdGroupData> adGroupStatusesByIds,
                                                      List<GdiGroup> adGroups,
                                                      GdiGroupFilter filter) {
        if (adGroups.isEmpty()) {
            return adGroups;
        }

        // Значения filter.getArchived(): true - показ архивных; false - не показываем архивные (для показа всех,
        // кроме архивных);  null - для показа всех, либо показа по фильтру статусов
        boolean withoutArchived = Boolean.FALSE.equals(filter.getArchived());
        Set<GdiGroupPrimaryStatus> statuses = new HashSet<>(emptyIfNull(filter.getPrimaryStatusIn()));
        if (Boolean.TRUE.equals(filter.getArchived())) {
            statuses.add(GdiGroupPrimaryStatus.ARCHIVED);
        }

        return adGroups.stream()
                .filter(g -> statuses.isEmpty()
                        || statuses.contains(AdGroupPrimaryStatusCalculator
                        .convertToPrimaryStatus(adGroupStatusesByIds.get(g.getId()))))
                .filter(g -> !withoutArchived
                        || !adGroupStatusesByIds.containsKey(g.getId())
                        || adGroupStatusesByIds.get(g.getId()).getStatus().isEmpty()
                        || !adGroupStatusesByIds.get(g.getId()).getStatus().get().equals(GdSelfStatusEnum.ARCHIVED))
                .filter(g -> ReasonsFilterUtils.isValid(filter.getReasonsContainSome(),
                        adGroupStatusesByIds.get(g.getId())))
                .collect(Collectors.toList());
    }
}
