package ru.yandex.direct.grid.processing.service.group.converter;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import io.leangen.graphql.annotations.GraphQLNonNull;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import ru.yandex.direct.common.TranslationService;
import ru.yandex.direct.core.entity.adgroup.container.ComplexContentPromotionAdGroup;
import ru.yandex.direct.core.entity.adgroup.container.ComplexCpmAdGroup;
import ru.yandex.direct.core.entity.adgroup.container.ComplexDynamicAdGroup;
import ru.yandex.direct.core.entity.adgroup.container.ComplexMcBannerAdGroup;
import ru.yandex.direct.core.entity.adgroup.container.ComplexMobileContentAdGroup;
import ru.yandex.direct.core.entity.adgroup.container.ComplexPerformanceAdGroup;
import ru.yandex.direct.core.entity.adgroup.container.ComplexTextAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.AdShowType;
import ru.yandex.direct.core.entity.adgroup.model.ContentPromotionAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.CpmAudioAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.CpmBannerAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.CpmGeoPinAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.CpmGeoproductAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.CpmIndoorAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.CpmOutdoorAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.CpmVideoAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.CpmYndxFrontpageAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.CriterionType;
import ru.yandex.direct.core.entity.adgroup.model.DynamicAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.McBannerAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.MobileContentAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.MobileContentAdGroupDeviceTypeTargeting;
import ru.yandex.direct.core.entity.adgroup.model.MobileContentAdGroupNetworkTargeting;
import ru.yandex.direct.core.entity.adgroup.model.PageBlock;
import ru.yandex.direct.core.entity.adgroup.model.PerformanceAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.TextAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.UsersSegment;
import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.core.entity.banner.model.PerformanceBannerMain;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.keyword.service.KeywordUtils;
import ru.yandex.direct.core.entity.offerretargeting.model.OfferRetargeting;
import ru.yandex.direct.core.entity.relevancematch.model.RelevanceMatch;
import ru.yandex.direct.core.entity.relevancematch.model.RelevanceMatchCategory;
import ru.yandex.direct.core.entity.retargeting.RetargetingTranslations;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.model.Rule;
import ru.yandex.direct.core.entity.retargeting.model.TargetInterest;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.grid.processing.model.bidmodifier.mutation.GdUpdateBidModifiers;
import ru.yandex.direct.grid.processing.model.bidmodifier.mutation.GdUpdateBidModifiersItem;
import ru.yandex.direct.grid.processing.model.group.GdAdGroupPageBlock;
import ru.yandex.direct.grid.processing.model.group.GdCriterionType;
import ru.yandex.direct.grid.processing.model.group.GdVideoGoalType;
import ru.yandex.direct.grid.processing.model.group.mutation.GdAddAdGroupMinusKeywordsPayloadItem;
import ru.yandex.direct.grid.processing.model.group.mutation.GdAddAdGroupPayload;
import ru.yandex.direct.grid.processing.model.group.mutation.GdAddAdGroupPayloadItem;
import ru.yandex.direct.grid.processing.model.group.mutation.GdAddContentPromotionAdGroupItem;
import ru.yandex.direct.grid.processing.model.group.mutation.GdAddMcBannerAdGroupItem;
import ru.yandex.direct.grid.processing.model.group.mutation.GdAddMobileContentAdGroupItem;
import ru.yandex.direct.grid.processing.model.group.mutation.GdRelevanceMatchCategory;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateAdGroupInterestItem;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateAdGroupKeywordItem;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateAdGroupOfferRetargetingItem;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateAdGroupPayload;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateAdGroupPayloadItem;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateAdGroupRelevanceMatchItem;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateAdGroupRetargetingItem;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateContentPromotionAdGroupItem;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateCpmAdGroupItem;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateDynamicAdGroupItem;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateMcBannerAdGroupItem;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateMobileContentAdGroupItem;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdatePerformanceAdGroupItem;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateTextAdGroupItem;
import ru.yandex.direct.grid.processing.model.retargeting.GdRetargetingConditionRuleItemReq;
import ru.yandex.direct.grid.processing.model.retargeting.mutation.GdUpdateCpmRetargetingConditionItem;
import ru.yandex.direct.grid.processing.service.showcondition.converter.RetargetingConverter;
import ru.yandex.direct.grid.processing.util.ResultConverterHelper;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.utils.ListUtils;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static java.util.Objects.nonNull;
import static java.util.function.Function.identity;
import static org.apache.commons.lang3.StringUtils.defaultIfEmpty;
import static org.springframework.util.CollectionUtils.isEmpty;
import static ru.yandex.direct.core.entity.adgroup.model.CriterionType.CONTENT_CATEGORY;
import static ru.yandex.direct.core.entity.adgroup.model.CriterionType.KEYWORD;
import static ru.yandex.direct.core.entity.adgroup.model.CriterionType.USER_PROFILE;
import static ru.yandex.direct.core.entity.keyword.service.KeywordUtils.hasAutotargetingPrefix;
import static ru.yandex.direct.core.entity.keyword.service.KeywordUtils.phraseWithoutAutotargetingPrefix;
import static ru.yandex.direct.core.entity.retargeting.model.ConditionType.interests;
import static ru.yandex.direct.core.entity.retargeting.service.RetargetingUtils.getTargetInterestWithInterestsCondition;
import static ru.yandex.direct.grid.processing.service.bidmodifier.BidModifierDataConverter.toComplexBidModifier;
import static ru.yandex.direct.grid.processing.service.group.AdGroupTypeUtils.CONTENT_PROMOTION_ADGROUP_CORE_TYPE_BY_GD_TYPE;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;
import static ru.yandex.direct.utils.ListUtils.integerToLongList;

@ParametersAreNonnullByDefault
public class AdGroupsMutationDataConverter {

    public static List<TargetInterest> toTargetInterests(@Nullable Long campaignId, @Nullable Long adGroupId,
                                                         List<GdUpdateAdGroupRetargetingItem> adGroupRetargetingItems,
                                                         List<GdUpdateAdGroupInterestItem> adGroupInterestItems) {
        // Порядок важен! Т.к. ретаргетинги и интересы обрабатываются вместе то по порядку определяем в ошибках
        // валидации про что идет речь
        List<TargetInterest> targetInterests =
                mapList(adGroupRetargetingItems, r -> toTargetInterest(campaignId, adGroupId, r));
        targetInterests.addAll(mapList(adGroupInterestItems, r -> toTargetInterest(campaignId, adGroupId, r)));
        return targetInterests;

    }

    public static TargetInterest toTargetInterest(Long adGroupId,
                                                  GdUpdateAdGroupRetargetingItem adGroupRetargetingItem) {
        return toTargetInterest(null, adGroupId, adGroupRetargetingItem);
    }

    public static TargetInterest toTargetInterest(@Nullable Long campaignId, @Nullable Long adGroupId,
                                                  GdUpdateAdGroupRetargetingItem adGroupRetargetingItem) {
        return new TargetInterest()
                .withCampaignId(campaignId)
                .withAdGroupId(adGroupId)
                .withId(adGroupRetargetingItem.getId())
                .withRetargetingConditionId(adGroupRetargetingItem.getRetCondId());
    }

    public static TargetInterest toTargetInterest(@Nullable Long campaignId,
                                                  @Nullable Long adGroupId,
                                                  GdUpdateAdGroupInterestItem adGroupInterestItem) {
        return new TargetInterest()
                .withCampaignId(campaignId)
                .withAdGroupId(adGroupId)
                .withId(adGroupInterestItem.getId())
                .withInterestId(adGroupInterestItem.getCategoryId());
    }

    public static TextAdGroup toTextAdGroup(GdUpdateTextAdGroupItem adGroupUpdateItem, AdGroupType adGroupType) {
        return new TextAdGroup()
                .withId(adGroupUpdateItem.getAdGroupId())
                .withName(adGroupUpdateItem.getAdGroupName())
                .withMinusKeywords(adGroupUpdateItem.getAdGroupMinusKeywords())
                .withLibraryMinusKeywordsIds(adGroupUpdateItem.getLibraryMinusKeywordsIds())
                .withGeo(mapList(adGroupUpdateItem.getRegionIds(), Integer::longValue))
                .withHyperGeoId(adGroupUpdateItem.getHyperGeoId())
                .withType(adGroupType)
                .withPageGroupTags(adGroupUpdateItem.getPageGroupTags())
                .withTargetTags(adGroupUpdateItem.getTargetTags())
                .withContentCategoriesRetargetingConditionRules(
                        toCoreContentCategoriesRetargetingRules(
                                adGroupUpdateItem.getContentCategoriesRetargetingConditionRules()))
                .withFieldToUseAsName(adGroupUpdateItem.getFieldToUseAsName())
                .withFieldToUseAsBody(adGroupUpdateItem.getFieldToUseAsBody())
                .withFeedId(adGroupUpdateItem.getSmartFeedId())
                .withUsersSegments(Boolean.TRUE.equals(adGroupUpdateItem.getCollectAudience()) ?
                        List.of(new UsersSegment()
                                .withAdGroupId(adGroupUpdateItem.getAdGroupId())
                                .withType(AdShowType.START)) : emptyList());
    }

    public static List<Rule> toCoreContentCategoriesRetargetingRules(List<GdRetargetingConditionRuleItemReq> rules) {
        if (rules == null) {
            return null;
        }

        List<Rule> coreRules = StreamEx.of(rules)
                .map(RetargetingConverter::toCoreRetargetingConditionRule)
                .remove(rule -> rule.getGoals() == null || rule.getGoals().isEmpty())
                .toList();

        if (coreRules.isEmpty()) {
            return null;
        }

        return coreRules;
    }

    public static List<Keyword> toCoreKeywords(List<GdUpdateAdGroupKeywordItem> updateKeywordItems,
                                               @Nullable List<Keyword> adGroupKeywords) {
        return KeywordUtils.mergeNewKeywordsWithExisting(
                mapList(updateKeywordItems, AdGroupsMutationDataConverter::toCoreKeyword),
                adGroupKeywords
        );
    }

    public static Keyword toCoreKeyword(GdUpdateAdGroupKeywordItem keywordItem) {
        return new Keyword()
                .withId(keywordItem.getId())
                .withPhrase(phraseWithoutAutotargetingPrefix(keywordItem.getPhrase()))
                .withIsAutotargeting(hasAutotargetingPrefix(keywordItem.getPhrase()));
    }

    public static List<RelevanceMatch> toCoreRelevanceMatches(GdUpdateAdGroupRelevanceMatchItem relevanceMatchItem) {
        if (!relevanceMatchItem.getIsActive()) {
            return Collections.emptyList();
        }
        return singletonList(new RelevanceMatch()
                .withId(relevanceMatchItem.getId())
                .withRelevanceMatchCategories(
                        toCoreRelevanceMatchCategories(relevanceMatchItem.getRelevanceMatchCategories())));
    }

    public static List<OfferRetargeting> toCoreOfferRetargetings(GdUpdateAdGroupOfferRetargetingItem offerRetargetingItem) {
        if (!offerRetargetingItem.getIsActive()) {
            return Collections.emptyList();
        }
        return singletonList(new OfferRetargeting().withId(offerRetargetingItem.getId()));
    }

    @Nullable
    public static Set<RelevanceMatchCategory> toCoreRelevanceMatchCategories(
            @Nullable Set<GdRelevanceMatchCategory> relevanceMatchCategories) {
        return mapSet(relevanceMatchCategories, category -> RelevanceMatchCategory.valueOf(category.getTypedValue()));
    }

    public static List<GdAddAdGroupMinusKeywordsPayloadItem> toAddedItems(
            @Nullable Map<Long, List<String>> normalMinusKeywords) {
        if (normalMinusKeywords == null) {
            return Collections.emptyList();
        }
        return EntryStream.of(normalMinusKeywords)
                .mapKeyValue((adGroupId, minusKeywords) -> new GdAddAdGroupMinusKeywordsPayloadItem()
                        .withAdGroupId(adGroupId)
                        .withAddedMinusKeywords(minusKeywords))
                .toList();
    }

    /**
     * Преобразует запрос на редактирование группы в контейнер с параметрами для комплексной операции.
     *
     * @param adGroupUpdateItems   исходный запрос на редактирование группы
     * @param keywordsByAdGroupIds ключевые слова группы, сохраненные до текущего момента
     * @param adGroupTypes         тип группы
     * @param interestsByAdGroupId условия ретаргетинга, сгруппированные по ID группы
     * @param interestsConditions  формулы ретаргетинга с типом interests, уже привязанные к группам
     * @param clientId             ID клиента
     * @return {@link List<ComplexTextAdGroup>}  контейнер для выполнения коплексной операции
     */
    public static List<ComplexTextAdGroup> toComplexAdGroups(List<GdUpdateTextAdGroupItem> adGroupUpdateItems,
                                                             Map<Long, List<Keyword>> keywordsByAdGroupIds,
                                                             Map<Long, AdGroupType> adGroupTypes,
                                                             Map<Long, List<TargetInterest>> interestsByAdGroupId,
                                                             Map<Long, RetargetingCondition> interestsConditions,
                                                             ClientId clientId,
                                                             TranslationService translationService,
                                                             FeatureService featureService) {
        boolean isSearchRetargetingEnabled = featureService
                .isEnabledForClientId(clientId, FeatureName.SEARCH_RETARGETING_ENABLED);
        return mapList(adGroupUpdateItems, adGroupUpdateItem -> toComplexTextAdGroup(adGroupUpdateItem,
                keywordsByAdGroupIds.get(adGroupUpdateItem.getAdGroupId()),
                adGroupTypes.get(adGroupUpdateItem.getAdGroupId()),
                interestsByAdGroupId.get(adGroupUpdateItem.getAdGroupId()),
                clientId, interestsConditions.keySet(), translationService, isSearchRetargetingEnabled));
    }

    private static ComplexTextAdGroup toComplexTextAdGroup(GdUpdateTextAdGroupItem adGroupUpdateItem,
                                                           @Nullable List<Keyword> adGroupKeywords,
                                                           AdGroupType adGroupType,
                                                           @Nullable List<TargetInterest> targetInterests,
                                                           ClientId clientId, Set<Long> interestsRetCondIds,
                                                           TranslationService translationService,
                                                           boolean isSearchRetargetingEnabled) {
        ComplexTextAdGroup complexTextAdGroup = new ComplexTextAdGroup();

        List<TargetInterest> targetInterestsToUpdate = mapList(adGroupUpdateItem.getRetargetings(),
                r -> toTargetInterest(adGroupUpdateItem.getAdGroupId(), r));

        if (adGroupUpdateItem.getRetargetingCondition() != null
                && !isEmpty(adGroupUpdateItem.getRetargetingCondition().getConditionRules())) {
            TargetInterest targetInterestWithInterestsCondition = getTargetInterestWithInterestsCondition(
                    interestsRetCondIds,
                    adGroupUpdateItem.getAdGroupId(),
                    targetInterests
            );

            RetargetingCondition retargetingCondition = toRetargetingCondition(clientId,
                    targetInterestWithInterestsCondition, adGroupUpdateItem.getRetargetingCondition(),
                    false, translationService);

            complexTextAdGroup
                    .withRetargetingCondition(retargetingCondition);

            if (isEmpty(targetInterestsToUpdate)) {
                targetInterestsToUpdate = singletonList(targetInterestWithInterestsCondition);
            } else {
                targetInterestsToUpdate.add(targetInterestWithInterestsCondition);
            }
        }

        return complexTextAdGroup
                .withKeywords(adGroupUpdateItem.getKeywords() != null
                        ? toCoreKeywords(adGroupUpdateItem.getKeywords(), adGroupKeywords)
                        : null)  //Todo: удалить проверку на null c выходом изменений на фронте (DIRECT-88468)
                .withRelevanceMatches(adGroupUpdateItem.getRelevanceMatch() != null
                        ? toCoreRelevanceMatches(adGroupUpdateItem.getRelevanceMatch())
                        : null)  //Todo: удалить проверку на null c выходом изменений на фронте (DIRECT-89923)
                .withOfferRetargetings(adGroupUpdateItem.getOfferRetargeting() != null
                        ? toCoreOfferRetargetings(adGroupUpdateItem.getOfferRetargeting())
                        : null)
                .withAdGroup(toTextAdGroup(adGroupUpdateItem, adGroupType))
                .withComplexBidModifier(isSearchRetargetingEnabled
                        ? toComplexBidModifier(
                        adGroupUpdateItem.getSearchRetargetings(), adGroupUpdateItem.getBidModifiers(),
                        null, adGroupUpdateItem.getAdGroupId())
                        : toComplexBidModifier(adGroupUpdateItem.getBidModifiers()))
                .withTargetInterests(targetInterestsToUpdate);
    }

    public static ComplexTextAdGroup toComplexTextAdGroupForCa(GdUpdateTextAdGroupItem adGroupUpdateItem,
                                                               @Nullable List<Keyword> adGroupKeywords,
                                                               AdGroupType adGroupType,
                                                               @Nullable List<TargetInterest> targetInterests,
                                                               ClientId clientId, Set<Long> interestsRetCondIds,
                                                               TranslationService translationService,
                                                               boolean isSearchRetargetingEnabled,
                                                               List<Goal> goals) {
        ComplexTextAdGroup complexTextAdGroup = new ComplexTextAdGroup();

        List<TargetInterest> targetInterestsToUpdate = mapList(adGroupUpdateItem.getRetargetings(),
                r -> toTargetInterest(adGroupUpdateItem.getAdGroupId(), r));

        if (!isEmpty(goals)) {
            var targetInterestWithInterestsCondition = getTargetInterestWithInterestsCondition(
                    interestsRetCondIds,
                    adGroupUpdateItem.getAdGroupId(),
                    targetInterests
            );

            var retargetingCondition = toCaRetargetingCondition(clientId,
                    targetInterestWithInterestsCondition, goals, translationService);

            complexTextAdGroup.withRetargetingCondition(retargetingCondition);

            if (isEmpty(targetInterestsToUpdate)) {
                targetInterestsToUpdate = singletonList(targetInterestWithInterestsCondition);
            } else {
                targetInterestsToUpdate.add(targetInterestWithInterestsCondition);
            }
        }

        return complexTextAdGroup
                .withKeywords(adGroupUpdateItem.getKeywords() != null
                        ? toCoreKeywords(adGroupUpdateItem.getKeywords(), adGroupKeywords)
                        : null)  //Todo: удалить проверку на null c выходом изменений на фронте (DIRECT-88468)
                .withRelevanceMatches(adGroupUpdateItem.getRelevanceMatch() != null
                        ? toCoreRelevanceMatches(adGroupUpdateItem.getRelevanceMatch())
                        : null)  //Todo: удалить проверку на null c выходом изменений на фронте (DIRECT-88468)
                .withOfferRetargetings(adGroupUpdateItem.getOfferRetargeting() != null
                        ? toCoreOfferRetargetings(adGroupUpdateItem.getOfferRetargeting())
                        : null)
                .withAdGroup(toTextAdGroup(adGroupUpdateItem, adGroupType))
                .withComplexBidModifier(isSearchRetargetingEnabled
                        ? toComplexBidModifier(
                        adGroupUpdateItem.getSearchRetargetings(), adGroupUpdateItem.getBidModifiers(),
                        null, adGroupUpdateItem.getAdGroupId())
                        : toComplexBidModifier(adGroupUpdateItem.getBidModifiers()))
                .withTargetInterests(targetInterestsToUpdate);
    }

    public static RetargetingCondition toRetargetingCondition(ClientId clientId,
                                                              TargetInterest targetInterestWithInterestsCondition,
                                                              GdUpdateCpmRetargetingConditionItem retargetingCondition,
                                                              boolean defaultNameIfEmpty,
                                                              TranslationService translationService) {
        List<GdRetargetingConditionRuleItemReq> conditionRules = ifNotNull(retargetingCondition,
                GdUpdateCpmRetargetingConditionItem::getConditionRules);
        String retargetingName = ifNotNull(retargetingCondition,
                GdUpdateCpmRetargetingConditionItem::getName);
        RetargetingCondition condition = toCoreRetargetingCondition(clientId,
                targetInterestWithInterestsCondition.getRetargetingConditionId(),
                defaultNameIfEmpty ?
                        defaultIfEmpty(retargetingName,
                                translationService.translate(RetargetingTranslations.INSTANCE.newAudience()))
                        : retargetingName,
                nvl(conditionRules, emptyList()));
        condition.withTargetInterest(targetInterestWithInterestsCondition);

        return condition;
    }

    public static RetargetingCondition toCaRetargetingCondition(ClientId clientId,
                                                                TargetInterest targetInterest,
                                                                List<Goal> goals,
                                                                TranslationService translationService) {
        return (RetargetingCondition) new RetargetingCondition()
                .withId(targetInterest.getRetargetingConditionId())
                .withName(translationService.translate(RetargetingTranslations.INSTANCE.newAudience()))
                .withType(interests)
                .withClientId(clientId.asLong())
                .withTargetInterest(targetInterest)
                .withRules(List.of(RetargetingConverter.toCaCoreRetargetingConditionRule(goals)));
    }

    /**
     * Создать пустой ответ GdAddAdGroupPayload без элементов и ошибок валидации
     */
    public static GdAddAdGroupPayload createEmptyGdAddAdGroupPayload() {
        return new GdAddAdGroupPayload().withAddedAdGroupItems(emptyList());
    }

    /**
     * Создать пустой ответ GdUpdateAdGroupPayload без элементов и ошибок валидации
     */
    public static GdUpdateAdGroupPayload createEmptyGdUpdateAdGroupPayload() {
        return new GdUpdateAdGroupPayload().withUpdatedAdGroupItems(emptyList());
    }

    public static ComplexPerformanceAdGroup toComplexPerformanceAdGroup(
            @GraphQLNonNull GdUpdatePerformanceAdGroupItem item,
            // nullable до тех пор, пока полностью не перейдем на новую схему
            @Nullable Map<Long, PerformanceBannerMain> bannerByAdGroupId,
            boolean isSearchRetargetingEnabled) {
        List<Long> geo = mapList(item.getRegionIds(), Integer::longValue);
        return new ComplexPerformanceAdGroup()
                .withAdGroup(new PerformanceAdGroup()
                        .withId(item.getId())
                        .withType(AdGroupType.PERFORMANCE)
                        .withName(item.getName())
                        .withTrackingParams(item.getTrackingParams())
                        .withFieldToUseAsName(item.getFieldToUseAsName())
                        .withFieldToUseAsBody(item.getFieldToUseAsBody())
                        .withMinusKeywords(item.getMinusKeywords())
                        .withLibraryMinusKeywordsIds(item.getLibraryMinusKeywordsIds())
                        .withGeo(geo)
                        .withPageGroupTags(item.getPageGroupTags())
                        .withTargetTags(item.getTargetTags())
                        .withHyperGeoId(item.getHyperGeoId())
                )
                .withComplexBidModifier(isSearchRetargetingEnabled
                        ? toComplexBidModifier(item.getSearchRetargetings(), item.getBidModifiers(), null, item.getId())
                        : toComplexBidModifier(item.getBidModifiers()))
                .withBanners(bannerByAdGroupId == null ? null : List.of(new PerformanceBannerMain()
                        .withId(ifNotNull(bannerByAdGroupId.get(item.getId()), Banner::getId))
                        .withAdGroupId(item.getId())
                        .withLogoImageHash(item.getLogoImageHash())));
    }

    public static List<GdUpdateAdGroupPayloadItem> getGdUpdateAdGroupPayloadItems(MassResult<Long> result) {
        List<Long> adGroupIds = ResultConverterHelper.getSuccessfullyUpdatedIds(result, identity());
        return mapList(adGroupIds, g -> new GdUpdateAdGroupPayloadItem().withAdGroupId(g));
    }

    public static List<GdAddAdGroupPayloadItem> getGdAddAdGroupPayloadItems(MassResult<Long> result) {
        List<Long> adGroupIds = ResultConverterHelper.getSuccessfullyUpdatedIds(result, identity());
        return mapList(adGroupIds, g -> new GdAddAdGroupPayloadItem().withAdGroupId(g));
    }


    public static ComplexCpmAdGroup fromUpdateItemToComplexCpmAdGroup(ClientId clientId,
                                                                      GdUpdateCpmAdGroupItem cpmAdGroupItem,
                                                                      List<TargetInterest> targetInterests,
                                                                      boolean isNew,
                                                                      TranslationService translationService,
                                                                      boolean isSearchRetargetingEnabled) {
        AdGroup adGroup = cpmAdGroupToCoreCpmBannerAdGroup(cpmAdGroupItem);
        GdUpdateBidModifiers bidModifiers = cpmAdGroupItem.getBidModifiers();

        // Обнуляем CampaignId у корректировок по новым группам, чтобы обойти ненужную проверку в классах типа
        // BidModifierValidationMobileTypeSupport
        // Сами проверки уже выполняются в классах AddComplexCpmAdGroupValidationService и
        // UpdateComplexCpmAdGroupValidationService
        // Смотри тест SaveCpmAdGroupMutationServiceTest#createCpmAdGroups_bidmodifiers (падает без обнуления
        // CampaignId)
        if (adGroup.getId() == null && bidModifiers != null) {
            ArrayList<GdUpdateBidModifiersItem> bidModifiersItems = new ArrayList<>(asList(
                    bidModifiers.getBidModifierDemographics(),
                    bidModifiers.getBidModifierAdType(),
                    bidModifiers.getBidModifierDesktop(),
                    bidModifiers.getBidModifierDesktopOnly(),
                    bidModifiers.getBidModifierSmartTV(),
                    bidModifiers.getBidModifierGeo(),
                    bidModifiers.getBidModifierInventory(),
                    bidModifiers.getBidModifierMobile(),
                    bidModifiers.getBidModifierTablet(),
                    bidModifiers.getBidModifierRetargeting(),
                    bidModifiers.getBidModifierSmartTgo(),
                    bidModifiers.getBidModifierVideo(),
                    bidModifiers.getBidModifierWeather()
            ));
            bidModifiersItems.addAll(nvl(bidModifiers.getBidModifierExpress(), emptyList()));
            filterAndMapList(bidModifiersItems, Objects::nonNull, b -> b.withCampaignId(null));
        }

        ComplexCpmAdGroup complexCpmAdGroup = new ComplexCpmAdGroup()
                .withAdGroup(adGroup)
                .withComplexBidModifier(isSearchRetargetingEnabled
                        ? toComplexBidModifier(cpmAdGroupItem.getSearchRetargetings(), bidModifiers,
                        cpmAdGroupItem.getCampaignId(), cpmAdGroupItem.getAdGroupId())
                        : toComplexBidModifier(bidModifiers)
                );

        // Для охватных групп может быть одно или ни одного условия ретаргетинга.
        // Его нет для баннеров с ключевыми словами, для остальных оно должно существовать с момента создания группы.
        if (cpmAdGroupItem.getKeywords() == null) {
            TargetInterest targetInterest;
            if (isNew) {
                targetInterest = new TargetInterest();
            } else {
                checkState(targetInterests.size() == 1, "There should be exactly one target for group %s, found %s",
                        cpmAdGroupItem.getAdGroupId(), targetInterests.size());
                targetInterest = targetInterests.get(0);
            }

            //Создаем условие ретаргетинга, даже если в retargetingCondition пришел null, это нужно для индор,
            // аутдор и прайсовых групп,
            //у которых условие ретаргетинга пустое и поэтому с фронта его не требуем
            RetargetingCondition retargetingCondition = toRetargetingCondition(clientId, targetInterest,
                    cpmAdGroupItem.getRetargetingCondition(), true, translationService);

            complexCpmAdGroup
                    .withRetargetingConditions(singletonList(retargetingCondition))
                    .withTargetInterests(singletonList(targetInterest));
        }

        return complexCpmAdGroup
                .withKeywords(mapList(cpmAdGroupItem.getKeywords(), AdGroupsMutationDataConverter::toCoreKeyword))
                .withUsersSegments(mapList(cpmAdGroupItem.getVideoGoals(),
                        g -> toCoreVideoGoal(g, cpmAdGroupItem.getAdGroupId())));
    }

    public static boolean hasContentCategories(GdUpdateCpmAdGroupItem cpmAdGroup) {
        return cpmAdGroup.getContentCategoriesRetargetingConditionRules() != null
                && cpmAdGroup.getContentCategoriesRetargetingConditionRules().stream()
                .map(GdRetargetingConditionRuleItemReq::getGoals)
                .mapToLong(List::size)
                .sum() != 0;
    }

    private static UsersSegment toCoreVideoGoal(GdVideoGoalType videoGoalType, Long adGroupId) {
        return new UsersSegment()
                .withAdGroupId(adGroupId)
                .withType(GdVideoGoalType.toSource(videoGoalType));
    }

    private static AdGroup cpmAdGroupToCoreCpmBannerAdGroup(GdUpdateCpmAdGroupItem cpmAdGroup) {
        List<GdAdGroupPageBlock> pageBlocks = nvl(cpmAdGroup.getPageBlocks(), emptyList());

        AdGroup adGroup;
        switch (cpmAdGroup.getType()) {
            case CPM_BANNER:
            case CPM_PRICE_BANNER: {
                adGroup = new CpmBannerAdGroup()
                        .withType(AdGroupType.CPM_BANNER)
                        .withForInBanners(cpmAdGroup.getForInBanner())
                        .withCriterionType(calcCpmCriterionType(cpmAdGroup))
                        .withPriority(cpmAdGroup.getPriority());
                break;
            }
            case CPM_VIDEO:
            case CPM_PRICE_FRONTPAGE_VIDEO:
            case CPM_PRICE_VIDEO: {
                adGroup = new CpmVideoAdGroup()
                        .withType(AdGroupType.CPM_VIDEO)
                        .withIsNonSkippable(cpmAdGroup.getIsNonSkippable())
                        .withCriterionType(calcCpmCriterionType(cpmAdGroup))
                        .withPriority(cpmAdGroup.getPriority());
                break;
            }
            case CPM_AUDIO:
            case CPM_PRICE_AUDIO: {
                adGroup = new CpmAudioAdGroup()
                        .withType(AdGroupType.CPM_AUDIO)
                        .withPriority(cpmAdGroup.getPriority());
                break;
            }
            case CPM_OUTDOOR: {
                adGroup = new CpmOutdoorAdGroup()
                        .withType(AdGroupType.CPM_OUTDOOR)
                        .withPageBlocks(mapList(pageBlocks, AdGroupsMutationDataConverter::toCorePageBlock));
                break;
            }
            case CPM_INDOOR: {
                adGroup = new CpmIndoorAdGroup()
                        .withType(AdGroupType.CPM_INDOOR)
                        .withPageBlocks(mapList(pageBlocks, AdGroupsMutationDataConverter::toCorePageBlock));
                break;
            }
            case CPM_GEOPRODUCT: {
                adGroup = new CpmGeoproductAdGroup()
                        .withType(AdGroupType.CPM_GEOPRODUCT);
                break;
            }
            case CPM_GEO_PIN: {
                adGroup = new CpmGeoPinAdGroup()
                        .withType(AdGroupType.CPM_GEO_PIN);
                break;
            }
            case CPM_PRICE:
            case CPM_YNDX_FRONTPAGE: {
                adGroup = new CpmYndxFrontpageAdGroup()
                        .withType(AdGroupType.CPM_YNDX_FRONTPAGE)
                        .withPriority(cpmAdGroup.getPriority());
                break;
            }
            default: {
                throw new IllegalArgumentException("Unsupported type");
            }
        }

        return adGroup.withId(cpmAdGroup.getAdGroupId())
                .withCampaignId(cpmAdGroup.getCampaignId())
                .withName(cpmAdGroup.getAdGroupName())
                .withGeo(ifNotNull(cpmAdGroup.getRegionIds(), ListUtils::integerToLongList))
                .withPageGroupTags(cpmAdGroup.getPageGroupTags())
                .withTargetTags(cpmAdGroup.getTargetTags())
                .withProjectParamConditions(cpmAdGroup.getProjectParamConditions())
                .withContentCategoriesRetargetingConditionRules(
                        toCoreContentCategoriesRetargetingRules(
                                cpmAdGroup.getContentCategoriesRetargetingConditionRules()));
    }

    private static RetargetingCondition toCoreRetargetingCondition(ClientId clientId, @Nullable Long retCondId,
                                                                   String name,
                                                                   List<GdRetargetingConditionRuleItemReq>
                                                                           conditionRules) {
        return (RetargetingCondition) new RetargetingCondition()
                .withId(retCondId)
                .withName(name)
                .withType(interests)
                .withClientId(clientId.asLong())
                .withRules(mapList(conditionRules, RetargetingConverter::toCoreRetargetingConditionRule));
    }

    private static RetargetingCondition toCaCoreRetargetingCondition(ClientId clientId, List<Goal> goals) {
        return (RetargetingCondition) new RetargetingCondition()
                .withType(interests)
                .withClientId(clientId.asLong())
                .withRules(List.of(RetargetingConverter.toCaCoreRetargetingConditionRule(goals)));
    }

    private static PageBlock toCorePageBlock(GdAdGroupPageBlock pageBlockItem) {
        return new PageBlock()
                .withImpId(pageBlockItem.getImpId())
                .withPageId(pageBlockItem.getPageId());
    }

    private static CriterionType calcCpmCriterionType(GdUpdateCpmAdGroupItem cpmAdGroup) {
        if (nonNull(cpmAdGroup.getCriterionType())) {
            return GdCriterionType.toSource(cpmAdGroup.getCriterionType());
        }

        if (!isEmpty(cpmAdGroup.getKeywords())) {
            return KEYWORD;
        } else if (nonNull(cpmAdGroup.getRetargetingCondition())) {
            return USER_PROFILE;
        } else if (hasContentCategories(cpmAdGroup)) {
            return CONTENT_CATEGORY;
        }

        return USER_PROFILE;
    }

    public static List<ComplexMcBannerAdGroup> toComplexMcBannerAdGroups(List<GdAddMcBannerAdGroupItem> adGroupAddItems,
                                                                         boolean isSearchRetargetingEnabled) {
        return mapList(adGroupAddItems, item -> new ComplexMcBannerAdGroup()
                .withAdGroup(toMcBannerAdGroup(item))
                .withKeywords(toCoreKeywords(item.getKeywords(), null))
                .withComplexBidModifier(isSearchRetargetingEnabled
                        ? toComplexBidModifier(item.getSearchRetargetings(), item.getBidModifiers(),
                        item.getCampaignId(), null)
                        : toComplexBidModifier(item.getBidModifiers()))
        );
    }

    public static List<ComplexMcBannerAdGroup> toComplexMcBannerAdGroups(List<GdUpdateMcBannerAdGroupItem>
                                                                                 adGroupUpdateItems,
                                                                         Map<Long, List<Keyword>> keywordsByAdGroupIds,
                                                                         Map<Long, AdGroupType> adGroupTypes,
                                                                         boolean isSearchRetargetingEnabled) {
        return mapList(adGroupUpdateItems, adGroupUpdateItem -> toComplexMcBannerAdGroup(adGroupUpdateItem,
                keywordsByAdGroupIds.get(adGroupUpdateItem.getAdGroupId()),
                adGroupTypes.get(adGroupUpdateItem.getAdGroupId()),
                isSearchRetargetingEnabled));
    }

    private static ComplexMcBannerAdGroup toComplexMcBannerAdGroup(GdUpdateMcBannerAdGroupItem adGroupUpdateItem,
                                                                   @Nullable List<Keyword> adGroupKeywords,
                                                                   AdGroupType adGroupType,
                                                                   boolean isSearchRetargetingEnabled) {
        return new ComplexMcBannerAdGroup()
                .withKeywords(toCoreKeywords(adGroupUpdateItem.getKeywords(), adGroupKeywords))
                .withAdGroup(toMcBannerAdGroup(adGroupUpdateItem, adGroupType))
                .withComplexBidModifier(isSearchRetargetingEnabled
                        ? toComplexBidModifier(
                        adGroupUpdateItem.getSearchRetargetings(), adGroupUpdateItem.getBidModifiers(),
                        null, adGroupUpdateItem.getAdGroupId())
                        : toComplexBidModifier(adGroupUpdateItem.getBidModifiers())
                );
    }

    private static AdGroup toMcBannerAdGroup(GdAddMcBannerAdGroupItem item) {
        return new McBannerAdGroup()
                .withCampaignId(item.getCampaignId())
                .withType(AdGroupType.MCBANNER)
                .withName(item.getName())
                .withMinusKeywords(item.getAdGroupMinusKeywords())
                .withLibraryMinusKeywordsIds(item.getLibraryMinusKeywordsIds())
                .withGeo(mapList(item.getRegionIds(), Integer::longValue))
                .withPageGroupTags(item.getPageGroupTags())
                .withTargetTags(item.getTargetTags());
    }

    public static AdGroup toMcBannerAdGroup(GdUpdateMcBannerAdGroupItem adGroupUpdateItem, AdGroupType adGroupType) {
        return new AdGroup()
                .withId(adGroupUpdateItem.getAdGroupId())
                .withName(adGroupUpdateItem.getAdGroupName())
                .withMinusKeywords(adGroupUpdateItem.getAdGroupMinusKeywords())
                .withLibraryMinusKeywordsIds(adGroupUpdateItem.getLibraryMinusKeywordsIds())
                .withGeo(mapList(adGroupUpdateItem.getRegionIds(), Integer::longValue))
                .withType(adGroupType)
                .withPageGroupTags(adGroupUpdateItem.getPageGroupTags())
                .withTargetTags(adGroupUpdateItem.getTargetTags());
    }

    public static ComplexContentPromotionAdGroup toComplexContentPromotionAdGroup(GdAddContentPromotionAdGroupItem adGroupAddItem,
                                                                                  boolean isSearchRetargetingEnabled) {
        return new ComplexContentPromotionAdGroup()
                .withKeywords(toCoreKeywords(adGroupAddItem.getKeywords(), null))
                .withAdGroup(toContentPromotionAdGroup(adGroupAddItem))
                .withComplexBidModifier(
                        isSearchRetargetingEnabled
                                ? toComplexBidModifier(adGroupAddItem.getSearchRetargetings(),
                                adGroupAddItem.getBidModifiers(),
                                adGroupAddItem.getCampaignId(), null)
                                : toComplexBidModifier(adGroupAddItem.getBidModifiers())
                )
                .withRelevanceMatches(toCoreRelevanceMatches(adGroupAddItem.getRelevanceMatch()));
    }

    public static ComplexContentPromotionAdGroup toComplexContentPromotionAdGroup(GdUpdateContentPromotionAdGroupItem adGroupUpdateItem,
                                                                                  @Nullable List<Keyword> adGroupKeywords,
                                                                                  AdGroupType adGroupType,
                                                                                  boolean isSearchRetargetingEnabled) {
        return new ComplexContentPromotionAdGroup()
                .withKeywords(toCoreKeywords(adGroupUpdateItem.getKeywords(), adGroupKeywords))
                .withAdGroup(toContentPromotionAdGroup(adGroupUpdateItem, adGroupType))
                .withComplexBidModifier(isSearchRetargetingEnabled
                        ? toComplexBidModifier(adGroupUpdateItem.getSearchRetargetings(),
                        adGroupUpdateItem.getBidModifiers(),
                        null, adGroupUpdateItem.getAdGroupId())
                        : toComplexBidModifier(adGroupUpdateItem.getBidModifiers())
                )
                .withRelevanceMatches(toCoreRelevanceMatches(adGroupUpdateItem.getRelevanceMatch()));
    }

    private static AdGroup toContentPromotionAdGroup(GdAddContentPromotionAdGroupItem item) {
        Long campaignId = item.getCampaignId();
        return new ContentPromotionAdGroup()
                .withType(AdGroupType.CONTENT_PROMOTION)
                .withContentPromotionType(CONTENT_PROMOTION_ADGROUP_CORE_TYPE_BY_GD_TYPE.get(item.getContentPromotionGroupType()))
                .withName(item.getName())
                .withCampaignId(campaignId)
                .withGeo(integerToLongList(item.getRegionIds()))
                .withLibraryMinusKeywordsIds(item.getLibraryMinusKeywordsIds())
                .withMinusKeywords(item.getAdGroupMinusKeywords())
                .withPageGroupTags(item.getPageGroupTags())
                .withTargetTags(item.getTargetTags());
    }

    private static AdGroup toContentPromotionAdGroup(GdUpdateContentPromotionAdGroupItem adGroupUpdateItem,
                                                     AdGroupType adGroupType) {
        return new ContentPromotionAdGroup()
                .withId(adGroupUpdateItem.getAdGroupId())
                .withName(adGroupUpdateItem.getAdGroupName())
                .withMinusKeywords(adGroupUpdateItem.getAdGroupMinusKeywords())
                .withLibraryMinusKeywordsIds(adGroupUpdateItem.getLibraryMinusKeywordsIds())
                .withGeo(mapList(adGroupUpdateItem.getRegionIds(), Integer::longValue))
                .withType(adGroupType)
                .withPageGroupTags(adGroupUpdateItem.getPageGroupTags())
                .withTargetTags(adGroupUpdateItem.getTargetTags());
    }


    /**
     * Преобразует запрос на редактирование группы в контейнер с параметрами для комплексной операции.
     *
     * @param adGroupUpdateItems   исходный запрос на редактирование группы
     * @param keywordsByAdGroupIds ключевые слова группы, сохраненные до текущего момента
     * @param adGroupTypes         тип группы
     * @return {@link List< ComplexMobileContentAdGroup >}  контейнер для выполнения коплексной операции
     */
    public static List<ComplexMobileContentAdGroup> toComplexMobileContentAdGroups(
            List<GdUpdateMobileContentAdGroupItem> adGroupUpdateItems,
            Map<Long, List<Keyword>> keywordsByAdGroupIds,
            Map<Long, AdGroupType> adGroupTypes,
            Map<Long, String> adGroupIdToStoreUrl,
            boolean isSearchRetargetingEnabled) {
        return mapList(adGroupUpdateItems, adGroupUpdateItem -> toComplexMobileContentAdGroup(adGroupUpdateItem,
                keywordsByAdGroupIds.get(adGroupUpdateItem.getAdGroupId()),
                adGroupTypes.get(adGroupUpdateItem.getAdGroupId()),
                adGroupIdToStoreUrl.get(adGroupUpdateItem.getAdGroupId()),
                isSearchRetargetingEnabled));
    }

    private static ComplexMobileContentAdGroup toComplexMobileContentAdGroup(
            GdUpdateMobileContentAdGroupItem adGroupUpdateItem,
            @Nullable List<Keyword> adGroupKeywords,
            AdGroupType adGroupType,
            String storeUrl,
            boolean isSearchRetargetingEnabled) {
        return new ComplexMobileContentAdGroup()
                .withKeywords(toCoreKeywords(adGroupUpdateItem.getKeywords(), adGroupKeywords))
                .withRelevanceMatches(toCoreRelevanceMatches(adGroupUpdateItem.getRelevanceMatch()))
                .withAdGroup(toMobileContentAdGroup(adGroupUpdateItem, adGroupType, storeUrl))
                .withTargetInterests(toTargetInterests(null, adGroupUpdateItem.getAdGroupId(),
                        adGroupUpdateItem.getRetargetings(), adGroupUpdateItem.getInterests()))
                .withComplexBidModifier(isSearchRetargetingEnabled
                        ? toComplexBidModifier(adGroupUpdateItem.getSearchRetargetings(),
                        adGroupUpdateItem.getBidModifiers(),
                        null, adGroupUpdateItem.getAdGroupId())
                        : toComplexBidModifier(adGroupUpdateItem.getBidModifiers())
                );
    }

    public static AdGroup toMobileContentAdGroup(GdUpdateMobileContentAdGroupItem adGroupUpdateItem,
                                                 AdGroupType adGroupType,
                                                 String storeUrl) {
        return new MobileContentAdGroup()
                .withId(adGroupUpdateItem.getAdGroupId())
                .withName(adGroupUpdateItem.getAdGroupName())
                .withMinusKeywords(adGroupUpdateItem.getAdGroupMinusKeywords())
                .withLibraryMinusKeywordsIds(adGroupUpdateItem.getLibraryMinusKeywordsIds())
                .withGeo(mapList(adGroupUpdateItem.getRegionIds(), Integer::longValue))
                .withType(adGroupType)
                .withDeviceTypeTargeting(mapSet(adGroupUpdateItem.getDeviceTypeTargeting(), deviceType
                        -> MobileContentAdGroupDeviceTypeTargeting.fromTypedValue(deviceType.getTypedValue())))
                .withNetworkTargeting(mapSet(adGroupUpdateItem.getNetworkTargeting(), networkTargeting
                        -> MobileContentAdGroupNetworkTargeting.fromTypedValue(networkTargeting.getTypedValue())))
                .withMinimalOperatingSystemVersion(adGroupUpdateItem.getCurrentMinimalOsVersion())
                .withStoreUrl(storeUrl)
                .withPageGroupTags(adGroupUpdateItem.getPageGroupTags())
                .withTargetTags(adGroupUpdateItem.getTargetTags());

    }

    /**
     * Преобразует запрос на добавление группы в контейнер с параметрами для комплексной операции.
     *
     * @param groupAddItem         исходный запрос на добавление группы
     * @param campaignIdToStoreUrl ссылка на приложение в магазине по кампании
     */
    public static ComplexMobileContentAdGroup toComplexMobileContentAdGroup(GdAddMobileContentAdGroupItem groupAddItem,
                                                                            Map<Long, String> campaignIdToStoreUrl,
                                                                            boolean isSearchRetargetingsEnabled) {
        String storeUrl = campaignIdToStoreUrl.get(groupAddItem.getCampaignId());
        return new ComplexMobileContentAdGroup()
                .withKeywords(toCoreKeywords(groupAddItem.getKeywords(), null))
                .withRelevanceMatches(toCoreRelevanceMatches(groupAddItem.getRelevanceMatch()))
                .withAdGroup(toMobileContentAdGroup(groupAddItem, storeUrl))
                .withTargetInterests(toTargetInterests(groupAddItem.getCampaignId(), null,
                        groupAddItem.getRetargetings(), groupAddItem.getInterests()))
                .withComplexBidModifier(isSearchRetargetingsEnabled
                        ? toComplexBidModifier(groupAddItem.getSearchRetargetings(), groupAddItem.getBidModifiers(),
                        groupAddItem.getCampaignId(), null)
                        : toComplexBidModifier(groupAddItem.getBidModifiers())
                );
    }

    private static AdGroup toMobileContentAdGroup(GdAddMobileContentAdGroupItem groupAddItem, String storeUrl) {
        return new MobileContentAdGroup()
                .withName(groupAddItem.getName())
                .withCampaignId(groupAddItem.getCampaignId())
                .withMinusKeywords(groupAddItem.getAdGroupMinusKeywords())
                .withLibraryMinusKeywordsIds(groupAddItem.getLibraryMinusKeywordsIds())
                .withGeo(mapList(groupAddItem.getRegionIds(), Integer::longValue))
                .withType(AdGroupType.MOBILE_CONTENT)
                .withDeviceTypeTargeting(mapSet(groupAddItem.getDeviceTypeTargeting(), deviceType
                        -> MobileContentAdGroupDeviceTypeTargeting.fromTypedValue(deviceType.getTypedValue())))
                .withNetworkTargeting(mapSet(groupAddItem.getNetworkTargeting(), networkTargeting
                        -> MobileContentAdGroupNetworkTargeting.fromTypedValue(networkTargeting.getTypedValue())))
                .withMinimalOperatingSystemVersion(groupAddItem.getCurrentMinimalOsVersion())
                .withStoreUrl(storeUrl)
                .withPageGroupTags(groupAddItem.getPageGroupTags())
                .withTargetTags(groupAddItem.getTargetTags());
    }

    public static ComplexDynamicAdGroup toComplexDynamicAdGroup(GdUpdateDynamicAdGroupItem adGroupUpdateItem,
                                                                boolean isSearchRetargetingEnabled) {
        return new ComplexDynamicAdGroup()
                .withAdGroup(toDynamicAdGroup(adGroupUpdateItem))
                .withComplexBidModifier(isSearchRetargetingEnabled
                        ? toComplexBidModifier(
                        adGroupUpdateItem.getSearchRetargetings(), adGroupUpdateItem.getBidModifiers(),
                        null, adGroupUpdateItem.getId())
                        : toComplexBidModifier(adGroupUpdateItem.getBidModifiers())
                );
    }

    private static AdGroup toDynamicAdGroup(GdUpdateDynamicAdGroupItem adGroupUpdateItem) {
        return new DynamicAdGroup()
                .withType(AdGroupType.DYNAMIC)
                .withId(adGroupUpdateItem.getId())
                .withName(adGroupUpdateItem.getName())
                .withFieldToUseAsName(adGroupUpdateItem.getFieldToUseAsName())
                .withFieldToUseAsBody(adGroupUpdateItem.getFieldToUseAsBody())
                .withTrackingParams(adGroupUpdateItem.getTrackingParams())
                .withMinusKeywords(adGroupUpdateItem.getMinusKeywords())
                .withLibraryMinusKeywordsIds(adGroupUpdateItem.getLibraryMinusKeywordsIds())
                .withGeo(mapList(adGroupUpdateItem.getRegionIds(), Integer::longValue))
                .withPageGroupTags(adGroupUpdateItem.getPageGroupTags())
                .withTargetTags(adGroupUpdateItem.getTargetTags())
                .withHyperGeoId(adGroupUpdateItem.getHyperGeoId())
                .withRelevanceMatchCategories(toCoreRelevanceMatchCategories(
                        ifNotNull(adGroupUpdateItem.getRelevanceMatch(),
                                GdUpdateAdGroupRelevanceMatchItem::getRelevanceMatchCategories)));
    }

}
