package ru.yandex.direct.grid.processing.service.group.internalad

import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.adgroup.model.InternalAdGroup
import ru.yandex.direct.core.entity.adgroup.service.AdGroupService
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.AdGroupAdditionalTargeting
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.AdGroupAdditionalTargetingJoinType
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.AdGroupAdditionalTargetingMode
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.service.AdGroupAdditionalTargetingService
import ru.yandex.direct.dbutil.model.ClientId
import kotlin.reflect.KClass

@Component
open class InternalAdGroupsLoader(
    private val adGroupService: AdGroupService,
    private val adGroupAdditionalTargetingService: AdGroupAdditionalTargetingService,
) {
    fun load(clientId: ClientId, internalAdGroupIds: Collection<Long>): List<InternalAdGroupWithTargeting> {
        val adGroups = adGroupService.getAdGroups(clientId, internalAdGroupIds).filterIsInstance<InternalAdGroup>()
        val targetingByAdGroupId = adGroupAdditionalTargetingService.getTargetingMapByAdGroupIds(
            clientId, internalAdGroupIds)
        return adGroups.map { adGroup ->
            val targetings = targetingByAdGroupId[adGroup.id] ?: listOf()
            InternalAdGroupWithTargeting(adGroup, TargetingIndex.fromTargetingCollection(targetings))
        }
    }
}

class TargetingIndex {
    private val storage: MutableMap<Key<*>, AdGroupAdditionalTargeting> = mutableMapOf()

    fun store(targeting: AdGroupAdditionalTargeting) {
        storage[Key(targeting.javaClass.kotlin, targeting.targetingMode, targeting.joinType)] = targeting
    }

    fun erase(key: Key<*>) = storage.remove(key)

    operator fun <TARG : AdGroupAdditionalTargeting> get(key: Key<TARG>): TARG? {
        @Suppress("UNCHECKED_CAST")
        return storage[key] as TARG?
    }

    operator fun <T : AdGroupAdditionalTargeting> get(
        targetingClass: KClass<T>,
        targetingMode: AdGroupAdditionalTargetingMode,
        joinType: AdGroupAdditionalTargetingJoinType,
    ): T? {
        return get(Key(targetingClass, targetingMode, joinType))
    }

    operator fun <T : AdGroupAdditionalTargeting> get(
        targetingClass: KClass<T>,
        targetingMode: AdGroupAdditionalTargetingMode
    ): List<T> {
        return enumValues<AdGroupAdditionalTargetingJoinType>().mapNotNull { get(targetingClass, targetingMode, it) }
    }

    operator fun <T : AdGroupAdditionalTargeting> get(targetingClass: KClass<T>): List<T> {
        return enumValues<AdGroupAdditionalTargetingMode>().flatMap { get(targetingClass, it) }
    }

    fun toList() = storage.values.toList()

    data class Key<TARG : AdGroupAdditionalTargeting>(
        val clazz: KClass<out TARG>,
        val targetingMode: AdGroupAdditionalTargetingMode,
        val joinType: AdGroupAdditionalTargetingJoinType,
    )

    companion object {
        fun fromTargetingCollection(targetings: Collection<AdGroupAdditionalTargeting>): TargetingIndex {
            val storage = TargetingIndex()
            targetings.forEach { storage.store(it) }
            return storage
        }
    }
}

class InternalAdGroupWithTargeting(
    val adGroup: InternalAdGroup,
    val targetings: TargetingIndex,
)
