package ru.yandex.direct.grid.processing.service.group.loader;

import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import org.dataloader.MappedBatchLoaderWithContext;
import org.springframework.context.annotation.Scope;
import org.springframework.context.annotation.ScopedProxyMode;
import org.springframework.stereotype.Component;
import org.springframework.web.context.WebApplicationContext;

import ru.yandex.direct.core.entity.addition.callout.repository.CalloutRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.client.GdClientInfo;
import ru.yandex.direct.grid.processing.service.dataloader.GridBatchingDataLoader;
import ru.yandex.direct.grid.processing.service.dataloader.GridContextProvider;

import static com.google.common.base.Preconditions.checkNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;

@Component
// DataLoader'ы хранят состояние, поэтому жить должны в рамках запроса
@Scope(value = WebApplicationContext.SCOPE_REQUEST, proxyMode = ScopedProxyMode.TARGET_CLASS)
@ParametersAreNonnullByDefault
public class AdGroupsHasCalloutsDataLoader extends GridBatchingDataLoader<Long, Boolean> {

    public AdGroupsHasCalloutsDataLoader(GridContextProvider gridContextProvider, CalloutRepository calloutRepository) {
        this.dataLoader = mappedDataLoader(gridContextProvider, getBatchLoadFunction(calloutRepository));
    }

    private MappedBatchLoaderWithContext<Long, Boolean> getBatchLoadFunction(CalloutRepository calloutRepository) {
        return (adGroupIds, environment) -> {
            GridGraphQLContext context = environment.getContext();
            GdClientInfo queriedClient = context.getQueriedClient();
            checkNotNull(queriedClient, "queriedClient should be set in gridContext");

            ClientId clientId = ClientId.fromLong(queriedClient.getId());
            Set<Long> adGroupIdsWithExistingCallouts = calloutRepository
                    .getAdGroupIdsWithExistingCallouts(queriedClient.getShard(), clientId, adGroupIds);

            Map<Long, Boolean> result =
                    listToMap(adGroupIds, Function.identity(), adGroupIdsWithExistingCallouts::contains);
            return CompletableFuture.completedFuture(result);
        };
    }

}
