package ru.yandex.direct.grid.processing.service.group.loader;

import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import org.dataloader.DataLoader;
import org.dataloader.MappedBatchLoaderWithContext;
import org.springframework.context.annotation.Scope;
import org.springframework.context.annotation.ScopedProxyMode;
import org.springframework.stereotype.Component;
import org.springframework.web.context.WebApplicationContext;

import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.service.BannerService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.client.GdClientInfo;
import ru.yandex.direct.grid.processing.service.dataloader.GridBatchingDataLoader;
import ru.yandex.direct.grid.processing.service.dataloader.GridContextProvider;

import static com.google.common.base.Preconditions.checkNotNull;
import static ru.yandex.direct.grid.processing.service.group.AdGroupActionConditionsUtil.isCpmPriceDefaultAdGroup;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;

/**
 * {@link DataLoader} для отложенной batch'евой загрузки информации о возможности удаления групп объявлений
 * В интерфейсе группу можно удалять, если все объявления группы доступны для удаления или если группа пустая.
 * Для оптимизации проверяем только возможность удаления главного объявления группы.
 * Предполагаем, что если можно удалить главное объявление, то и остальные объявления тоже можно удалять. В старом
 * интерфейсе так
 *
 * @see GridBatchingDataLoader
 * @see ru.yandex.direct.grid.processing.service.dataloader.GridDataLoaderRegistry
 */
@Component
// DataLoader'ы хранят состояние, поэтому жить должны в рамках запроса
@Scope(value = WebApplicationContext.SCOPE_REQUEST, proxyMode = ScopedProxyMode.TARGET_CLASS)
@ParametersAreNonnullByDefault
public class CanBeDeletedAdGroupsDataLoader extends GridBatchingDataLoader<Long, Boolean> {

    public CanBeDeletedAdGroupsDataLoader(GridContextProvider gridContextProvider, BannerService bannerService) {
        this.dataLoader = mappedDataLoader(gridContextProvider, getBatchLoadFunction(bannerService));
    }

    private MappedBatchLoaderWithContext<Long, Boolean> getBatchLoadFunction(BannerService bannerService) {
        return (adGroupIds, environment) -> {
            GridGraphQLContext context = environment.getContext();
            GdClientInfo queriedClient = context.getQueriedClient();
            checkNotNull(queriedClient, "queriedClient should be set in gridContext");
            ClientId clientId = ClientId.fromLong(queriedClient.getId());

            Map<Long, BannerWithSystemFields> mainAdsByAdGroupId = context.getMainAdsByAdGroupId();
            if (mainAdsByAdGroupId == null) {
                /*
                Если не нашли в контексте, то выгружаем из базы.
                Такое может быть если rowset резолвера adGroups выгрузили из кэша, т.к. мы не сохраняем mainAd в кэш
                 */
                mainAdsByAdGroupId = bannerService.getMainBannerByAdGroupIds(clientId, adGroupIds);
            }
            List<BannerWithSystemFields> mainAdsExceptCpmPriceDefault =
                    filterList(mainAdsByAdGroupId.values(), banner -> adGroupIds.contains(banner.getAdGroupId())
                            && !isCpmPriceDefaultAdGroup(context, banner.getAdGroupId()));
            Map<Long, Boolean> canBeDeletedBanners = bannerService
                    .getCanBeDeletedBanners(queriedClient.getShard(), clientId, mainAdsExceptCpmPriceDefault);

            // заполняем можно ли удалять не cpmPriceDefault группы, у которых есть баннеры
            Map<Long, Boolean> result = listToMap(mainAdsExceptCpmPriceDefault, BannerWithSystemFields::getAdGroupId,
                    banner -> canBeDeletedBanners.get(banner.getId()));
            // дефолтные cpm_price группы удалять нельзя, даже если в них нет объявлений
            adGroupIds.forEach(adGroupId -> {
                if (isCpmPriceDefaultAdGroup(context, adGroupId)) {
                    result.put(adGroupId, false);
                }
            });
            // не cpmPriceDefault группы без объявлений помечаем как удаляемые
            adGroupIds.forEach(adGroupId -> result.putIfAbsent(adGroupId, true));
            return CompletableFuture.completedFuture(result);
        };
    }

}
