package ru.yandex.direct.grid.processing.service.group.loader;

import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.dataloader.MappedBatchLoaderWithContext;
import org.springframework.context.annotation.Scope;
import org.springframework.context.annotation.ScopedProxyMode;
import org.springframework.stereotype.Component;
import org.springframework.web.context.WebApplicationContext;

import ru.yandex.direct.core.entity.keyword.service.KeywordService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.client.GdClientInfo;
import ru.yandex.direct.grid.processing.service.dataloader.GridBatchingDataLoader;
import ru.yandex.direct.grid.processing.service.dataloader.GridContextProvider;

import static com.google.common.base.Preconditions.checkNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;

@Component
// DataLoader'ы хранят состояние, поэтому жить должны в рамках запроса
@Scope(value = WebApplicationContext.SCOPE_REQUEST, proxyMode = ScopedProxyMode.TARGET_CLASS)
@ParametersAreNonnullByDefault
public class KeywordsCountDataLoader extends GridBatchingDataLoader<Long, Integer> {

    public KeywordsCountDataLoader(GridContextProvider gridContextProvider, KeywordService keywordService) {
        this.dataLoader = mappedDataLoader(gridContextProvider, getBatchLoadFunction(keywordService));
    }

    private MappedBatchLoaderWithContext<Long, Integer> getBatchLoadFunction(KeywordService keywordService) {
        return (adGroupIds, environment) -> {
            GridGraphQLContext context = environment.getContext();
            GdClientInfo queriedClient = context.getQueriedClient();
            checkNotNull(queriedClient, "queriedClient should be set in gridContext");

            List<Long> campaignIds = environment.getKeyContexts().values()
                    .stream().map(a -> (Long) a).collect(Collectors.toList());
            var keywordQuantitiesByAdGroupIds = keywordService
                    .getKeywordQuantitiesByAdGroupIds(ClientId.fromLong(queriedClient.getId()), campaignIds,
                            adGroupIds);

            var result = listToMap(adGroupIds, Function.identity(),
                    adGroupId -> keywordQuantitiesByAdGroupIds.getOrDefault(adGroupId, 0));
            return CompletableFuture.completedFuture(result);
        };
    }
}
