package ru.yandex.direct.grid.processing.service.group.mutation;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Preconditions;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.container.InternalAdGroupAddItem;
import ru.yandex.direct.core.entity.adgroup.container.InternalAdGroupOperationContainer;
import ru.yandex.direct.core.entity.adgroup.container.InternalAdGroupOperationContainer.RequestSource;
import ru.yandex.direct.core.entity.adgroup.service.complex.internal.ComplexInternalAdGroupService;
import ru.yandex.direct.core.entity.internalads.service.InternalAdsProductService;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.grid.processing.model.api.GdValidationResult;
import ru.yandex.direct.grid.processing.model.group.mutation.GdAddAdGroupPayload;
import ru.yandex.direct.grid.processing.model.group.mutation.GdAddAdGroupPayloadItem;
import ru.yandex.direct.grid.processing.model.group.mutation.GdAddInternalAdGroups;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.result.MassResult;

import static ru.yandex.direct.grid.processing.util.ResponseConverter.getResults;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class AddInternalAdGroupsMutationService {
    private final InternalAdsProductService internalAdsProductService;
    private final ComplexInternalAdGroupService complexInternalAdGroupService;
    private final InternalAdGroupMutationValidationService mutationValidationService;

    public AddInternalAdGroupsMutationService(InternalAdsProductService internalAdsProductService,
                                              ComplexInternalAdGroupService complexInternalAdGroupService,
                                              InternalAdGroupMutationValidationService mutationValidationService) {
        this.internalAdsProductService = internalAdsProductService;
        this.complexInternalAdGroupService = complexInternalAdGroupService;
        this.mutationValidationService = mutationValidationService;
    }

    /**
     * Создание групп объявлений для внутренней рекламы.
     *
     * @param operatorUid идентификатор пользователя оператора
     * @param owner       идентификаторы клиента, которому принадлежат создаваемые группы
     * @param input       запрос, содержащий список данных добавляемых групп
     */
    public GdAddAdGroupPayload addInternalAdGroups(Long operatorUid, UidAndClientId owner,
                                                   GdAddInternalAdGroups input) {
        Preconditions.checkState(internalAdsProductService.clientCanHaveInternalAdCampaigns(owner.getClientId()),
                "Internal ads are not supported for client");

        mutationValidationService.validateAddGroup(input);

        List<InternalAdGroupAddItem> coreAddItems = mapList(
                input.getAddItems(), item -> GdInternalGroupConverters.toCoreAddItem(owner.getClientId(), item));

        var operationContainer = new InternalAdGroupOperationContainer(
                Applicability.PARTIAL, operatorUid, owner, true, RequestSource.FRONT
        );
        MassResult<Long> result = complexInternalAdGroupService.add(coreAddItems, operationContainer);

        GdValidationResult validationResult =
                mutationValidationService.getAddValidationResult(
                        result.getValidationResult());

        List<GdAddAdGroupPayloadItem> payloadItems =
                getResults(result, id -> new GdAddAdGroupPayloadItem().withAdGroupId(id));

        return new GdAddAdGroupPayload()
                .withAddedAdGroupItems(payloadItems)
                .withValidationResult(validationResult);
    }
}
