package ru.yandex.direct.grid.processing.service.group.mutation;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.HashMultiset;
import com.google.common.collect.ImmutableMap;
import one.util.streamex.StreamEx;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.container.InternalAdGroupAddItem;
import ru.yandex.direct.core.entity.adgroup.container.InternalAdGroupUpdateItem;
import ru.yandex.direct.core.entity.adgroup.container.UntypedAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.InternalAdGroup;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.AdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.CallerReferrersAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.ClidTypesAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.ClidsAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.DesktopInstalledAppsAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.DeviceIdsAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.FeaturesInPPAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.HasLCookieAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.HasPassportIdAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.InterfaceLangsAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.InternalNetworkAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.IsDefaultYandexSearchAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.IsPPLoggedInAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.IsVirusedAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.IsYandexPlusAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.MobileInstalledAppsAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.PlusUserSegmentsAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.QueryOptionsAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.QueryReferersAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.SearchTextAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.ShowDatesAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.SidsAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.TestIdsAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.UserAgentsAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.UuidsAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.YandexUidsAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.YpCookiesAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.YsCookiesAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.uatraits.model.BrowserEnginesAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.uatraits.model.BrowserNamesAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.uatraits.model.DeviceNamesAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.uatraits.model.DeviceVendorsAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.uatraits.model.IsMobileAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.uatraits.model.IsTabletAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.uatraits.model.IsTouchAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.uatraits.model.OsFamiliesAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.uatraits.model.OsNamesAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingConditionBase;
import ru.yandex.direct.grid.model.campaign.timetarget.GdTimeTarget;
import ru.yandex.direct.grid.processing.model.api.GdValidationResult;
import ru.yandex.direct.grid.processing.model.group.additionaltargeting.GdAdditionalTargetingJoinType;
import ru.yandex.direct.grid.processing.model.group.additionaltargeting.GdAdditionalTargetingMode;
import ru.yandex.direct.grid.processing.model.group.additionaltargeting.mutation.GdAdditionalTargetingBrowserEnginesRequest;
import ru.yandex.direct.grid.processing.model.group.additionaltargeting.mutation.GdAdditionalTargetingBrowserNamesRequest;
import ru.yandex.direct.grid.processing.model.group.additionaltargeting.mutation.GdAdditionalTargetingCallerReferrersRequest;
import ru.yandex.direct.grid.processing.model.group.additionaltargeting.mutation.GdAdditionalTargetingClidTypesRequest;
import ru.yandex.direct.grid.processing.model.group.additionaltargeting.mutation.GdAdditionalTargetingClidsRequest;
import ru.yandex.direct.grid.processing.model.group.additionaltargeting.mutation.GdAdditionalTargetingDesktopInstalledAppsRequest;
import ru.yandex.direct.grid.processing.model.group.additionaltargeting.mutation.GdAdditionalTargetingDeviceIdsRequest;
import ru.yandex.direct.grid.processing.model.group.additionaltargeting.mutation.GdAdditionalTargetingDeviceNamesRequest;
import ru.yandex.direct.grid.processing.model.group.additionaltargeting.mutation.GdAdditionalTargetingDeviceVendorsRequest;
import ru.yandex.direct.grid.processing.model.group.additionaltargeting.mutation.GdAdditionalTargetingFeaturesInPPRequest;
import ru.yandex.direct.grid.processing.model.group.additionaltargeting.mutation.GdAdditionalTargetingInterfaceLangsRequest;
import ru.yandex.direct.grid.processing.model.group.additionaltargeting.mutation.GdAdditionalTargetingMobileInstalledAppsRequest;
import ru.yandex.direct.grid.processing.model.group.additionaltargeting.mutation.GdAdditionalTargetingOsFamiliesRequest;
import ru.yandex.direct.grid.processing.model.group.additionaltargeting.mutation.GdAdditionalTargetingOsNamesRequest;
import ru.yandex.direct.grid.processing.model.group.additionaltargeting.mutation.GdAdditionalTargetingPlusUserSegmentsRequest;
import ru.yandex.direct.grid.processing.model.group.additionaltargeting.mutation.GdAdditionalTargetingQueryOptionsRequest;
import ru.yandex.direct.grid.processing.model.group.additionaltargeting.mutation.GdAdditionalTargetingQueryReferersRequest;
import ru.yandex.direct.grid.processing.model.group.additionaltargeting.mutation.GdAdditionalTargetingRequest;
import ru.yandex.direct.grid.processing.model.group.additionaltargeting.mutation.GdAdditionalTargetingSearchTextRequest;
import ru.yandex.direct.grid.processing.model.group.additionaltargeting.mutation.GdAdditionalTargetingShowDatesRequest;
import ru.yandex.direct.grid.processing.model.group.additionaltargeting.mutation.GdAdditionalTargetingSidsRequest;
import ru.yandex.direct.grid.processing.model.group.additionaltargeting.mutation.GdAdditionalTargetingTestIdsRequest;
import ru.yandex.direct.grid.processing.model.group.additionaltargeting.mutation.GdAdditionalTargetingTimeRequest;
import ru.yandex.direct.grid.processing.model.group.additionaltargeting.mutation.GdAdditionalTargetingUnion;
import ru.yandex.direct.grid.processing.model.group.additionaltargeting.mutation.GdAdditionalTargetingUserAgentsRequest;
import ru.yandex.direct.grid.processing.model.group.additionaltargeting.mutation.GdAdditionalTargetingUuidsRequest;
import ru.yandex.direct.grid.processing.model.group.additionaltargeting.mutation.GdAdditionalTargetingYandexUidsRequest;
import ru.yandex.direct.grid.processing.model.group.additionaltargeting.mutation.GdAdditionalTargetingYpCookiesRequest;
import ru.yandex.direct.grid.processing.model.group.additionaltargeting.mutation.GdAdditionalTargetingYsCookiesRequest;
import ru.yandex.direct.grid.processing.model.group.mutation.GdAddInternalAdGroups;
import ru.yandex.direct.grid.processing.model.group.mutation.GdAddInternalAdGroupsItem;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateInternalAdGroups;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateInternalAdGroupsItem;
import ru.yandex.direct.grid.processing.model.retargeting.mutation.GdUpdateInternalAdRetargetingConditionItem;
import ru.yandex.direct.grid.processing.service.validation.GridDefectIds;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;
import ru.yandex.direct.grid.processing.service.validation.presentation.SkipByDefaultMappingPathNodeConverter;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.DefaultPathNodeConverterProvider;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.PathNodeConverter;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static java.util.Arrays.asList;
import static org.joda.time.DateTimeConstants.DAYS_PER_WEEK;
import static org.joda.time.DateTimeConstants.HOURS_PER_DAY;
import static ru.yandex.direct.core.validation.ValidationUtils.hasValidationIssues;
import static ru.yandex.direct.grid.processing.service.campaign.CampaignValidationService.HOLIDAYS_SETTINGS_VALIDATOR;
import static ru.yandex.direct.grid.processing.service.validation.GridDefectDefinitions.invalidUnion;
import static ru.yandex.direct.grid.processing.service.validation.GridValidationResultConversionService.buildGridValidationResult;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.notEmptyCollection;
import static ru.yandex.direct.validation.constraint.CommonConstraints.isEqual;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.defect.CollectionDefects.duplicatedElement;
import static ru.yandex.direct.validation.defect.CommonDefects.invalidValue;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.PathHelper.path;

@Service
@ParametersAreNonnullByDefault
public class InternalAdGroupMutationValidationService {
    // Положительные и отрицательные таргетинги одно типа разрешены для всех не булевых таргетингов
    private static final Set<Class<? extends GdAdditionalTargetingRequest>> TARGETING_AND_FILTERING_ALLOWED = Set.of(
            GdAdditionalTargetingBrowserEnginesRequest.class,
            GdAdditionalTargetingBrowserNamesRequest.class,
            GdAdditionalTargetingClidTypesRequest.class,
            GdAdditionalTargetingClidsRequest.class,
            GdAdditionalTargetingDesktopInstalledAppsRequest.class,
            GdAdditionalTargetingDeviceNamesRequest.class,
            GdAdditionalTargetingDeviceVendorsRequest.class,
            GdAdditionalTargetingFeaturesInPPRequest.class,
            GdAdditionalTargetingInterfaceLangsRequest.class,
            GdAdditionalTargetingMobileInstalledAppsRequest.class,
            GdAdditionalTargetingOsFamiliesRequest.class,
            GdAdditionalTargetingOsNamesRequest.class,
            GdAdditionalTargetingQueryOptionsRequest.class,
            GdAdditionalTargetingQueryReferersRequest.class,
            GdAdditionalTargetingCallerReferrersRequest.class,
            GdAdditionalTargetingShowDatesRequest.class,
            GdAdditionalTargetingTestIdsRequest.class,
            GdAdditionalTargetingUserAgentsRequest.class,
            GdAdditionalTargetingYandexUidsRequest.class,
            GdAdditionalTargetingYsCookiesRequest.class,
            GdAdditionalTargetingYpCookiesRequest.class,
            GdAdditionalTargetingSidsRequest.class,
            GdAdditionalTargetingUuidsRequest.class,
            GdAdditionalTargetingDeviceIdsRequest.class,
            GdAdditionalTargetingPlusUserSegmentsRequest.class,
            GdAdditionalTargetingSearchTextRequest.class
    );

    private static final Map<Class<?>, PathNodeConverter> COMMON_INTERNAL_GROUP_CONVERTERS =
            ImmutableMap.<Class<?>, PathNodeConverter>builder()
                    .put(HasPassportIdAdGroupAdditionalTargeting.class, additionalTargetingConverter(
                            "targetingHasPassportId"))
                    .put(IsVirusedAdGroupAdditionalTargeting.class, additionalTargetingConverter(
                            "targetingIsVirused"))
                    .put(HasLCookieAdGroupAdditionalTargeting.class, additionalTargetingConverter(
                            "targetingHasLCookie"))
                    .put(InternalNetworkAdGroupAdditionalTargeting.class, additionalTargetingConverter(
                            "targetingInternalNetwork"))
                    .put(IsMobileAdGroupAdditionalTargeting.class, additionalTargetingConverter("targetingIsMobile"))
                    .put(IsTabletAdGroupAdditionalTargeting.class, additionalTargetingConverter("targetingIsTablet"))
                    .put(IsTouchAdGroupAdditionalTargeting.class, additionalTargetingConverter("targetingIsTouch"))
                    .put(YandexUidsAdGroupAdditionalTargeting.class, additionalTargetingWithValueConverter(
                            "targetingYandexUids"))
                    .put(QueryReferersAdGroupAdditionalTargeting.class, additionalTargetingWithValueConverter(
                            "targetingQueryReferers"))
                    .put(CallerReferrersAdGroupAdditionalTargeting.class, additionalTargetingWithValueConverter(
                            "targetingCallerReferrers"))
                    .put(InterfaceLangsAdGroupAdditionalTargeting.class, additionalTargetingWithValueConverter(
                            "targetingInterfaceLangs"))
                    .put(UserAgentsAdGroupAdditionalTargeting.class, additionalTargetingWithValueConverter(
                            "targetingUserAgents"))
                    .put(BrowserEnginesAdGroupAdditionalTargeting.class, additionalTargetingWithValueConverter(
                            "targetingBrowserEngines"))
                    .put(BrowserNamesAdGroupAdditionalTargeting.class, additionalTargetingWithValueConverter(
                            "targetingBrowserNames"))
                    .put(OsFamiliesAdGroupAdditionalTargeting.class, additionalTargetingWithValueConverter(
                            "targetingOsFamilies"))
                    .put(OsNamesAdGroupAdditionalTargeting.class, additionalTargetingWithValueConverter(
                            "targetingOsNames"))
                    .put(DeviceVendorsAdGroupAdditionalTargeting.class, additionalTargetingWithValueConverter(
                            "targetingDeviceVendors"))
                    .put(DeviceNamesAdGroupAdditionalTargeting.class, additionalTargetingWithValueConverter(
                            "targetingDeviceNames"))
                    .put(ShowDatesAdGroupAdditionalTargeting.class, additionalTargetingWithValueConverter(
                            "targetingShowDates"))
                    .put(DesktopInstalledAppsAdGroupAdditionalTargeting.class, additionalTargetingWithValueConverter(
                            "targetingDesktopInstalledApps"))
                    .put(ClidTypesAdGroupAdditionalTargeting.class, additionalTargetingWithValueConverter(
                            "targetingClidTypes"))
                    .put(ClidsAdGroupAdditionalTargeting.class, additionalTargetingWithValueConverter(
                            "targetingClids"))
                    .put(QueryOptionsAdGroupAdditionalTargeting.class, additionalTargetingWithValueConverter(
                            "targetingQueryOptions"))
                    .put(TestIdsAdGroupAdditionalTargeting.class, additionalTargetingWithValueConverter(
                            "targetingTestIds"))
                    .put(YsCookiesAdGroupAdditionalTargeting.class, additionalTargetingWithValueConverter(
                            "targetingYsCookies"))
                    .put(IsYandexPlusAdGroupAdditionalTargeting.class, additionalTargetingConverter(
                            "targetingIsYandexPlus"))
                    .put(IsPPLoggedInAdGroupAdditionalTargeting.class, additionalTargetingConverter(
                            "targetingIsPPLoggedIn"
                    ))
                    .put(MobileInstalledAppsAdGroupAdditionalTargeting.class, additionalTargetingWithValueConverter(
                            "targetingMobileInstalledApps"))
                    .put(FeaturesInPPAdGroupAdditionalTargeting.class, additionalTargetingWithValueConverter(
                            "targetingFeaturesInPP"))
                    .put(YpCookiesAdGroupAdditionalTargeting.class, additionalTargetingWithValueConverter(
                            "targetingYpCookies"))
                    .put(IsDefaultYandexSearchAdGroupAdditionalTargeting.class, additionalTargetingConverter(
                            "targetingIsDefaultYandexSearch"))
                    .put(SidsAdGroupAdditionalTargeting.class, additionalTargetingWithValueConverter(
                            "targetingSids"))
                    .put(UuidsAdGroupAdditionalTargeting.class, additionalTargetingWithValueConverter(
                            "targetingUuids"))
                    .put(DeviceIdsAdGroupAdditionalTargeting.class, additionalTargetingWithValueConverter(
                            "targetingDeviceIds"))
                    .put(PlusUserSegmentsAdGroupAdditionalTargeting.class, additionalTargetingWithValueConverter(
                            "targetingPlusUserSegments"))
                    .put(SearchTextAdGroupAdditionalTargeting.class, additionalTargetingWithValueConverter(
                            "targetingSearchText"))
                    .build();

    private final GridValidationService gridValidationService;

    public InternalAdGroupMutationValidationService(GridValidationService gridValidationService) {
        this.gridValidationService = gridValidationService;
    }

    /**
     * Создать индекс дубликатов для определения повторяющихся таргетингов на группах. Таргетинги без группы в индексе
     * не учитываются.
     */
    private static HashMultiset<TargetingDescriptor> createIndexOfDuplicates(List<GdAdditionalTargetingRequest> targetings) {
        return targetings.stream()
                .map(TargetingDescriptor::create)
                .collect(Collectors.toCollection(HashMultiset::create));
    }

    private static List<GdAdditionalTargetingRequest> collectTargetings(GdAdditionalTargetingUnion union) {
        return StreamEx.of(GdAdditionalTargetingUnion.allModelProperties())
                .map(x -> x.getRaw(union))
                .select(GdAdditionalTargetingRequest.class)
                .toList();
    }

    private static SkipByDefaultMappingPathNodeConverter.Builder commonAdditionalTargetingConverter(String targetingField) {
        return SkipByDefaultMappingPathNodeConverter.builder()
                .replace(AdGroupAdditionalTargeting.TARGETING_MODE.name(),
                        asList(targetingField, "targetingMode"))
                .replace(AdGroupAdditionalTargeting.JOIN_TYPE.name(),
                        asList(targetingField, "joinType"));
    }

    private static SkipByDefaultMappingPathNodeConverter additionalTargetingConverter(String targetingField) {
        return commonAdditionalTargetingConverter(targetingField).build();
    }

    private static SkipByDefaultMappingPathNodeConverter additionalTargetingWithValueConverter(String targetingField) {
        SkipByDefaultMappingPathNodeConverter.Builder builder = commonAdditionalTargetingConverter(targetingField);

        // хардкод литерала "value" для упрощения читаемости кода
        // идиоматическим вариантом обращения к нему было бы использование соответствующего ModelProperty:
        // TargetingClass.VALUE.name()
        builder.replace("value", asList(targetingField, "value"));

        return builder.build();
    }

    /**
     * Констрейнт проверяет, что среди таргетингов нет дубликатов на группе
     */
    private static Constraint<GdAdditionalTargetingUnion, Defect> doesNotHaveDuplicate(
            HashMultiset<TargetingDescriptor> indexOfDuplicates) {
        return fromPredicate(union -> collectTargetings(union)
                        .stream()
                        .allMatch(targeting -> indexOfDuplicates.count(TargetingDescriptor.create(targeting)) == 1),
                duplicatedElement());
    }

    private static Constraint<GdAdditionalTargetingUnion, Defect> hasSingleTargetingSet() {
        return fromPredicate(union -> collectTargetings(union).size() == 1, invalidUnion());
    }

    private static boolean hasInvalidHoursNumberPerDay(List<List<Integer>> timeBoard) {
        return timeBoard.stream().anyMatch(day -> day.size() != HOURS_PER_DAY);
    }

    private static boolean hasInvalidCoefs(List<List<Integer>> timeBoard) {
        return timeBoard.stream().flatMap(List::stream).anyMatch(coef -> coef != 0 && coef != 100);
    }

    private static Constraint<List<List<Integer>>, Defect> validateTimeBoard() {
        return fromPredicate(timeBoard -> timeBoard.size() == DAYS_PER_WEEK
                        && !hasInvalidHoursNumberPerDay(timeBoard)
                        && !hasInvalidCoefs(timeBoard),
                new Defect<>(GridDefectIds.TimeTarget.INVALID_TIME_BOARD_FORMAT));
    }

    private static final Validator<GdTimeTarget, Defect> INTERNAL_AD_TIME_TARGET_VALIDATOR =
            timeTarget -> {
                ModelItemValidationBuilder<GdTimeTarget> vb = ModelItemValidationBuilder.of(timeTarget);
                vb.item(GdTimeTarget.TIME_BOARD).check(validateTimeBoard(), When.notNull());
                vb.item(GdTimeTarget.ID_TIME_ZONE).check(isEqual(-1L, invalidValue())); // таймзону нельзя задать
                vb.item(GdTimeTarget.USE_WORKING_WEEKENDS).check(notNull());
                vb.item(GdTimeTarget.ENABLED_HOLIDAYS_MODE).check(notNull());
                vb.item(GdTimeTarget.HOLIDAYS_SETTINGS).checkBy(HOLIDAYS_SETTINGS_VALIDATOR,
                        When.isTrue(timeTarget.getEnabledHolidaysMode() != null && timeTarget.getEnabledHolidaysMode()));
                return vb.getResult();
            };

    private static final Validator<GdAdditionalTargetingTimeRequest, Defect> TIME_TARGET_REQUEST_DEFECT_VALIDATOR =
            timeTargetReq -> {
                var vb = ModelItemValidationBuilder.of(timeTargetReq);
                vb.item(GdAdditionalTargetingTimeRequest.VALUE)
                        .check(notNull())
                        .checkBy(INTERNAL_AD_TIME_TARGET_VALIDATOR, When.notNull());
                return vb.getResult();
            };

    private static final Validator<GdAdditionalTargetingUnion, Defect> ADDITIONAL_TARGETING_VALIDATOR =
            union -> {
                var vb = ModelItemValidationBuilder.of(union);
                vb.item(GdAdditionalTargetingUnion.TARGETING_TIME)
                        .checkBy(TIME_TARGET_REQUEST_DEFECT_VALIDATOR, When.notNull());
                return vb.getResult();
            };

    private static final Validator<List<GdAdditionalTargetingUnion>, Defect> ADDITIONAL_TARGETINGS_VALIDATOR =
            unions -> {
                var targetings = StreamEx.of(unions)
                        .map(InternalAdGroupMutationValidationService::collectTargetings)
                        .flatMap(Collection::stream)
                        .toList();
                var indexOfDuplicates = createIndexOfDuplicates(targetings);
                return ListValidationBuilder.of(unions, Defect.class)
                        .checkEach(hasSingleTargetingSet())
                        .checkEach(doesNotHaveDuplicate(indexOfDuplicates))
                        .checkEach(notNull())
                        .checkEachBy(ADDITIONAL_TARGETING_VALIDATOR, When.notNull())
                        .getResult();
            };

    private static final Validator<GdUpdateInternalAdRetargetingConditionItem, Defect> RETARGETING_CONDITION_ITEM_VALIDATOR =
            req -> {
                var lvb = ModelItemValidationBuilder.of(req);
                lvb.list(GdUpdateInternalAdRetargetingConditionItem.CONDITION_RULES)
                        .check(notEmptyCollection());
                return lvb.getResult();
            };

    private static final Validator<GdUpdateInternalAdGroupsItem, Defect> UPDATE_INTERNAL_AD_GROUP_ITEMS_VALIDATOR =
            req -> {
                var lvb = ModelItemValidationBuilder.of(req);
                lvb.list(GdUpdateInternalAdGroupsItem.TARGETINGS)
                        .checkBy(ADDITIONAL_TARGETINGS_VALIDATOR, When.notNull());
                lvb.list(GdUpdateInternalAdGroupsItem.RETARGETING_CONDITIONS)
                        .checkEachBy(RETARGETING_CONDITION_ITEM_VALIDATOR, When.notNull());
                return lvb.getResult();
            };

    private static final Validator<GdAddInternalAdGroupsItem, Defect> ADD_INTERNAL_AD_GROUP_ITEMS_VALIDATOR =
            req -> {
                var lvb = ModelItemValidationBuilder.of(req);
                lvb.list(GdAddInternalAdGroupsItem.TARGETINGS)
                        .checkBy(ADDITIONAL_TARGETINGS_VALIDATOR, When.notNull());
                lvb.list(GdAddInternalAdGroupsItem.RETARGETING_CONDITIONS)
                        .checkEachBy(RETARGETING_CONDITION_ITEM_VALIDATOR, When.notNull());
                return lvb.getResult();
            };

    private static final Validator<GdUpdateInternalAdGroups, Defect> UPDATE_INTERNAL_AD_GROUPS_VALIDATOR =
            req -> {
                ItemValidationBuilder<GdUpdateInternalAdGroups, Defect> lvb = ModelItemValidationBuilder.of(req);
                lvb.list(req.getUpdateItems(), GdUpdateInternalAdGroups.UPDATE_ITEMS.name())
                        .checkEachBy(UPDATE_INTERNAL_AD_GROUP_ITEMS_VALIDATOR);
                return lvb.getResult();
            };

    private static final Validator<GdAddInternalAdGroups, Defect> ADD_INTERNAL_AD_GROUPS_VALIDATOR =
            req -> {
                ItemValidationBuilder<GdAddInternalAdGroups, Defect> lvb = ModelItemValidationBuilder.of(req);
                lvb.list(req.getAddItems(), GdAddInternalAdGroups.ADD_ITEMS.name())
                        .checkEachBy(ADD_INTERNAL_AD_GROUP_ITEMS_VALIDATOR);
                return lvb.getResult();
            };

    void validateUpdateGroup(GdUpdateInternalAdGroups input) {
        gridValidationService.applyValidator(UPDATE_INTERNAL_AD_GROUPS_VALIDATOR, input, false);
    }

    void validateAddGroup(GdAddInternalAdGroups input) {
        gridValidationService.applyValidator(ADD_INTERNAL_AD_GROUPS_VALIDATOR, input, false);
    }

    @Nullable
    GdValidationResult getUpdateValidationResult(ValidationResult<?, Defect> vr) {
        if (hasValidationIssues(vr)) {
            return buildGridValidationResult(
                    vr, path(field(GdUpdateInternalAdGroups.UPDATE_ITEMS)), createUpdateErrorPathConverters());
        }
        return null;
    }

    @Nullable
    GdValidationResult getAddValidationResult(ValidationResult<?, Defect> vr) {
        if (hasValidationIssues(vr)) {
            return buildGridValidationResult(
                    vr, path(field(GdAddInternalAdGroups.ADD_ITEMS)), createAddErrorPathConverters());
        }
        return null;
    }

    private static DefaultPathNodeConverterProvider createUpdateErrorPathConverters() {
        PathNodeConverter adGroupConverter = SkipByDefaultMappingPathNodeConverter.builder()
                .replace(InternalAdGroup.ID, GdUpdateInternalAdGroupsItem.ID)
                .replace(InternalAdGroup.NAME, GdUpdateInternalAdGroupsItem.NAME)
                .replace(InternalAdGroup.GEO, GdUpdateInternalAdGroupsItem.REGION_IDS)
                .replace(InternalAdGroup.LEVEL, GdUpdateInternalAdGroupsItem.LEVEL)
                .replace(InternalAdGroup.RF, GdUpdateInternalAdGroupsItem.RF)
                .replace(InternalAdGroup.RF_RESET, GdUpdateInternalAdGroupsItem.RF_RESET)
                .replace(InternalAdGroup.MAX_CLICKS_COUNT, GdUpdateInternalAdGroupsItem.MAX_CLICKS_COUNT)
                .replace(InternalAdGroup.MAX_CLICKS_PERIOD, GdUpdateInternalAdGroupsItem.MAX_CLICKS_PERIOD)
                .replace(InternalAdGroup.MAX_STOPS_COUNT, GdUpdateInternalAdGroupsItem.MAX_STOPS_COUNT)
                .replace(InternalAdGroup.MAX_STOPS_PERIOD, GdUpdateInternalAdGroupsItem.MAX_STOPS_PERIOD)
                .replace(InternalAdGroup.START_TIME, GdUpdateInternalAdGroupsItem.START_TIME)
                .replace(InternalAdGroup.FINISH_TIME, GdUpdateInternalAdGroupsItem.FINISH_TIME)
                .build();
        return buildPathConverter(InternalAdGroupUpdateItem.class, adGroupConverter,
                InternalAdGroupUpdateItem.ADDITIONAL_TARGETINGS, GdUpdateInternalAdGroupsItem.TARGETINGS);
    }

    private static DefaultPathNodeConverterProvider createAddErrorPathConverters() {
        PathNodeConverter adGroupConverter = SkipByDefaultMappingPathNodeConverter.builder()
                .replace(InternalAdGroup.CAMPAIGN_ID, GdAddInternalAdGroupsItem.CAMPAIGN_ID)
                .replace(InternalAdGroup.NAME, GdAddInternalAdGroupsItem.NAME)
                .replace(InternalAdGroup.GEO, GdAddInternalAdGroupsItem.REGION_IDS)
                .replace(InternalAdGroup.LEVEL, GdAddInternalAdGroupsItem.LEVEL)
                .replace(InternalAdGroup.RF, GdAddInternalAdGroupsItem.RF)
                .replace(InternalAdGroup.RF_RESET, GdAddInternalAdGroupsItem.RF_RESET)
                .replace(InternalAdGroup.START_TIME, GdAddInternalAdGroupsItem.START_TIME)
                .replace(InternalAdGroup.FINISH_TIME, GdAddInternalAdGroupsItem.FINISH_TIME)
                .build();
        return buildPathConverter(InternalAdGroupAddItem.class, adGroupConverter,
                InternalAdGroupAddItem.ADDITIONAL_TARGETINGS, GdAddInternalAdGroupsItem.TARGETINGS);
    }

    private static PathNodeConverter createInternalAdGroupItemConverter(
            ModelProperty<?, ?> targetingsSource, ModelProperty<?, ?> targetingsDest) {
        return SkipByDefaultMappingPathNodeConverter.builder()
                .replace(targetingsSource, targetingsDest)
                .replace(InternalAdGroupAddItem.RETARGETING_CONDITIONS,
                        GdAddInternalAdGroupsItem.RETARGETING_CONDITIONS)
                .build();
    }

    private static DefaultPathNodeConverterProvider buildPathConverter(
            Class<?> cls, PathNodeConverter converter,
            ModelProperty<?, ?> targetingsSource, ModelProperty<?, ?> targetingsDest) {
        PathNodeConverter retargetingConditionConverter = SkipByDefaultMappingPathNodeConverter.builder()
                .replace(RetargetingConditionBase.RULES, GdUpdateInternalAdRetargetingConditionItem.CONDITION_RULES)
                .build();

        DefaultPathNodeConverterProvider.Builder builder = DefaultPathNodeConverterProvider.builder()
                .register(cls, createInternalAdGroupItemConverter(targetingsSource, targetingsDest))
                .register(InternalAdGroup.class, converter)
                .register(UntypedAdGroup.class, converter)
                .register(RetargetingCondition.class, retargetingConditionConverter);

        COMMON_INTERNAL_GROUP_CONVERTERS.forEach(builder::register);
        return builder.build();
    }

    private static class TargetingDescriptor {
        private final Class<? extends GdAdditionalTargetingRequest> type;
        private final GdAdditionalTargetingMode targetingMode;
        private final GdAdditionalTargetingJoinType targetingJoinType;

        private TargetingDescriptor(Class<? extends GdAdditionalTargetingRequest> type,
                                    GdAdditionalTargetingMode targetingMode,
                                    GdAdditionalTargetingJoinType targetingJoinType) {
            this.type = type;
            this.targetingMode = targetingMode;
            this.targetingJoinType = targetingJoinType;
        }

        static TargetingDescriptor create(GdAdditionalTargetingRequest targeting) {
            return new TargetingDescriptor(
                    targeting.getClass(),
                    TARGETING_AND_FILTERING_ALLOWED.contains(targeting.getClass())
                            ? targeting.getTargetingMode()
                            : GdAdditionalTargetingMode.TARGETING,
                    targeting.getJoinType()
            );
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            TargetingDescriptor that = (TargetingDescriptor) o;
            return type.equals(that.type) &&
                    targetingMode == that.targetingMode &&
                    targetingJoinType == that.targetingJoinType;
        }

        @Override
        public int hashCode() {
            return Objects.hash(type, targetingMode, targetingJoinType);
        }
    }
}
