package ru.yandex.direct.grid.processing.service.group.mutation;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupSimple;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.service.AdGroupService;
import ru.yandex.direct.core.entity.client.service.ClientGeoService;
import ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseValidator;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.processing.model.api.GdValidationResult;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateAdGroupPayloadItem;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateAdGroupRegions;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateAdGroupRegionsPayload;
import ru.yandex.direct.grid.processing.service.group.validation.AdGroupMassActionsValidationService;
import ru.yandex.direct.grid.processing.service.validation.presentation.SkipByDefaultMappingPathNodeConverter;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.Path;
import ru.yandex.direct.validation.result.PathNodeConverter;
import ru.yandex.direct.validation.result.PathNodeConverterProvider;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.function.Function.identity;
import static ru.yandex.direct.core.entity.adgroup.repository.typesupport.AdGroupTypesMapper.getTypedEmptyModelChanges;
import static ru.yandex.direct.grid.core.entity.banner.service.GridAdMassChangeUtils.getIndexMap;
import static ru.yandex.direct.grid.processing.service.group.converter.AdGroupsMutationDataConverter.getGdUpdateAdGroupPayloadItems;
import static ru.yandex.direct.grid.processing.service.validation.GridValidationResultConversionService.buildGridValidationResultIfErrors;
import static ru.yandex.direct.operation.tree.TreeOperationUtils.mergeSubListValidationResults;
import static ru.yandex.direct.utils.FunctionalUtils.flatMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.PathHelper.path;

@Service
@ParametersAreNonnullByDefault
public class UpdateAdGroupRegionsMutationService {

    public static final Path VALIDATION_RESULT_PATH = path(field(GdUpdateAdGroupRegions.AD_GROUP_IDS));

    private final AdGroupService adGroupService;
    private final AdGroupMassActionsValidationService adGroupMassActionsValidationService;
    private final ClientGeoService clientGeoService;
    private final PathNodeConverterProvider pathNodeConverterProvider;

    @Autowired
    public UpdateAdGroupRegionsMutationService(AdGroupService adGroupService, ClientGeoService clientGeoService,
                                               AdGroupMassActionsValidationService adGroupMassActionsValidationService) {
        this.adGroupService = adGroupService;
        this.adGroupMassActionsValidationService = adGroupMassActionsValidationService;
        this.clientGeoService = clientGeoService;
        this.pathNodeConverterProvider = createPathNodeConverters();
    }

    /**
     * Добавление, удаление или замена регионов групп объявлений
     */
    public GdUpdateAdGroupRegionsPayload updateAdGroupRegions(ClientId clientId, Long operatorUid,
                                                              GdUpdateAdGroupRegions input) {
        GeoTree geoTree = clientGeoService.getClientTranslocalGeoTree(clientId);
        adGroupMassActionsValidationService.validateUpdateAdGroupRegionsRequest(input, geoTree, clientId);

        Map<Long, AdGroupSimple> actualAdGroups = getActualAdGroups(clientId, input);

        Map<Long, List<Long>> actualAdGroupToRegionIds = EntryStream.of(actualAdGroups)
                .mapValues(AdGroupSimple::getGeo)
                .mapValues(geo -> clientGeoService.convertForWeb(geo, geoTree))
                .toImmutableMap();
        Map<Long, AdGroupType> actualAdGroupToType = EntryStream.of(actualAdGroups)
                .mapValues(AdGroupSimple::getType)
                .toImmutableMap();

        ValidationResult<List<Long>, Defect> vr =
                adGroupMassActionsValidationService.preValidateUpdateAdGroupRegions(input, actualAdGroupToRegionIds);

        List<Long> validAdGroupIds = ValidationResult.getValidItems(vr);
        if (validAdGroupIds.isEmpty()) {
            return buildGdUpdateAdGroupRegionsPayload(input, vr, Collections.emptyList());
        }

        List<ModelChanges<AdGroup>> modelChangesList = getUpdateRegionsModelChanges(input, validAdGroupIds,
                actualAdGroupToRegionIds, actualAdGroupToType, geoTree);
        MassResult<Long> result = adGroupService.updateAdGroupsPartialWithFullValidation(modelChangesList, geoTree,
                MinusPhraseValidator.ValidationMode.ONE_ERROR_PER_TYPE, operatorUid, clientId);

        return buildGdUpdateAdGroupRegionsPayload(input, vr, result);
    }

    private GdUpdateAdGroupRegionsPayload buildGdUpdateAdGroupRegionsPayload(GdUpdateAdGroupRegions input,
                                                                             ValidationResult<List<Long>, Defect> vr,
                                                                             MassResult<Long> result) {
        //noinspection unchecked
        var operationVr = (ValidationResult<List<Long>, Defect>) result.getValidationResult();
        Map<Integer, Integer> destToSourceIndexMap = getIndexMap(ValidationResult.getValidItemsWithIndex(vr));
        mergeSubListValidationResults(vr, operationVr, destToSourceIndexMap);

        List<GdUpdateAdGroupPayloadItem> successfullyUpdatedAdGroups = getGdUpdateAdGroupPayloadItems(result);

        return buildGdUpdateAdGroupRegionsPayload(input, vr, successfullyUpdatedAdGroups);
    }

    private GdUpdateAdGroupRegionsPayload buildGdUpdateAdGroupRegionsPayload(
            GdUpdateAdGroupRegions input,
            ValidationResult<List<Long>, Defect> vr,
            List<GdUpdateAdGroupPayloadItem> successfullyUpdatedAdGroups) {

        GdValidationResult validationResult = buildGridValidationResultIfErrors(vr, VALIDATION_RESULT_PATH,
                pathNodeConverterProvider);

        Set<Long> updatedAdGroupIds = listToSet(successfullyUpdatedAdGroups, GdUpdateAdGroupPayloadItem::getAdGroupId);
        Set<Long> skippedAdGroupIds = StreamEx.of(input.getAdGroupIds())
                .remove(updatedAdGroupIds::contains)
                .toSet();

        return new GdUpdateAdGroupRegionsPayload()
                .withUpdatedAdGroupItems(successfullyUpdatedAdGroups)
                .withSkippedAdGroupIds(skippedAdGroupIds)
                .withValidationResult(validationResult);
    }

    /**
     * Получить AdGroupSimple для данных AdGroupIds или всех групп кампаний из CampaignIds,
     * если делается поиск по кампаниям то в input проставляется AdGroupIds данных кампаний в случайном порядке.
     */
    private Map<Long, AdGroupSimple> getActualAdGroups(ClientId clientId, GdUpdateAdGroupRegions input) {
        if (input.getAdGroupIds() != null) {
            return adGroupService.getSimpleAdGroups(clientId, input.getAdGroupIds());
        }
        Map<Long, List<AdGroupSimple>> simpleAdGroups =
                adGroupService.getSimpleAdGroupsByCampaignIds(clientId, input.getCampaignIds());

        var adGroups = flatMap(simpleAdGroups.values(), identity());
        input.setAdGroupIds(mapList(adGroups, AdGroupSimple::getId));

        return StreamEx.of(adGroups)
                .mapToEntry(AdGroupSimple::getId, identity())
                .toImmutableMap();
    }

    /**
     * Заменить, добавить или убрать (в зависимости от action) регионы групп
     * Вернет список ModelChanges содержащий изменения регионов группы.
     * Порядок списка ModelChanges будет соответствовать порядку input.getAdGroupIds()
     *
     * @param validAdGroupIds      id групп, которые прошли превалидацию
     * @param actualAdGroupRegions текущие регионы у групп
     * @param actualAdGroupTypes   типы групп
     * @param geoTree              гео дерево клиента
     * @return список изменений регионов у группы
     */
    private static List<ModelChanges<AdGroup>> getUpdateRegionsModelChanges(GdUpdateAdGroupRegions input,
                                                                            List<Long> validAdGroupIds,
                                                                            Map<Long, List<Long>> actualAdGroupRegions,
                                                                            Map<Long, AdGroupType> actualAdGroupTypes,
                                                                            GeoTree geoTree) {
        return StreamEx.of(validAdGroupIds)
                .map(adGroupId ->
                        getTypedEmptyModelChanges(actualAdGroupTypes.getOrDefault(adGroupId, AdGroupType.BASE), adGroupId)
                                .process(input.getHyperGeoId(), AdGroup.HYPER_GEO_ID)
                                .process(input.getRegionIds(), AdGroup.GEO,
                                        regionIds -> getNewRegionIds(adGroupId, actualAdGroupRegions, input, geoTree)))
                .toList();
    }

    /**
     * Возвращает новый регион на группу в зависимости от input.getAction()
     * для несуществующей группы возвращаем пустой список,
     * чтобы порядок для ModelChanges соответствовал порядку input.getAdGroupIds()
     */
    static List<Long> getNewRegionIds(Long adGroupId, Map<Long, List<Long>> actualAdGroupRegions,
                                      GdUpdateAdGroupRegions input, GeoTree geoTree) {
        if (!actualAdGroupRegions.containsKey(adGroupId)) {
            return Collections.emptyList();
        }

        switch (input.getAction()) {
            case ADD:
                return geoTree.includeRegions(actualAdGroupRegions.get(adGroupId), input.getRegionIds());
            case REMOVE:
                return geoTree.excludeRegions(actualAdGroupRegions.get(adGroupId), input.getRegionIds());
            case REPLACE:
                return input.getRegionIds();
            default:
                throw new IllegalArgumentException("Unknown update mode: " + input.getAction());
        }
    }

    private static PathNodeConverterProvider createPathNodeConverters() {
        // не отдаем фронту пути из core, т.к. в запросе есть только список adGroupIds
        PathNodeConverter skipMappingPathNodeConverter = SkipByDefaultMappingPathNodeConverter.builder().build();

        return clazz -> skipMappingPathNodeConverter;
    }
}
