package ru.yandex.direct.grid.processing.service.group.mutation;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Preconditions;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.container.InternalAdGroupOperationContainer;
import ru.yandex.direct.core.entity.adgroup.container.InternalAdGroupOperationContainer.RequestSource;
import ru.yandex.direct.core.entity.adgroup.container.InternalAdGroupUpdateItem;
import ru.yandex.direct.core.entity.adgroup.service.complex.internal.ComplexInternalAdGroupService;
import ru.yandex.direct.core.entity.internalads.service.InternalAdsProductService;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.grid.processing.model.api.GdValidationResult;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateAdGroupPayload;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateAdGroupPayloadItem;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateInternalAdGroups;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.result.MassResult;

import static ru.yandex.direct.grid.processing.util.ResponseConverter.getResults;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class UpdateInternalAdGroupsMutationService {

    private final InternalAdsProductService internalAdsProductService;
    private final ComplexInternalAdGroupService complexInternalAdGroupService;
    private final InternalAdGroupMutationValidationService mutationValidationService;

    public UpdateInternalAdGroupsMutationService(InternalAdsProductService internalAdsProductService,
                                                 ComplexInternalAdGroupService complexInternalAdGroupService,
                                                 InternalAdGroupMutationValidationService mutationValidationService) {
        this.internalAdsProductService = internalAdsProductService;
        this.complexInternalAdGroupService = complexInternalAdGroupService;
        this.mutationValidationService = mutationValidationService;
    }

    /**
     * Обновление групп объявлений для внутренней рекламы
     *
     * @param owner       идентификаторы клиента, которому принадлежат создаваемые группы
     * @param operatorUid идентификатор пользователя оператора
     * @param input       запрос, содержащий список данных для обновления групп
     */
    public GdUpdateAdGroupPayload updateInternalAdGroups(UidAndClientId owner, Long operatorUid,
                                                         GdUpdateInternalAdGroups input) {
        Preconditions.checkState(internalAdsProductService.clientCanHaveInternalAdCampaigns(owner.getClientId()),
                "Internal ads are not supported for client");

        mutationValidationService.validateUpdateGroup(input);

        List<InternalAdGroupUpdateItem> coreUpdateItems = mapList(
                input.getUpdateItems(), item -> GdInternalGroupConverters.toCoreUpdateItem(owner.getClientId(), item));

        var operationContainer = new InternalAdGroupOperationContainer(
                Applicability.PARTIAL, operatorUid, owner, true, RequestSource.FRONT
        );
        MassResult<Long> result = complexInternalAdGroupService.update(coreUpdateItems, operationContainer);

        GdValidationResult validationResult = mutationValidationService.getUpdateValidationResult(
                result.getValidationResult());

        List<GdUpdateAdGroupPayloadItem> payloadItems =
                getResults(result, id -> new GdUpdateAdGroupPayloadItem().withAdGroupId(id));

        return new GdUpdateAdGroupPayload()
                .withUpdatedAdGroupItems(payloadItems)
                .withValidationResult(validationResult);
    }
}
