package ru.yandex.direct.grid.processing.service.group.type;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Maps;
import com.google.common.collect.Multimap;
import com.google.common.collect.Multimaps;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.model.entity.adgroup.GdAdGroupType;
import ru.yandex.direct.grid.processing.model.group.GdAdGroup;

import static ru.yandex.direct.grid.processing.service.group.type.GdAdGroupTypeAssertionsMap.assertGdAdGroupHasConsistentType;

@Component
@ParametersAreNonnullByDefault
public class GroupTypeFacade {
    private final Map<GdAdGroupType, GroupTypeSupport> supportMap;

    @Autowired
    public GroupTypeFacade(List<GroupTypeSupport> adGroupTypeSupports) {
        this.supportMap = Maps.uniqueIndex(adGroupTypeSupports, GroupTypeSupport::getAdGroupType);
    }

    /**
     * Для каждой группы выставляет доступные типы смежных объектов(баннеры, условия нацелевания).
     */
    @SuppressWarnings("unchecked")
    public void setAvailableEntityTypes(int shard, ClientId clientId, Long operatorUid,
                                        Collection<GdAdGroup> gdAdGroups) {
        collectByTypeSupports(gdAdGroups).asMap().forEach(
                ((adGroupTypeSupport, adGroups) -> {
                    adGroupTypeSupport.setAvailableShowConditionTypesForAdGroup(shard, clientId, operatorUid, adGroups);
                    adGroupTypeSupport.setAvailableBannerTypesForAdGroup(shard, clientId, adGroups);
                }));
    }

    private Multimap<GroupTypeSupport, GdAdGroup> collectByTypeSupports(Collection<GdAdGroup> gdAdGroups) {
        return Multimaps.index(gdAdGroups, this::getTypeSupport);
    }

    private <G extends GdAdGroup> GroupTypeSupport<G> getTypeSupport(G gdAdGroup) {
        assertGdAdGroupHasConsistentType(gdAdGroup);
        return getTypeSupport(gdAdGroup.getType());
    }

    @SuppressWarnings("unchecked")
    private <G extends GdAdGroup> GroupTypeSupport<G> getTypeSupport(GdAdGroupType adGroupType) {
        return supportMap.get(adGroupType);
    }
}
