package ru.yandex.direct.grid.processing.service.group.validation;

import java.util.Collection;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Iterables;
import one.util.streamex.EntryStream;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.copyentity.CopyResultIds;
import ru.yandex.direct.core.entity.adgroup.container.ComplexTextAdGroup;
import ru.yandex.direct.core.entity.adgroup.container.UntypedAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupWithType;
import ru.yandex.direct.core.entity.adgroup.model.TextAdGroup;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.banner.container.ComplexBanner;
import ru.yandex.direct.core.entity.banner.model.TextBanner;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.ComplexBidModifier;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithType;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.client.model.ClientLimits;
import ru.yandex.direct.core.entity.client.service.ClientLimitsService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.relevancematch.model.RelevanceMatch;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingConditionBase;
import ru.yandex.direct.core.entity.retargeting.model.TargetInterest;
import ru.yandex.direct.core.entity.vcard.model.Vcard;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.grid.processing.model.api.GdValidationResult;
import ru.yandex.direct.grid.processing.model.group.mutation.GdCopyAdGroups;
import ru.yandex.direct.grid.processing.service.group.AvailableAdGroupTypesCalculator;
import ru.yandex.direct.grid.processing.service.validation.presentation.SkipByDefaultMappingPathNodeConverter;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.DefaultPathNodeConverterProvider;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.Path;
import ru.yandex.direct.validation.result.PathNodeConverterProvider;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static java.util.Collections.singletonList;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.adGroupTypeNotSupported;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.maxAdGroupsInCampaign;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstraints.campaignIsNotArchived;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.campaignTypeNotSupported;
import static ru.yandex.direct.core.validation.defects.Defects.badStatusCampaignArchived;
import static ru.yandex.direct.grid.processing.service.validation.GridValidationResultConversionService.buildGridValidationResultIfErrors;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notInSet;
import static ru.yandex.direct.validation.defect.CommonDefects.objectNotFound;
import static ru.yandex.direct.validation.result.PathHelper.path;

@Service
@ParametersAreNonnullByDefault
public class AdGroupCopyValidationService {

    private static final String AD_FIELD_NAME = "ad";
    private static final String HIERARCHICAL_MULTIPLIERS_FIELD_NAME = "hierarchicalMultipliers";

    private final AdGroupRepository adGroupRepository;
    private final CampaignRepository campaignRepository;
    private final ClientLimitsService clientLimitsService;
    private final PathNodeConverterProvider errorPathConverterProvider;
    private final ShardHelper shardHelper;
    private final FeatureService featureService;

    @Autowired
    public AdGroupCopyValidationService(AdGroupRepository adGroupRepository,
                                        CampaignRepository campaignRepository,
                                        ClientLimitsService clientLimitsService,
                                        ShardHelper shardHelper,
                                        FeatureService featureService) {
        this.adGroupRepository = adGroupRepository;
        this.campaignRepository = campaignRepository;
        this.clientLimitsService = clientLimitsService;
        this.shardHelper = shardHelper;
        this.featureService = featureService;
        this.errorPathConverterProvider = createErrorPathConverters();
    }

    @Nullable
    public GdValidationResult validateCopyAdGroups(ClientId clientId, GdCopyAdGroups input) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        Map<Long, AdGroupWithType> adGroupsWithType =
                adGroupRepository.getAdGroupsWithType(shard, clientId, input.getAdGroupIds());
        Set<Long> existingAdGroupIds = adGroupsWithType.keySet();
        Set<Long> adGroupsInArchivedCampaigns =
                adGroupRepository.getAdGroupIdsInArchivedCampaigns(shard, existingAdGroupIds);
        Map<Long, CampaignWithType> campaignsWithTypeByAdGroupIds =
                campaignRepository.getCampaignsWithTypeByAdGroupIds(shard, clientId, existingAdGroupIds);

        Set<String> enabledFeatures = featureService.getEnabledForClientId(clientId);
        ModelItemValidationBuilder<GdCopyAdGroups> vb = ModelItemValidationBuilder.of(input);
        vb.list(GdCopyAdGroups.AD_GROUP_IDS)
                .checkEach(inSet(existingAdGroupIds), objectNotFound())
                .checkEach(adGroupTypeIsApplicable(adGroupsWithType, campaignsWithTypeByAdGroupIds, enabledFeatures),
                        When.isValid())
                .checkEach(notInSet(adGroupsInArchivedCampaigns), badStatusCampaignArchived(), When.isValid());

        if (input.getDestinationCampaignId() != null) {
            validateDestinationCampaign(vb, input.getDestinationCampaignId(), clientId, shard);
        }

        return buildGridValidationResultIfErrors(vb.getResult(), path(), errorPathConverterProvider);
    }

    private void validateDestinationCampaign(ModelItemValidationBuilder<GdCopyAdGroups> vb,
                                             Long destinationCampaignId, ClientId clientId, int shard) {
        Map<Long, CampaignWithType> campaignsWithTypeByCampaignIds = campaignRepository
                .getCampaignsWithTypeByCampaignIds(shard, clientId, singletonList(destinationCampaignId));
        Set<Long> existingCampaignIds = campaignsWithTypeByCampaignIds.keySet();
        Map<Long, Long> adGroupCountByCampaignIds =
                adGroupRepository.getAdGroupCountByCampaignIds(shard, existingCampaignIds);
        ClientLimits clientLimits = getClientLimits(clientId);
        Set<Long> archivedCampaignIds = EntryStream.of(campaignsWithTypeByCampaignIds)
                .filterValues(CampaignWithType::isArchived)
                .keys()
                .toSet();

        Set<String> enabledFeatures = featureService.getEnabledForClientId(clientId);
        vb.item(GdCopyAdGroups.DESTINATION_CAMPAIGN_ID)
                .check(inSet(existingCampaignIds), objectNotFound())
                .check(campaignTypeIsApplicable(campaignsWithTypeByCampaignIds, enabledFeatures), When.isValid())
                .check(maxAdGroupCountInCampaign(adGroupCountByCampaignIds, clientLimits), When.isValid())
                .check(campaignIsNotArchived(archivedCampaignIds), When.isValid());
    }

    @Nullable
    public GdValidationResult getValidationResult(MassResult<Long> result, Path path) {
        return buildGridValidationResultIfErrors(result.getValidationResult(), path, errorPathConverterProvider);
    }

    private ClientLimits getClientLimits(ClientId clientId) {
        ClientLimits defaultClientLimits = new ClientLimits();
        defaultClientLimits.withBannersCountLimit(0L);

        Collection<ClientLimits> clientLimitsList = clientLimitsService.massGetClientLimits(singletonList(clientId));
        return Iterables.getFirst(clientLimitsList, defaultClientLimits);
    }


    private static Constraint<Long, Defect> adGroupTypeIsApplicable(Map<Long, AdGroupWithType> adGroupsWithType,
                                                                    Map<Long, CampaignWithType> campaignsWithTypeByAdGroupIds,
                                                                    Set<String> enabledFeatures) {
        Predicate<Long> predicate = adGroupId -> {
            AdGroupType adGroupType = adGroupsWithType.get(adGroupId).getType();
            CampaignType campaignType = campaignsWithTypeByAdGroupIds.get(adGroupId).getType();
            return AvailableAdGroupTypesCalculator.canCopyAdGroup(adGroupType, campaignType, enabledFeatures);
        };
        return Constraint.fromPredicate(predicate, adGroupTypeNotSupported());
    }

    private static Constraint<Long, Defect> campaignTypeIsApplicable(
            Map<Long, CampaignWithType> campaignsWithTypeByCampaignIds, Set<String> enabledFeatures) {
        Predicate<Long> predicate = campaignId -> {
            CampaignType type = campaignsWithTypeByCampaignIds.get(campaignId).getType();
            return AvailableAdGroupTypesCalculator.canCopyAdGroupForCampaign(type, enabledFeatures);
        };
        return Constraint.fromPredicate(predicate, campaignTypeNotSupported());
    }

    private static Constraint<Long, Defect> maxAdGroupCountInCampaign(Map<Long, Long> adGroupCountByCampaignIds,
                                                                      ClientLimits clientLimits) {
        return fromPredicate(campaignId ->
                        adGroupCountByCampaignIds.getOrDefault(campaignId, 0L) <= clientLimits.getBannersCountLimitOrDefault(),
                maxAdGroupsInCampaign(clientLimits.getBannersCountLimitOrDefault()));
    }

    private static DefaultPathNodeConverterProvider createErrorPathConverters() {
        return DefaultPathNodeConverterProvider.builder()
                .register(UntypedAdGroup.class, SkipByDefaultMappingPathNodeConverter.emptyConverter())
                .register(TextAdGroup.class, SkipByDefaultMappingPathNodeConverter.builder()
                        .replace(ComplexTextAdGroup.COMPLEX_BANNERS.name(), AD_FIELD_NAME)
                        .replace(ComplexTextAdGroup.KEYWORDS, ComplexTextAdGroup.KEYWORDS)
                        .replace(ComplexTextAdGroup.TARGET_INTERESTS, ComplexTextAdGroup.TARGET_INTERESTS)
                        .replace(ComplexTextAdGroup.RELEVANCE_MATCHES, ComplexTextAdGroup.RELEVANCE_MATCHES)
                        .replace(ComplexTextAdGroup.COMPLEX_BID_MODIFIER.name(), HIERARCHICAL_MULTIPLIERS_FIELD_NAME)
                        .replace(TextAdGroup.TAGS, TextAdGroup.TAGS)
                        .build())

                .register(TextBanner.class, SkipByDefaultMappingPathNodeConverter.builder()
                        .replace(ComplexBanner.VCARD, ComplexBanner.VCARD)
                        .build())
                .register(Vcard.class, SkipByDefaultMappingPathNodeConverter.emptyConverter())

                .register(Keyword.class, SkipByDefaultMappingPathNodeConverter.emptyConverter())
                .register(TargetInterest.class, SkipByDefaultMappingPathNodeConverter.emptyConverter())
                .register(RelevanceMatch.class, SkipByDefaultMappingPathNodeConverter.emptyConverter())
                .register(ComplexBidModifier.class, SkipByDefaultMappingPathNodeConverter.emptyConverter())
                .register(CopyResultIds.class, SkipByDefaultMappingPathNodeConverter.builder()
                        .replace(Keyword.class.getSimpleName(), ComplexTextAdGroup.KEYWORDS.name())
                        .replace(RetargetingConditionBase.class.getSimpleName(),
                                ComplexTextAdGroup.TARGET_INTERESTS.name())
                        .replace(RelevanceMatch.class.getSimpleName(), ComplexTextAdGroup.RELEVANCE_MATCHES.name())
                        .replace(BidModifier.class.getSimpleName(), HIERARCHICAL_MULTIPLIERS_FIELD_NAME)
                        .build())
                .build();
    }

}
