package ru.yandex.direct.grid.processing.service.group.validation

import org.springframework.stereotype.Service
import ru.yandex.direct.core.validation.ValidationUtils
import ru.yandex.direct.grid.processing.model.api.GdValidationResult
import ru.yandex.direct.grid.processing.model.group.GdInternalAdGroupFieldChange
import ru.yandex.direct.grid.processing.model.group.GdInternalAdGroupFieldChangeValue
import ru.yandex.direct.grid.processing.model.group.GdInternalAdGroupFieldChangeValueUnion
import ru.yandex.direct.grid.processing.model.group.GdInternalAdGroupsAggregatedState
import ru.yandex.direct.grid.processing.model.group.GdInternalAdGroupsMassUpdate
import ru.yandex.direct.grid.processing.service.validation.GridDefectDefinitions
import ru.yandex.direct.grid.processing.service.validation.GridValidationResultConversionService.buildGridValidationResult
import ru.yandex.direct.grid.processing.service.validation.GridValidationService
import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.builder.Constraint.fromPredicate
import ru.yandex.direct.validation.constraint.CollectionConstraints.eachValidId
import ru.yandex.direct.validation.constraint.CollectionConstraints.maxListSize
import ru.yandex.direct.validation.constraint.CollectionConstraints.notEmptyCollection
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.Path
import ru.yandex.direct.validation.result.PathHelper.field
import ru.yandex.direct.validation.result.PathHelper.path
import ru.yandex.direct.validation.result.PathNode
import ru.yandex.direct.validation.result.PathNodeConverter
import ru.yandex.direct.validation.result.PathNodeConverterProvider
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.validation.util.item
import ru.yandex.direct.validation.util.list
import ru.yandex.direct.validation.util.validateModel

const val MAX_UPDATE_ITEMS = 200

@Service
class AdGroupInternalMassUpdateValidationService(private val gridValidationService: GridValidationService) {

    fun validateInternalAdGroupsAggregatedState(input: GdInternalAdGroupsAggregatedState) =
        gridValidationService.applyValidator(::validateGdInternalAdGroupsAggregatedState, input, false)

    fun validateInternalAdGroupsMassUpdate(input: GdInternalAdGroupsMassUpdate) =
        gridValidationService.applyValidator(::validateGdInternalAdGroupsMassUpdate, input, false)

    fun validateGdInternalAdGroupsAggregatedState(
        obj: GdInternalAdGroupsAggregatedState
    ): ValidationResult<GdInternalAdGroupsAggregatedState, Defect<*>> {
        return validateModel(obj) {
            item(GdInternalAdGroupsAggregatedState.AD_GROUP_IDS) {
                check(notEmptyCollection())
                check(maxListSize(MAX_UPDATE_ITEMS))
                check(eachValidId())
            }
        }
    }

    private fun validateGdInternalAdGroupFieldChange(obj: GdInternalAdGroupFieldChange) =
        validateModel(obj) {
            item(GdInternalAdGroupFieldChange.IDS) {
                check(notEmptyCollection())
                check(maxListSize(MAX_UPDATE_ITEMS))
                check(eachValidId())
            }
            item(GdInternalAdGroupFieldChange.VALUE) {
                check(unionHasSingleValue())
            }
        }

    fun validateGdInternalAdGroupsMassUpdate(obj: GdInternalAdGroupsMassUpdate) =
        validateModel(obj) {
            list(GdInternalAdGroupsMassUpdate.CHANGES) {
                check(notEmptyCollection())
                check(maxListSize(MAX_UPDATE_ITEMS))
                checkEachBy(::validateGdInternalAdGroupFieldChange)
            }
        }

    private fun unionHasSingleValue(): Constraint<GdInternalAdGroupFieldChangeValueUnion, Defect<*>> {
        return fromPredicate({ union -> collectUnionValues(union).size == 1 }, GridDefectDefinitions.invalidUnion())
    }

    private fun collectUnionValues(value: GdInternalAdGroupFieldChangeValueUnion) =
        GdInternalAdGroupFieldChangeValueUnion.allModelProperties()
            .mapNotNull { it.getRaw(value) }
            .filterIsInstance<GdInternalAdGroupFieldChangeValue<Any>>()

    fun getValidationResult(vr: ValidationResult<*, Defect<*>>): GdValidationResult? {
        return if (ValidationUtils.hasValidationIssues(vr)) {
            buildGridValidationResult(vr, path(field("adGroups")), createErrorPathConverters())
        } else null
    }

    private fun createErrorPathConverters(): PathNodeConverterProvider {
        val identityPathConverter = object : PathNodeConverter {
            override fun convert(field: PathNode.Field) = Path(listOf(field))
            override fun convert(field: PathNode.Field, index: PathNode.Index) = Path(listOf(field, index))
        }
        return PathNodeConverterProvider { identityPathConverter }
    }
}
