package ru.yandex.direct.grid.processing.service.group.validation;

import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import io.leangen.graphql.util.Utils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.service.AdGroupService;
import ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.service.CampaignService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.feature.service.validation.FeatureDefects;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.constraint.CommonConstraints;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.web.core.entity.inventori.validation.CampaignDefectIds;

@Service
@ParametersAreNonnullByDefault
public class AdGroupKeywordRecommendationValidationService {

    private final Logger logger = LoggerFactory.getLogger(AdGroupKeywordRecommendationValidationService.class);

    private final CampaignService campaignService;
    private final AdGroupService adGroupService;
    private final FeatureService featureService;

    public AdGroupKeywordRecommendationValidationService(CampaignService campaignService,
                                                         AdGroupService adGroupService,
                                                         FeatureService featureService) {
        this.campaignService = campaignService;
        this.adGroupService = adGroupService;
        this.featureService = featureService;
    }

    public ValidationResult<CampaignType, Defect> validateCampaignType(ClientId clientId,
                                                                       @Nullable Long adGroupId,
                                                                       @Nullable Long campaignId) {
        ItemValidationBuilder<CampaignType, Defect> vb;
        if (adGroupId != null) {
            var campaignIds = adGroupService.getCampaignIdsByAdgroupIds(clientId, List.of(adGroupId)).values();
            var campaignType = campaignService.getCampaignsTypes(campaignIds).values().stream().findAny();
            vb = ItemValidationBuilder.of(campaignType.orElse(null));
        } else if (campaignId != null) {
            var campaignType = campaignService.getCampaignsTypes(Set.of(campaignId)).values().stream().findAny();
            vb = ItemValidationBuilder.of(campaignType.orElse(null));
        } else {
            vb = ItemValidationBuilder.of(null);
            logger.error("AdGroupId and CampaignId cannot be null together");
        }

        vb.check(CommonConstraints.notNull())
                .check(checkCampaignType());

        return vb.getResult();
    }

    public ValidationResult<ClientId, Defect> validateClientIdWithEnabledFeature(ClientId clientId) {
        ItemValidationBuilder<ClientId, Defect> vb = ItemValidationBuilder.of(clientId);
        vb.check(Constraint.fromPredicate(this::isFeatureEnabled, invalidClientId()));
        return vb.getResult();
    }

    private boolean isFeatureEnabled(ClientId clientId) {
        return featureService.isEnabledForClientId(clientId, FeatureName.RELEVANCE_MATCH_CATEGORIES_ALLOWED_IN_UC);
    }

    public ValidationResult<List<Long>, Defect> validateCampaignId(List<Long> adGroupIds) {
        ItemValidationBuilder<List<Long>, Defect> vb = ItemValidationBuilder.of(adGroupIds);
        vb.check(Constraint.fromPredicate(Utils::isNotEmpty, invalidCampaign()));
        return vb.getResult();
    }

    public static Defect<CampaignType> invalidCampaignType() {
        return new Defect(CampaignDefectIds.CampaignTypeDefects.INVALID_CAMPAIGN_TYPE);
    }

    private Constraint<CampaignType, Defect> checkCampaignType() {
        return Constraint.fromPredicate(SUPPORTED_CAMPAIGN_TYPES::contains, invalidCampaignType());
    }

    public static Defect<ClientId> invalidClientId() {
        return new Defect(FeatureDefects.featureIsAlreadyDisabledForRole().defectId());
    }

    private static Defect<Long> invalidCampaign() {
        return new Defect(AdGroupDefects.notFound().defectId());
    }

    public static final Set<CampaignType> SUPPORTED_CAMPAIGN_TYPES = Set.of(CampaignType.TEXT);
}
