package ru.yandex.direct.grid.processing.service.group.validation;

import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.client.Constants;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.region.validation.RegionIdsValidator;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.processing.model.client.GdOperatorAction;
import ru.yandex.direct.grid.processing.model.group.mutation.GdChangeAdGroupsRelevanceMatch;
import ru.yandex.direct.grid.processing.model.group.mutation.GdCopyAdGroups;
import ru.yandex.direct.grid.processing.model.group.mutation.GdRemoderateAdsCallouts;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateAdGroupRegions;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateAdGroupRegionsAction;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.constraint.CollectionConstraints;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.region.validation.RegionIdDefects.geoIncorrectUseOfZeroRegion;
import static ru.yandex.direct.core.validation.defects.RightsDefects.noRights;
import static ru.yandex.direct.grid.processing.service.operator.OperatorAllowedActionsUtils.hasAction;
import static ru.yandex.direct.grid.processing.service.validation.GridValidationService.checkMutuallyExclusiveFields;
import static ru.yandex.direct.regions.Region.GLOBAL_REGION_ID;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.maxListSize;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.notEmptyCollection;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.unique;
import static ru.yandex.direct.validation.constraint.CommonConstraints.isNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.constraint.NumberConstraints.notLessThan;

@Service
@ParametersAreNonnullByDefault
public class AdGroupMassActionsValidationService {

    public static final int MAX_AD_GROUPS_COUNT_PER_UPDATE = Math.toIntExact(Constants.DEFAULT_BANNERS_COUNT_LIMIT);
    private static final List<Long> GLOBAL_GEO = List.of(GLOBAL_REGION_ID);

    private final GridValidationService gridValidationService;
    private final RegionIdsValidator regionsValidator;
    private final FeatureService featureService;

    private static final Validator<List<Long>, Defect> AD_IDS_VALIDATOR = req -> {
        ListValidationBuilder<Long, Defect> vb = ListValidationBuilder.of(req);
        vb
                .check(notNull())
                .check(notEmptyCollection())
                .checkEach(notNull(), When.isValid())
                .checkEach(validId(), When.isValid())
                .checkEach(unique(), When.isValid());
        return vb.getResult();
    };

    private static final Validator<GdChangeAdGroupsRelevanceMatch, Defect> CHANGE_AD_GROUPS_RELEVANCE_MATCH_VALIDATOR =
            req -> {
                ModelItemValidationBuilder<GdChangeAdGroupsRelevanceMatch> vb = ModelItemValidationBuilder.of(req);
                vb.list(GdChangeAdGroupsRelevanceMatch.AD_GROUP_IDS)
                        .checkBy(AD_IDS_VALIDATOR);
                vb.item(GdChangeAdGroupsRelevanceMatch.ENABLE_RELEVANCE_MATCH)
                        .check(notNull());
                return vb.getResult();
            };

    @Autowired
    public AdGroupMassActionsValidationService(GridValidationService gridValidationService,
                                               RegionIdsValidator regionsValidator,
                                               FeatureService featureService) {
        this.gridValidationService = gridValidationService;
        this.regionsValidator = regionsValidator;
        this.featureService = featureService;
    }

    public void validateRemoderateAdsCalloutsRequest(User operator, GdRemoderateAdsCallouts request) {
        gridValidationService.applyValidator(new RemoderateAdsCalloutsValidator(operator), request, false);
    }

    public void validateCopyAdGroupsRequest(User operator, GdCopyAdGroups request) {
        gridValidationService.applyValidator(new CopyAdGroupsValidator(operator), request, false);
    }

    public void validateChangeAdGroupsRelevanceMatch(User operator, GdChangeAdGroupsRelevanceMatch request) {
        gridValidationService.applyValidator(CHANGE_AD_GROUPS_RELEVANCE_MATCH_VALIDATOR, request, false);
    }

    public void validateUpdateAdGroupRegionsRequest(GdUpdateAdGroupRegions request,
                                                    GeoTree geoTree,
                                                    ClientId clientId) {
        gridValidationService
                .applyValidator(new UpdateAdGroupRegionsValidator(geoTree, clientId), request, false);
    }

    public ValidationResult<List<Long>, Defect> preValidateUpdateAdGroupRegions(
            GdUpdateAdGroupRegions input,
            Map<Long, List<Long>> actualAdGroupToRegionIds) {
        Set<Long> adGroupsWithGlobalGeo = EntryStream.of(actualAdGroupToRegionIds)
                .filterValues(GLOBAL_GEO::equals)
                .keys()
                .toSet();

        return ListValidationBuilder.<Long, Defect>of(input.getAdGroupIds())
                .checkEach(dontRemoveFromGlobalGeo(adGroupsWithGlobalGeo),
                        When.isTrue(input.getAction() == GdUpdateAdGroupRegionsAction.REMOVE))
                .getResult();
    }

    private static Constraint<Long, Defect> dontRemoveFromGlobalGeo(Set<Long> adGroupWithGlobalGeo) {
        return Constraint.fromPredicate(id -> !adGroupWithGlobalGeo.contains(id), geoIncorrectUseOfZeroRegion());
    }

    private class UpdateAdGroupRegionsValidator implements Validator<GdUpdateAdGroupRegions, Defect> {

        private final GeoTree geoTree;
        private final ClientId clientId;

        UpdateAdGroupRegionsValidator(GeoTree geoTree, ClientId clientId) {
            this.geoTree = geoTree;
            this.clientId = clientId;
        }

        @Override
        public ValidationResult<GdUpdateAdGroupRegions, Defect> apply(GdUpdateAdGroupRegions request) {
            ModelItemValidationBuilder<GdUpdateAdGroupRegions> vb = ModelItemValidationBuilder.of(request);
            vb.check(checkMutuallyExclusiveFields(List.of(
                    GdUpdateAdGroupRegions.CAMPAIGN_IDS, GdUpdateAdGroupRegions.AD_GROUP_IDS)));

            validateGeoAndHyperGeo(vb, request);

            vb.list(GdUpdateAdGroupRegions.CAMPAIGN_IDS)
                    .check(maxListSize(1), When.notNull())
                    .checkEach(validId());

            vb.list(GdUpdateAdGroupRegions.AD_GROUP_IDS)
                    .check(notEmptyCollection())
                    .check(maxListSize(AdGroupMassActionsValidationService.MAX_AD_GROUPS_COUNT_PER_UPDATE))
                    .checkEach(validId());
            return vb.getResult();
        }

        private void validateGeoAndHyperGeo(ModelItemValidationBuilder<GdUpdateAdGroupRegions> vb,
                                            GdUpdateAdGroupRegions request) {
            boolean updateToHyperGeo = request.getHyperGeoId() != null;

            vb.list(GdUpdateAdGroupRegions.REGION_IDS)
                    .check(notNull(), When.isFalse(updateToHyperGeo))
                    .check(isNull(), When.isTrue(updateToHyperGeo))
                    .checkEach(notLessThan(0L),
                            When.isFalse(request.getAction() == GdUpdateAdGroupRegionsAction.REPLACE))
                    .checkBy(regionIds -> regionsValidator.apply(regionIds, geoTree), When.notNull());
        }
    }

    private static class RemoderateAdsCalloutsValidator implements Validator<GdRemoderateAdsCallouts, Defect> {

        private final User operator;

        private RemoderateAdsCalloutsValidator(User operator) {
            this.operator = operator;
        }

        @Override
        public ValidationResult<GdRemoderateAdsCallouts, Defect> apply(GdRemoderateAdsCallouts request) {
            ModelItemValidationBuilder<GdRemoderateAdsCallouts> vb = ModelItemValidationBuilder.of(request);

            vb.check(operatorHasAccess(this.operator));

            vb.item(GdRemoderateAdsCallouts.AD_GROUP_IDS)
                    .check(notEmptyCollection())
                    .check(CollectionConstraints.maxSetSize(AdGroupMassActionsValidationService.MAX_AD_GROUPS_COUNT_PER_UPDATE));
            return vb.getResult();
        }

        private static Constraint<GdRemoderateAdsCallouts, Defect> operatorHasAccess(User operator) {
            return fromPredicate(
                    request -> request.getModerateAccept()
                            ? hasAction(GdOperatorAction.ACCEPT_ADS_CALLOUTS_MODERATION, operator)
                            : hasAction(GdOperatorAction.REMODERATE_ADS_CALLOUTS, operator),
                    noRights());
        }
    }

    private static class CopyAdGroupsValidator implements Validator<GdCopyAdGroups, Defect> {

        private final User operator;

        private CopyAdGroupsValidator(User operator) {
            this.operator = operator;
        }

        @Override
        public ValidationResult<GdCopyAdGroups, Defect> apply(GdCopyAdGroups request) {
            ModelItemValidationBuilder<GdCopyAdGroups> vb = ModelItemValidationBuilder.of(request);

            vb.check(operatorHasAccess(this.operator));

            vb.list(GdCopyAdGroups.AD_GROUP_IDS)
                    .check(notEmptyCollection())
                    .check(maxListSize(AdGroupMassActionsValidationService.MAX_AD_GROUPS_COUNT_PER_UPDATE))
                    .checkEach(validId());
            vb.item(GdCopyAdGroups.DESTINATION_CAMPAIGN_ID)
                    .check(validId(), When.notNull());
            return vb.getResult();
        }

        private static Constraint<GdCopyAdGroups, Defect> operatorHasAccess(User operator) {
            return fromPredicate(ignore -> hasAction(GdOperatorAction.COPY, operator), noRights());
        }
    }

}
