package ru.yandex.direct.grid.processing.service.group.validation;

import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.container.UntypedAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.ContentPromotionAdGroup;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.processing.model.api.GdValidationResult;
import ru.yandex.direct.grid.processing.model.group.mutation.GdAddContentPromotionAdGroup;
import ru.yandex.direct.grid.processing.model.group.mutation.GdAddContentPromotionAdGroupItem;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateContentPromotionAdGroup;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateContentPromotionAdGroupItem;
import ru.yandex.direct.grid.processing.service.validation.GridDefectDefinitions;
import ru.yandex.direct.grid.processing.service.validation.GridValidationResultConversionService;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.DefaultPathNodeConverterProvider;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.Path;
import ru.yandex.direct.validation.result.PathNodeConverterProvider;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.adgroup.model.AdGroupType.CONTENT_PROMOTION;
import static ru.yandex.direct.grid.processing.service.group.validation.AdGroupValidationUtils.checkKeywordsUniformityForAdd;
import static ru.yandex.direct.grid.processing.service.group.validation.AdGroupValidationUtils.checkKeywordsUniformityForUpdate;
import static ru.yandex.direct.grid.processing.service.group.validation.AdGroupValidationUtils.checkRelevanceMatchUniformityForAdd;
import static ru.yandex.direct.grid.processing.service.group.validation.AdGroupValidationUtils.checkRelevanceMatchUniformityForUpdate;
import static ru.yandex.direct.grid.processing.service.validation.presentation.AdGroupConverters.CONTENT_PROMOTION_AD_GROUP_PATH_CONVERTER;
import static ru.yandex.direct.grid.processing.service.validation.presentation.AdGroupConverters.UPDATE_AD_GROUP_PATH_CONVERTER;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.notEmptyCollection;
import static ru.yandex.direct.validation.constraint.CommonConstraints.eachNotNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.constraint.NumberConstraints.inRange;
import static ru.yandex.direct.validation.defect.CommonDefects.objectNotFound;

@Service
@ParametersAreNonnullByDefault
public class ContentPromotionAdGroupValidationService {

    private final GridValidationService gridValidationService;
    private final ClientService clientService;

    private final PathNodeConverterProvider pathNodeConverterProvider;

    @Autowired
    public ContentPromotionAdGroupValidationService(GridValidationService gridValidationService,
                                                    ClientService clientService) {
        this.gridValidationService = gridValidationService;
        this.clientService = clientService;

        this.pathNodeConverterProvider = DefaultPathNodeConverterProvider.builder()
                .register(UntypedAdGroup.class, UPDATE_AD_GROUP_PATH_CONVERTER)
                .register(ContentPromotionAdGroup.class, CONTENT_PROMOTION_AD_GROUP_PATH_CONVERTER)
                .build();
    }

    // Валидация добавления

    public void validateAddContentPromotionAdGroups(GdAddContentPromotionAdGroup gdAddRequest, ClientId clientId) {
        gridValidationService.applyValidator(addRequest -> validateAddRequest(addRequest, clientId), gdAddRequest, false);
    }

    private ValidationResult<GdAddContentPromotionAdGroup, Defect> validateAddRequest(GdAddContentPromotionAdGroup gdAddRequest,
                                                                                      ClientId clientId) {
        ModelItemValidationBuilder<GdAddContentPromotionAdGroup> vb = ModelItemValidationBuilder.of(gdAddRequest);

        Currency clientWorkCurrency = clientService.getWorkCurrency(clientId);

        vb.list(GdAddContentPromotionAdGroup.ADD_ITEMS)
                .check(notNull())
                .check(notEmptyCollection(), When.isValid())
                .check(checkKeywordsUniformityForAdd(), When.isValid())
                .check(checkRelevanceMatchUniformityForAdd(), When.isValid())
                .checkEachBy(addItem -> validateAddItem(addItem, clientWorkCurrency), When.isValid());

        return vb.getResult();
    }

    private ValidationResult<GdAddContentPromotionAdGroupItem, Defect> validateAddItem(GdAddContentPromotionAdGroupItem gdAddItem,
                                                                                       Currency clientWorkCurrency) {
        ModelItemValidationBuilder<GdAddContentPromotionAdGroupItem> vb = ModelItemValidationBuilder.of(gdAddItem);

        vb.item(GdAddContentPromotionAdGroupItem.CAMPAIGN_ID)
                .check(notNull())
                .check(validId(), When.isValid());
        vb.item(GdAddContentPromotionAdGroupItem.AD_GROUP_MINUS_KEYWORDS)
                .check(eachNotNull());
        vb.list(GdAddContentPromotionAdGroupItem.LIBRARY_MINUS_KEYWORDS_IDS)
                .check(eachNotNull())
                .checkEach(validId(), When.isValid());
        vb.item(GdAddContentPromotionAdGroupItem.KEYWORDS)
                .check(eachNotNull());
        vb.item(GdAddContentPromotionAdGroupItem.GENERAL_PRICE)
                .check(inRange(clientWorkCurrency.getMinPrice(), clientWorkCurrency.getMaxPrice()), When.notNull());

        return vb.getResult();
    }

    // Превалидация обновления

    public void preValidateUpdateContentPromotionAdGroups(GdUpdateContentPromotionAdGroup adGroupsUpdateRequest) {
        gridValidationService.applyValidator(this::preValidateUpdateRequest, adGroupsUpdateRequest, false);
    }

    private ValidationResult<GdUpdateContentPromotionAdGroup, Defect> preValidateUpdateRequest(GdUpdateContentPromotionAdGroup gdUpdateRequest) {
        ModelItemValidationBuilder<GdUpdateContentPromotionAdGroup> vb = ModelItemValidationBuilder.of(gdUpdateRequest);

        vb.list(GdUpdateContentPromotionAdGroup.UPDATE_ITEMS)
                .check(notNull())
                .check(notEmptyCollection(), When.isValid())
                .check(checkKeywordsUniformityForUpdate(), When.isValid())
                .check(checkRelevanceMatchUniformityForUpdate(), When.isValid())
                .checkEachBy(this::preValidateUpdateItem, When.isValid());

        return vb.getResult();
    }

    private ValidationResult<GdUpdateContentPromotionAdGroupItem, Defect> preValidateUpdateItem(GdUpdateContentPromotionAdGroupItem gdUpdateItem) {
        ModelItemValidationBuilder<GdUpdateContentPromotionAdGroupItem> vb = ModelItemValidationBuilder.of(gdUpdateItem);

        vb.item(GdUpdateContentPromotionAdGroupItem.AD_GROUP_ID)
                .check(notNull())
                .check(validId(), When.isValid());
        vb.item(GdUpdateContentPromotionAdGroupItem.AD_GROUP_MINUS_KEYWORDS)
                .check(eachNotNull());
        vb.list(GdUpdateContentPromotionAdGroupItem.LIBRARY_MINUS_KEYWORDS_IDS)
                .check(eachNotNull())
                .checkEach(validId(), When.isValid());
        vb.item(GdUpdateContentPromotionAdGroupItem.KEYWORDS)
                .check(eachNotNull());

        return vb.getResult();
    }

    // Валидация обновления

    public void validateUpdateContentPromotionAdGroups(GdUpdateContentPromotionAdGroup adGroupsUpdateRequest,
                                                       ClientId clientId,
                                                       Map<Long, AdGroupType> adGroupTypeByAdGroupId) {
        gridValidationService.applyValidator(gdUpdateRequest ->
                validateUpdateRequest(gdUpdateRequest, clientId, adGroupTypeByAdGroupId), adGroupsUpdateRequest, false);
    }

    private ValidationResult<GdUpdateContentPromotionAdGroup, Defect> validateUpdateRequest(GdUpdateContentPromotionAdGroup gdUpdateRequest,
                                                                                            ClientId clientId,
                                                                                            Map<Long, AdGroupType> adGroupTypeByAdGroupId) {
        Currency clientWorkCurrency = clientService.getWorkCurrency(clientId);

        Set<Long> existingAdGroupIds = adGroupTypeByAdGroupId.keySet();
        Set<Long> contentPromotionAdGroupIds = EntryStream.of(adGroupTypeByAdGroupId)
                .filterValues(CONTENT_PROMOTION::equals)
                .keys()
                .toSet();

        ModelItemValidationBuilder<GdUpdateContentPromotionAdGroup> vb = ModelItemValidationBuilder.of(gdUpdateRequest);

        vb.list(GdUpdateContentPromotionAdGroup.UPDATE_ITEMS)
                .checkEachBy(item -> validateUpdateItem(item, clientWorkCurrency, existingAdGroupIds, contentPromotionAdGroupIds));

        return vb.getResult();
    }

    private ValidationResult<GdUpdateContentPromotionAdGroupItem, Defect> validateUpdateItem(GdUpdateContentPromotionAdGroupItem gdUpdateItem,
                                                                                             Currency clientWorkCurrency,
                                                                                             Set<Long> existingAdGroupIds,
                                                                                             Set<Long> contentPromotionAdGroupIds) {
        ModelItemValidationBuilder<GdUpdateContentPromotionAdGroupItem> vb = ModelItemValidationBuilder.of(gdUpdateItem);

        vb.item(GdUpdateContentPromotionAdGroupItem.AD_GROUP_ID)
                .check(inSet(existingAdGroupIds), objectNotFound())
                .check(inSet(contentPromotionAdGroupIds), GridDefectDefinitions.unsupportedAdGroupType(), When.isValid());
        vb.item(GdUpdateContentPromotionAdGroupItem.GENERAL_PRICE)
                .check(inRange(clientWorkCurrency.getMinPrice(), clientWorkCurrency.getMaxPrice()), When.notNull());

        return vb.getResult();
    }

    // Получение результатов валидации

    @Nullable
    public GdValidationResult getValidationResult(ValidationResult<?, Defect> vr, Path path) {
        return GridValidationResultConversionService
                .buildGridValidationResultIfErrors(vr, path, pathNodeConverterProvider);
    }
}
