package ru.yandex.direct.grid.processing.service.group.validation;

import java.math.BigDecimal;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Predicate;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Sets;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.container.UntypedAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.PerformanceAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.TextAdGroup;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.currency.model.cpmyndxfrontpage.CpmYndxFrontpageAdGroupPriceRestrictions;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.grid.processing.model.api.GdValidationResult;
import ru.yandex.direct.grid.processing.model.group.GdCriterionType;
import ru.yandex.direct.grid.processing.model.group.mutation.GdCpmGroupType;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateAdGroupKeywordItem;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateCpmAdGroup;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateCpmAdGroupItem;
import ru.yandex.direct.grid.processing.model.retargeting.mutation.GdUpdateCpmRetargetingConditionItem;
import ru.yandex.direct.grid.processing.service.validation.GridValidationResultConversionService;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListItemValidator;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.DefaultPathNodeConverterProvider;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.Path;
import ru.yandex.direct.validation.result.PathNodeConverterProvider;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static org.apache.commons.lang3.ObjectUtils.firstNonNull;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.adGroupTypeNotSupported;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.eitherKeywordsOrRetargetingsAllowed;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.emptyContentCategoriesNotAllowed;
import static ru.yandex.direct.core.entity.bids.validation.BidsDefects.oneOfFieldsShouldBeSpecified;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.campaignNotFound;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.RetargetingDefectIds.Gen.INCONSISTENT_RETARGETING_CONDITION_BY_DEFAULT_ADGROUP;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.RetargetingDefectIds.Gen.INVALID_RETARGETING_CONDITION_BY_PRICE_PACKAGE;
import static ru.yandex.direct.core.validation.ValidationUtils.hasValidationIssues;
import static ru.yandex.direct.feature.FeatureName.CONTENT_CATEGORY_TARGETING_CPM_EXTENDED;
import static ru.yandex.direct.feature.FeatureName.CPM_BANNER_ENABLE_EMPTY_RET_COND_JSON;
import static ru.yandex.direct.feature.FeatureName.TARGETING_IS_NOT_REQUIRED_FOR_CPM_GEOPRODUCT_GROUP;
import static ru.yandex.direct.grid.processing.model.group.mutation.GdCpmGroupType.CPM_AUDIO;
import static ru.yandex.direct.grid.processing.model.group.mutation.GdCpmGroupType.CPM_BANNER;
import static ru.yandex.direct.grid.processing.model.group.mutation.GdCpmGroupType.CPM_GEOPRODUCT;
import static ru.yandex.direct.grid.processing.model.group.mutation.GdCpmGroupType.CPM_GEO_PIN;
import static ru.yandex.direct.grid.processing.model.group.mutation.GdCpmGroupType.CPM_INDOOR;
import static ru.yandex.direct.grid.processing.model.group.mutation.GdCpmGroupType.CPM_OUTDOOR;
import static ru.yandex.direct.grid.processing.model.group.mutation.GdCpmGroupType.CPM_PRICE;
import static ru.yandex.direct.grid.processing.model.group.mutation.GdCpmGroupType.CPM_VIDEO;
import static ru.yandex.direct.grid.processing.model.group.mutation.GdCpmGroupType.CPM_YNDX_FRONTPAGE;
import static ru.yandex.direct.grid.processing.service.group.AdGroupTypeUtils.getEditableCpmAdGroupTypes;
import static ru.yandex.direct.grid.processing.service.group.converter.AdGroupsMutationDataConverter.hasContentCategories;
import static ru.yandex.direct.grid.processing.service.showcondition.validation.RetargetingValidationService.UPDATE_CPM_RETARGETING_CONDITION_ITEM_VALIDATOR;
import static ru.yandex.direct.grid.processing.service.validation.presentation.AdGroupConverters.UPDATE_AD_GROUP_PATH_CONVERTER;
import static ru.yandex.direct.grid.processing.service.validation.presentation.AdGroupConverters.UPDATE_PERFORMANCE_AD_GROUP_PATH_CONVERTER;
import static ru.yandex.direct.grid.processing.service.validation.presentation.AdGroupConverters.UPDATE_TEXT_AD_GROUP_PATH_CONVERTER;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.notEmptyCollection;
import static ru.yandex.direct.validation.constraint.CommonConstraints.eachNotNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.constraint.NumberConstraints.inRange;

@Service
@ParametersAreNonnullByDefault
public class CpmAdGroupValidationService {
    private static final Predicate<GdUpdateCpmAdGroupItem> AD_GROUP_ID_OR_CAMPAIGN_ID_IS_NOT_NULL =
            item -> item.getAdGroupId() != null || item.getCampaignId() != null;
    private static final List<ModelProperty<?, ?>> AD_GROUP_ID_OR_CAMPAIGN_ID_SHOULD_BE_SPECIFIED =
            Arrays.asList(GdUpdateCpmAdGroupItem.CAMPAIGN_ID, GdUpdateCpmAdGroupItem.AD_GROUP_ID);
    private final PathNodeConverterProvider pathNodeConverterProvider;
    private final CampaignRepository campaignRepository;
    private final AdGroupRepository adGroupRepository;

    @Autowired
    public CpmAdGroupValidationService(CampaignRepository campaignRepository,
                                       AdGroupRepository adGroupRepository) {
        this.campaignRepository = campaignRepository;
        this.adGroupRepository = adGroupRepository;
        this.pathNodeConverterProvider = DefaultPathNodeConverterProvider.builder()
                .register(UntypedAdGroup.class, UPDATE_AD_GROUP_PATH_CONVERTER)
                .register(TextAdGroup.class, UPDATE_TEXT_AD_GROUP_PATH_CONVERTER)
                .register(PerformanceAdGroup.class, UPDATE_PERFORMANCE_AD_GROUP_PATH_CONVERTER)
                .build();
    }

    /**
     * Первичная валидация входных параметров
     *
     * @param shard                                номер шарда
     * @param keywordsByAdGroupIds                 текущие ключевые фразы на группах
     * @param adCpmGroupsUpdateRequest             запрос на обновление охватных групп
     * @param cpmYndxFrontpageCurrencyRestrictions мапа индекс на прайс рестрикшен
     */
    public ValidationResult<GdUpdateCpmAdGroup, Defect> validateUpdateCpmAdGroups(int shard, Currency workCurrency,
                                                                                  Map<Long, List<Keyword>>
                                                                                          keywordsByAdGroupIds,
                                                                                  GdUpdateCpmAdGroup
                                                                                          adCpmGroupsUpdateRequest,
                                                                                  Set<String> enabledFeatures,
                                                                                  Map<Integer,
                                                                                          CpmYndxFrontpageAdGroupPriceRestrictions> cpmYndxFrontpageCurrencyRestrictions) {
        CpmAdGroupUpdateValidator cmpAdGroupUpdateValidator =
                new CpmAdGroupUpdateValidator(shard, workCurrency, keywordsByAdGroupIds, enabledFeatures,
                        cpmYndxFrontpageCurrencyRestrictions);
        return cmpAdGroupUpdateValidator.apply(adCpmGroupsUpdateRequest);
    }

    @Nullable
    public GdValidationResult getValidationResult(ValidationResult<?, Defect> vr, Path path) {
        if (hasValidationIssues(vr)) {
            GdValidationResult validationResult = GridValidationResultConversionService
                    .buildGridValidationResult(vr, path, pathNodeConverterProvider);
            return concertGdValidationResult(validationResult);
        }
        return null;
    }

    /**
     * Для прайсовых кампаний баги INVALID_RETARGETING_CONDITION_BY_PRICE_PACKAGE и
     * INCONSISTENT_RETARGETING_CONDITION_BY_DEFAULT_ADGROUP должны относится к группе в целом в graphQL ответах
     */
    public GdValidationResult concertGdValidationResult(GdValidationResult validationResult) {
        StreamEx.of(validationResult.getErrors())
                .filter(e -> e.getCode().contains(INVALID_RETARGETING_CONDITION_BY_PRICE_PACKAGE.getCode())
                        || e.getCode().contains(INCONSISTENT_RETARGETING_CONDITION_BY_DEFAULT_ADGROUP.getCode()))
                .forEach(e -> e.setPath(topLevelPath(e.getPath())));
        return validationResult;
    }

    private String topLevelPath(String path) {
        if (!path.contains(".")) {
            return path;
        }
        return path.substring(0, path.indexOf("."));
    }

    private static final Validator<GdUpdateCpmRetargetingConditionItem, Defect>
            RETARGETING_CONDITION_RULES_ARE_NOT_EMPTY_VALIDATOR = item -> {
        ModelItemValidationBuilder<GdUpdateCpmRetargetingConditionItem> vb = ModelItemValidationBuilder.of(item);
        vb.list(GdUpdateCpmRetargetingConditionItem.CONDITION_RULES)
                .check(notEmptyCollection(), When.notNull());
        return vb.getResult();
    };

    private class CpmAdGroupUpdateItemValidator implements ListItemValidator<GdUpdateCpmAdGroupItem, Defect> {
        private final Currency workCurrency;
        private final Map<Long, Long> campaignIdsByAdGroupIds;
        private final Map<Long, Boolean> campaignAutoBudgetMap;
        private final Map<Long, CampaignType> campaignsTypeMap;
        private final Map<Long, List<Keyword>> keywordsByAdGroupIds;
        private final Set<String> enabledFeatures;
        private final Map<Integer, CpmYndxFrontpageAdGroupPriceRestrictions> cpmYndxFrontpageCurrencyRestrictions;

        CpmAdGroupUpdateItemValidator(
                Currency workCurrency, Map<Long, Long> campaignIdsByAdGroupIds,
                Map<Long, Boolean> campaignAutoBudgetMap, Map<Long, CampaignType> campaignsTypeMap,
                Map<Long, List<Keyword>> keywordsByAdGroupIds,
                Set<String> enabledFeatures,
                Map<Integer, CpmYndxFrontpageAdGroupPriceRestrictions> cpmYndxFrontpageCurrencyRestrictions) {
            this.workCurrency = workCurrency;
            this.campaignIdsByAdGroupIds = campaignIdsByAdGroupIds;
            this.campaignAutoBudgetMap = campaignAutoBudgetMap;
            this.campaignsTypeMap = campaignsTypeMap;
            this.keywordsByAdGroupIds = keywordsByAdGroupIds;
            this.enabledFeatures = enabledFeatures;
            this.cpmYndxFrontpageCurrencyRestrictions = cpmYndxFrontpageCurrencyRestrictions;
        }

        @Override
        public ValidationResult<GdUpdateCpmAdGroupItem, Defect> validate(int index, GdUpdateCpmAdGroupItem item) {
            ModelItemValidationBuilder<GdUpdateCpmAdGroupItem> vb = ModelItemValidationBuilder.of(item);

            vb.item(GdUpdateCpmAdGroupItem.CAMPAIGN_ID)
                    .check(validId())
                    .check(Constraint.fromPredicate(campaignAutoBudgetMap::containsKey, campaignNotFound()));
            vb.item(GdUpdateCpmAdGroupItem.AD_GROUP_ID)
                    .check(validId());
            vb.item(GdUpdateCpmAdGroupItem.KEYWORDS)
                    .check(notEmptyCollection(), When.notNull())
                    .check(eachNotNull());
            vb.item(GdUpdateCpmAdGroupItem.RETARGETING_CONDITION)
                    .checkBy(UPDATE_CPM_RETARGETING_CONDITION_ITEM_VALIDATOR, When.notNull())
                    .checkBy(RETARGETING_CONDITION_RULES_ARE_NOT_EMPTY_VALIDATOR,
                            When.isTrue(item.getRetargetingCondition() != null &&
                                    !enabledFeatures.contains(CPM_BANNER_ENABLE_EMPTY_RET_COND_JSON.getName())
                                    && item.getType() != GdCpmGroupType.CPM_YNDX_FRONTPAGE
                                    // Поправить в DIRECT-122633 - [Этап 3] Поддержка пакетных настроек
                                    // в кампании/группе/баннере
                                    && item.getType() != GdCpmGroupType.CPM_PRICE_VIDEO
                                    && item.getType() != GdCpmGroupType.CPM_PRICE_FRONTPAGE_VIDEO
                                    && item.getType() != GdCpmGroupType.CPM_PRICE_AUDIO
                                    && item.getType() != GdCpmGroupType.CPM_PRICE_BANNER
                            ));


            Set<GdCpmGroupType> editableCpmAdGroupTypes = getEditableCpmAdGroupTypes(enabledFeatures);
            vb.item(GdUpdateCpmAdGroupItem.TYPE)
                    .check(editableCpmAdGroupType(editableCpmAdGroupTypes));

            if (vb.getResult().hasAnyErrors()) {
                return vb.getResult();
            }

            Long adGroupId = item.getAdGroupId();
            Long campaignId = firstNonNull(item.getCampaignId(), campaignIdsByAdGroupIds.get(adGroupId));
            Boolean autoBudget = campaignAutoBudgetMap.get(campaignId);
            CampaignType campaignType = campaignsTypeMap.get(campaignId);

            boolean isIndoorOrOutdoor = item.getType() == CPM_INDOOR || item.getType() == CPM_OUTDOOR;
            vb.item(GdUpdateCpmAdGroupItem.PAGE_BLOCKS)
                    .check(notNull(), When.isTrue(isIndoorOrOutdoor))
                    .check(notEmptyCollection())
                    .check(eachNotNull());

            boolean retargetingsAreProvided = item.getRetargetingCondition() != null;
            Set<String> existingKeywords =
                    keywordsByAdGroupIds.getOrDefault(adGroupId, Collections.emptyList()).stream()
                            .map(Keyword::getPhrase)
                            .collect(toSet());
            @Nullable Set<String> updatedKeywords = item.getKeywords() != null ? item.getKeywords().stream()
                    .map(GdUpdateAdGroupKeywordItem::getPhrase)
                    .collect(toSet()) : Collections.emptySet();
            Set<String> newKeywords = Sets.difference(updatedKeywords, existingKeywords);
            BigDecimal minCpmPrice = getMinCpmPrice(index, item);

            vb.item(GdUpdateCpmAdGroupItem.GENERAL_PRICE)
                    // для прайсовых ставка не используется, любое значение переданное пользователем - игнорируем
                    //  - при добавлении сами добавим ставку из пакета
                    //  - при изменении менять ставку нельзя (а существующая ставка должна совпадать с пакетом)
                    .check(notNull(), When.isTrue(
                            campaignType != CampaignType.CPM_PRICE
                                    && !autoBudget
                                    && (retargetingsAreProvided || !newKeywords.isEmpty() || isIndoorOrOutdoor)))
                    .check(inRange(minCpmPrice, workCurrency.getMaxCpmPrice()), When.isValidAnd(When.notNull()));
            return vb.getResult();
        }

        private BigDecimal getMinCpmPrice(int index, GdUpdateCpmAdGroupItem item) {
            if (item.getType() == CPM_YNDX_FRONTPAGE) {
                return cpmYndxFrontpageCurrencyRestrictions.get(index).getCpmYndxFrontpageMinPrice();
            }
            return workCurrency.getMinCpmPrice();
        }
    }

    private class CpmAdGroupUpdateValidator implements Validator<GdUpdateCpmAdGroup, Defect> {
        private final int shard;
        private final Map<Long, List<Keyword>> keywordsByAdGroupIds;
        private Currency workCurrency;
        private final Set<String> enabledFeatures;
        private final Map<Integer, CpmYndxFrontpageAdGroupPriceRestrictions> cpmYndxFrontpageCurrencyRestrictions;

        CpmAdGroupUpdateValidator(int shard, Currency workCurrency, Map<Long, List<Keyword>> keywordsByAdGroupIds,
                                  Set<String> enabledFeatures,
                                  Map<Integer, CpmYndxFrontpageAdGroupPriceRestrictions> cpmYndxFrontpageCurrencyRestrictions) {
            this.shard = shard;
            this.workCurrency = workCurrency;
            this.keywordsByAdGroupIds = keywordsByAdGroupIds;
            this.enabledFeatures = enabledFeatures;
            this.cpmYndxFrontpageCurrencyRestrictions = cpmYndxFrontpageCurrencyRestrictions;
        }

        @Override
        public ValidationResult<GdUpdateCpmAdGroup, Defect> apply(GdUpdateCpmAdGroup req) {
            List<Long> adGroupIds = req.getUpdateCpmAdGroupItems().stream()
                    .map(GdUpdateCpmAdGroupItem::getAdGroupId)
                    .filter(Objects::nonNull)
                    .collect(toList());

            List<Long> campaignIds = req.getUpdateCpmAdGroupItems().stream()
                    .map(GdUpdateCpmAdGroupItem::getCampaignId)
                    .filter(Objects::nonNull)
                    .collect(toList());

            Map<Long, Long> campaignIdsByAdGroupIds = adGroupRepository.getCampaignIdsByAdGroupIds(shard, adGroupIds);
            List<Campaign> campaignsWithStrategy =
                    campaignRepository.getCampaignsWithStrategy(shard,
                            StreamEx.of(campaignIdsByAdGroupIds.values()).append(campaignIds).collect(toSet()));

            Map<Long, Boolean> campaignAutoBudgetMap = listToMap(campaignsWithStrategy, Campaign::getId,
                    c -> c.getStrategy().isAutoBudget());
            Map<Long, CampaignType> campaignsTypeMap = listToMap(campaignsWithStrategy, Campaign::getId,
                    Campaign::getType);

            ModelItemValidationBuilder<GdUpdateCpmAdGroup> vb = ModelItemValidationBuilder.of(req);

            CpmAdGroupUpdateItemValidator itemValidator = new CpmAdGroupUpdateItemValidator(workCurrency,
                    campaignIdsByAdGroupIds, campaignAutoBudgetMap, campaignsTypeMap, keywordsByAdGroupIds,
                    enabledFeatures, cpmYndxFrontpageCurrencyRestrictions);

            vb.list(GdUpdateCpmAdGroup.UPDATE_CPM_AD_GROUP_ITEMS)
                    .check(notEmptyCollection(), When.notNull())
                    .checkEachBy(CpmAdGroupValidationService::contentCategoriesNotEmpty)
                    .checkEach(eitherKeywordsOrRetargetingsLinked(enabledFeatures), When.isValid())
                    .checkEach(Constraint.fromPredicate(AD_GROUP_ID_OR_CAMPAIGN_ID_IS_NOT_NULL,
                            oneOfFieldsShouldBeSpecified(AD_GROUP_ID_OR_CAMPAIGN_ID_SHOULD_BE_SPECIFIED)))
                    .checkEachBy(itemValidator, When.isValid());
            return vb.getResult();
        }
    }

    /**
     * При нацеливании только на жанры и категории они должны быть заполнены
     */
    private static ValidationResult<GdUpdateCpmAdGroupItem, Defect> contentCategoriesNotEmpty(GdUpdateCpmAdGroupItem adGroup) {
        ModelItemValidationBuilder<GdUpdateCpmAdGroupItem> vb = ModelItemValidationBuilder.of(adGroup);
        vb.item(GdUpdateCpmAdGroupItem.CONTENT_CATEGORIES_RETARGETING_CONDITION_RULES)
                .check(Constraint.fromPredicate(
                        cpmAdGroup -> {
                            if (GdCriterionType.CONTENT_CATEGORY.equals(adGroup.getCriterionType())) {
                                return hasContentCategories(adGroup);
                            }

                            return true;
                        },
                        emptyContentCategoriesNotAllowed()));

        return vb.getResult();
    }

    /**
     * Группы должны быть нацелены на ключевые слова или профиль пользователя.
     * Для некоторых типов групп профиль остается пустым, поэтому от фронта его не требуем
     */
    private static Constraint<GdUpdateCpmAdGroupItem, Defect> eitherKeywordsOrRetargetingsLinked(Set<String> enabledFeatures) {
        return Constraint.fromPredicate(
                cpmAdGroup -> {
                    GdCpmGroupType type = cpmAdGroup.getType();
                    if (type == CPM_INDOOR
                            || type == CPM_OUTDOOR
                            || type == CPM_PRICE
                            || type == CPM_YNDX_FRONTPAGE) {
                        // аудитории и ключевые слова недоступны для этих групп
                        return true;
                    }

                    if (type == CPM_GEO_PIN) {
                        // ключевые слова недоступны, а аудитории необязательны
                        return true;
                    }

                    // Убрать во время реализации DIRECT-122633: [Этап 3] Поддержка пакетных настроек
                    // в кампании/группе/баннере
                    if (type == GdCpmGroupType.CPM_PRICE_VIDEO
                            || type == GdCpmGroupType.CPM_PRICE_FRONTPAGE_VIDEO
                            || type == GdCpmGroupType.CPM_PRICE_AUDIO
                            || type == GdCpmGroupType.CPM_PRICE_BANNER) {
                        return true;
                    }

                    boolean hasKeywords = cpmAdGroup.getKeywords() != null;
                    boolean hasRetargetings = cpmAdGroup.getRetargetingCondition() != null;
                    if (type == CPM_GEOPRODUCT &&
                            enabledFeatures.contains(TARGETING_IS_NOT_REQUIRED_FOR_CPM_GEOPRODUCT_GROUP.getName())) {
                        // при включении фичи, можно не заполнять и аудитории, и ключевые фразы
                        return !hasKeywords || !hasRetargetings;
                    }

                    if ((type == CPM_BANNER || type == CPM_VIDEO)
                            && (GdCriterionType.USER_PROFILE.equals(cpmAdGroup.getCriterionType())
                            || GdCriterionType.CONTENT_CATEGORY.equals(cpmAdGroup.getCriterionType()))
                            && enabledFeatures.contains(CPM_BANNER_ENABLE_EMPTY_RET_COND_JSON.getName())) {
                        // с фичей CPM_BANNER_ENABLE_EMPTY_RET_COND_JSON можно не заполнять профиль и категории контента
                        return true;
                    }

                    boolean hasContentCategories = hasContentCategories(cpmAdGroup);
                    if (enabledFeatures.contains(CONTENT_CATEGORY_TARGETING_CPM_EXTENDED.getName())) {
                        // можно задавать аудитории, ключевые фразы и/или категории контента
                        return hasRetargetings || hasContentCategories || hasKeywords;
                    }

                    // в остальных случаях должны быть заполнены либо ключевые фразы, либо аудитории, либо категории
                    // контента
                    return StreamEx.of(hasKeywords, hasRetargetings, hasContentCategories)
                            .mapToInt(b -> b ? 1 : 0)
                            .sum() == 1;
                },
                eitherKeywordsOrRetargetingsAllowed());
    }

    private static Constraint<GdCpmGroupType, Defect> editableCpmAdGroupType(Set<GdCpmGroupType>
                                                                                     editableCpmAdGroupTypes) {
        return Constraint.fromPredicate(
                groupType ->
                        !List.of(CPM_GEOPRODUCT, CPM_GEO_PIN, CPM_INDOOR, CPM_OUTDOOR, CPM_AUDIO).contains(groupType)
                                        || editableCpmAdGroupTypes.contains(groupType),
                adGroupTypeNotSupported());
    }
}
