package ru.yandex.direct.grid.processing.service.group.validation;

import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.container.UntypedAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.DynamicFeedAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.DynamicTextAdGroup;
import ru.yandex.direct.core.entity.adgroup.service.AdGroupService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.processing.model.api.GdValidationResult;
import ru.yandex.direct.grid.processing.model.group.mutation.GdAddDynamicAdGroup;
import ru.yandex.direct.grid.processing.model.group.mutation.GdAddDynamicAdGroupItem;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateAdGroupRelevanceMatchItem;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateDynamicAdGroup;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateDynamicAdGroupItem;
import ru.yandex.direct.grid.processing.service.validation.GridDefectDefinitions;
import ru.yandex.direct.grid.processing.service.validation.GridValidationResultConversionService;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.DefaultPathNodeConverterProvider;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.Path;
import ru.yandex.direct.validation.result.PathNodeConverterProvider;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.grid.processing.service.validation.GridValidationService.checkMutuallyExclusiveFields;
import static ru.yandex.direct.grid.processing.service.validation.presentation.AdGroupConverters.UPDATE_AD_GROUP_PATH_CONVERTER;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.notEmptyCollection;
import static ru.yandex.direct.validation.constraint.CommonConstraints.eachNotNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.isNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.isTrue;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.defect.CommonDefects.objectNotFound;

@Service
@ParametersAreNonnullByDefault
public class DynamicAdGroupValidationService {

    private static final Validator<GdAddDynamicAdGroupItem, Defect> AD_GROUP_ADD_ITEMS_VALIDATOR = items -> {
        ModelItemValidationBuilder<GdAddDynamicAdGroupItem> vb = ModelItemValidationBuilder.of(items);

        vb.check(checkMutuallyExclusiveFields(List.of(
                GdAddDynamicAdGroupItem.FEED_ID, GdAddDynamicAdGroupItem.DOMAIN_URL)));

        vb.item(GdAddDynamicAdGroupItem.CAMPAIGN_ID)
                .check(validId(), When.isValid());

        return vb.getResult();
    };

    private static final Validator<GdAddDynamicAdGroup, Defect> AD_GROUP_ADD_VALIDATOR = gdAddDynamicAdGroup -> {
        ModelItemValidationBuilder<GdAddDynamicAdGroup> vb = ModelItemValidationBuilder.of(gdAddDynamicAdGroup);
        vb.list(GdAddDynamicAdGroup.ADD_ITEMS)
                .check(notEmptyCollection(), When.notNull())
                .checkEachBy(AD_GROUP_ADD_ITEMS_VALIDATOR);
        return vb.getResult();
    };

    private final GridValidationService gridValidationService;
    private final AdGroupService adGroupService;
    private final PathNodeConverterProvider pathNodeConverterProvider;

    @Autowired
    public DynamicAdGroupValidationService(GridValidationService gridValidationService,
                                           AdGroupService adGroupService) {
        this.gridValidationService = gridValidationService;
        this.adGroupService = adGroupService;
        this.pathNodeConverterProvider = DefaultPathNodeConverterProvider.builder()
                .register(UntypedAdGroup.class, UPDATE_AD_GROUP_PATH_CONVERTER)
                .register(DynamicTextAdGroup.class, UPDATE_AD_GROUP_PATH_CONVERTER)
                .register(DynamicFeedAdGroup.class, UPDATE_AD_GROUP_PATH_CONVERTER)
                .build();
    }

    public void validateAddDynamicAdGroups(GdAddDynamicAdGroup gdAddRequest) {
        gridValidationService.applyValidator(AD_GROUP_ADD_VALIDATOR, gdAddRequest, false);
    }

    public void validateUpdateDynamicAdGroups(ClientId clientId, GdUpdateDynamicAdGroup gdUpdateRequest) {
        gridValidationService.applyValidator(req -> validateUpdateRequest(req, clientId), gdUpdateRequest, false);
    }

    private ValidationResult<GdUpdateDynamicAdGroup, Defect> validateUpdateRequest(GdUpdateDynamicAdGroup req,
                                                                                   ClientId clientId) {
        List<Long> adGroupIds = mapList(req.getUpdateItems(), GdUpdateDynamicAdGroupItem::getId);
        Map<Long, AdGroupType> adGroupTypes = adGroupService.getAdGroupTypes(clientId, adGroupIds);
        Set<Long> existAdGroupIds = adGroupTypes.keySet();
        Set<Long> dynamicAdGroupIds = EntryStream.of(adGroupTypes)
                .filterValues(v -> v == AdGroupType.DYNAMIC)
                .mapKeyValue((k, v) -> k)
                .toSet();
        ModelItemValidationBuilder<GdUpdateDynamicAdGroup> vb = ModelItemValidationBuilder.of(req);
        vb.list(GdUpdateDynamicAdGroup.UPDATE_ITEMS)
                .checkEachBy(item -> validateUpdateItem(item, existAdGroupIds, dynamicAdGroupIds));
        return vb.getResult();
    }

    private ValidationResult<GdUpdateDynamicAdGroupItem, Defect> validateUpdateItem(
            GdUpdateDynamicAdGroupItem item,
            Set<Long> existAdGroupIds,
            Set<Long> dynamicAdGroupIds) {
        ModelItemValidationBuilder<GdUpdateDynamicAdGroupItem> vb = ModelItemValidationBuilder.of(item);
        vb.item(GdUpdateDynamicAdGroupItem.ID)
                .check(validId())
                .check(inSet(existAdGroupIds), objectNotFound(), When.isValid())
                .check(inSet(dynamicAdGroupIds), GridDefectDefinitions.unsupportedAdGroupType(), When.isValid());
        vb.item(GdUpdateDynamicAdGroupItem.MINUS_KEYWORDS)
                .check(eachNotNull());
        vb.item(GdUpdateDynamicAdGroupItem.REGION_IDS)
                .check(eachNotNull());
        vb.item(GdUpdateDynamicAdGroupItem.RELEVANCE_MATCH)
                .checkBy(this::validateUpdateRelevanceMatchItem, When.notNull());
        return vb.getResult();
    }

    private ValidationResult<GdUpdateAdGroupRelevanceMatchItem, Defect> validateUpdateRelevanceMatchItem(
            GdUpdateAdGroupRelevanceMatchItem relevanceMatch) {
        ModelItemValidationBuilder<GdUpdateAdGroupRelevanceMatchItem> vb =
                ModelItemValidationBuilder.of(relevanceMatch);
        vb.item(GdUpdateAdGroupRelevanceMatchItem.ID)
                .check(isNull());
        vb.item(GdUpdateAdGroupRelevanceMatchItem.IS_ACTIVE)
                .check(isTrue());
        return vb.getResult();
    }

    @Nullable
    public GdValidationResult getValidationResult(ValidationResult<?, Defect> vr, Path path) {
        return GridValidationResultConversionService
                .buildGridValidationResultIfErrors(vr, path, pathNodeConverterProvider);
    }
}
