package ru.yandex.direct.grid.processing.service.group.validation;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.container.UntypedAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.McBannerAdGroup;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.processing.model.api.GdValidationResult;
import ru.yandex.direct.grid.processing.model.group.mutation.GdAddMcBannerAdGroup;
import ru.yandex.direct.grid.processing.model.group.mutation.GdAddMcBannerAdGroupItem;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateMcBannerAdGroup;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateMcBannerAdGroupItem;
import ru.yandex.direct.grid.processing.service.validation.GridDefectDefinitions;
import ru.yandex.direct.grid.processing.service.validation.GridValidationResultConversionService;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListConstraint;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.defect.CollectionDefects;
import ru.yandex.direct.validation.result.DefaultPathNodeConverterProvider;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.Path;
import ru.yandex.direct.validation.result.PathNodeConverterProvider;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.grid.processing.service.validation.presentation.AdGroupConverters.UPDATE_AD_GROUP_PATH_CONVERTER;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.notEmptyCollection;
import static ru.yandex.direct.validation.constraint.CommonConstraints.eachNotNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.constraint.NumberConstraints.inRange;

@Service
@ParametersAreNonnullByDefault
public class McBannerAdGroupValidationService {
    private final GridValidationService gridValidationService;
    private final PathNodeConverterProvider pathNodeConverterProvider;
    private final ClientService clientService;

    @Autowired
    public McBannerAdGroupValidationService(GridValidationService gridValidationService, ClientService clientService) {
        this.gridValidationService = gridValidationService;
        this.clientService = clientService;
        this.pathNodeConverterProvider = DefaultPathNodeConverterProvider.builder()
                .register(UntypedAdGroup.class, UPDATE_AD_GROUP_PATH_CONVERTER)
                .register(McBannerAdGroup.class, UPDATE_AD_GROUP_PATH_CONVERTER)
                .build();
    }

    //region TODO(darkkeks) DIRECT-130906 -- Убрать после полного перехода фронта на честное создание mcbanner групп
    public void validatePartialAddMcBannerAdGroups(GdAddMcBannerAdGroup gdAddRequest) {
        gridValidationService.applyValidator(AD_GROUP_PARTIAL_ADD_VALIDATOR, gdAddRequest, false);
    }
    //endregion

    /**
     * Первичная валидация входных параметров
     */
    public void validateAddMcBannerAdGroups(GdAddMcBannerAdGroup gdAddRequest) {
        gridValidationService.applyValidator(AD_GROUP_ADD_VALIDATOR, gdAddRequest, false);
    }

    /**
     * Расширенная валидация, требующая чтения дополнительных данных из БД
     */
    public void additionalValidationOfAddAdGroup(ClientId clientId, GdAddMcBannerAdGroup gdAddMcBannerAdGroup) {
        AdGroupAddValidator adGroupAddValidator = new AdGroupAddValidator(clientService.getWorkCurrency(clientId));
        gridValidationService.applyValidator(adGroupAddValidator, gdAddMcBannerAdGroup, false);
    }

    /**
     * Первичная валидация входных параметров
     *
     * @param adGroupsUpdateRequest запрос на обновление групп
     */
    public void validateUpdateAdGroups(GdUpdateMcBannerAdGroup adGroupsUpdateRequest) {
        gridValidationService.applyValidator(AD_GROUP_UPDATE_VALIDATOR, adGroupsUpdateRequest, false);
    }

    /**
     * Расширенная валидация, требующая чтения дополнительных данных из БД
     *
     * @param clientId              идентификатор клиента
     * @param adGroupTypeMap        мапа соответсвия идентификаторов групп и типов групп
     * @param adGroupsUpdateRequest запрос на обновление групп
     */
    public void additionalValidationOfUpdateAdGroup(ClientId clientId, Map<Long, AdGroupType> adGroupTypeMap,
                                                    GdUpdateMcBannerAdGroup adGroupsUpdateRequest) {
        AdGroupUpdateValidator adGroupUpdateValidator =
                new AdGroupUpdateValidator(adGroupTypeMap, clientService.getWorkCurrency(clientId));
        gridValidationService.applyValidator(adGroupUpdateValidator, adGroupsUpdateRequest, false);
    }

    @Nullable
    public GdValidationResult getValidationResult(ValidationResult<?, Defect> vr, Path path) {
        return GridValidationResultConversionService
                .buildGridValidationResultIfErrors(vr, path, pathNodeConverterProvider);
    }

    //region TODO(darkkeks) DIRECT-130906 -- Убрать после полного перехода фронта на честное создание mcbanner групп
    private static final Validator<GdAddMcBannerAdGroupItem, Defect> AD_GROUP_ADD_PARTIAL_ITEMS_VALIDATOR = items -> {
        ModelItemValidationBuilder<GdAddMcBannerAdGroupItem> vb = ModelItemValidationBuilder.of(items);
        vb.item(GdAddMcBannerAdGroupItem.CAMPAIGN_ID)
                .check(validId(), When.isValid());
        return vb.getResult();
    };

    private static final Validator<GdAddMcBannerAdGroup, Defect> AD_GROUP_PARTIAL_ADD_VALIDATOR = gdAddMcBannerAdGroup -> {
        ModelItemValidationBuilder<GdAddMcBannerAdGroup> vb = ModelItemValidationBuilder.of(gdAddMcBannerAdGroup);
        vb.list(GdAddMcBannerAdGroup.ADD_ITEMS)
                .check(notEmptyCollection(), When.notNull())
                .checkEachBy(AD_GROUP_ADD_PARTIAL_ITEMS_VALIDATOR);
        return vb.getResult();
    };
    //endregion

    private static final Validator<GdAddMcBannerAdGroupItem, Defect> AD_GROUP_ADD_ITEMS_VALIDATOR = items -> {
        ModelItemValidationBuilder<GdAddMcBannerAdGroupItem> vb = ModelItemValidationBuilder.of(items);

        vb.item(GdAddMcBannerAdGroupItem.CAMPAIGN_ID)
                .check(validId(), When.isValid());
        vb.item(GdAddMcBannerAdGroupItem.AD_GROUP_MINUS_KEYWORDS)
                .check(eachNotNull());
        vb.list(GdAddMcBannerAdGroupItem.LIBRARY_MINUS_KEYWORDS_IDS)
                .checkEach(validId());
        vb.item(GdAddMcBannerAdGroupItem.KEYWORDS)
                .check(eachNotNull());
        return vb.getResult();
    };

    private static final Validator<GdAddMcBannerAdGroup, Defect> AD_GROUP_ADD_VALIDATOR = gdAddMcBannerAdGroup -> {
        ModelItemValidationBuilder<GdAddMcBannerAdGroup> vb = ModelItemValidationBuilder.of(gdAddMcBannerAdGroup);
        vb.list(GdAddMcBannerAdGroup.ADD_ITEMS)
                .check(notEmptyCollection(), When.notNull())
                .check(checkAddKeywordsUniformity())
                .checkEachBy(AD_GROUP_ADD_ITEMS_VALIDATOR);
        return vb.getResult();
    };

    private static final Validator<GdUpdateMcBannerAdGroupItem, Defect> AD_GROUP_UPDATE_ITEMS_VALIDATOR = items -> {
        ModelItemValidationBuilder<GdUpdateMcBannerAdGroupItem> vb = ModelItemValidationBuilder.of(items);

        vb.item(GdUpdateMcBannerAdGroupItem.AD_GROUP_ID)
                .check(validId());
        vb.item(GdUpdateMcBannerAdGroupItem.AD_GROUP_MINUS_KEYWORDS)
                .check(eachNotNull());
        vb.list(GdUpdateMcBannerAdGroupItem.LIBRARY_MINUS_KEYWORDS_IDS)
                .checkEach(validId());
        vb.item(GdUpdateMcBannerAdGroupItem.KEYWORDS)
                .check(eachNotNull());
        return vb.getResult();
    };

    private static final Validator<GdUpdateMcBannerAdGroup, Defect> AD_GROUP_UPDATE_VALIDATOR = req -> {
        ModelItemValidationBuilder<GdUpdateMcBannerAdGroup> vb = ModelItemValidationBuilder.of(req);

        vb.list(GdUpdateMcBannerAdGroup.UPDATE_ITEMS)
                .check(notEmptyCollection(), When.notNull())
                .check(checkUpdateKeywordsUniformity())
                .checkEachBy(AD_GROUP_UPDATE_ITEMS_VALIDATOR);
        return vb.getResult();
    };

    private static Constraint<List<GdAddMcBannerAdGroupItem>, Defect> checkAddKeywordsUniformity() {
        return Constraint.fromPredicate(
                list -> list.stream().allMatch(item -> item.getKeywords() != null) ||
                        list.stream().allMatch(item -> item.getKeywords() == null), CollectionDefects.notContainNulls());
    }

    private static Constraint<List<GdUpdateMcBannerAdGroupItem>, Defect> checkUpdateKeywordsUniformity() {
        return Constraint.fromPredicate(
                list -> list.stream().allMatch(item -> item.getKeywords() != null) ||
                        list.stream().allMatch(item -> item.getKeywords() == null), CollectionDefects.notContainNulls());
    }

    private static class AdGroupAddItemsValidator implements Validator<GdAddMcBannerAdGroupItem, Defect> {
        private final Currency clientWorkCurrency;

        AdGroupAddItemsValidator(Currency clientWorkCurrency) {
            this.clientWorkCurrency = clientWorkCurrency;
        }

        @Override
        public ValidationResult<GdAddMcBannerAdGroupItem, Defect> apply(
                GdAddMcBannerAdGroupItem gdAddMcBannerAdGroupItem) {
            ModelItemValidationBuilder<GdAddMcBannerAdGroupItem> vb =
                    ModelItemValidationBuilder.of(gdAddMcBannerAdGroupItem);
            vb.item(GdAddMcBannerAdGroupItem.GENERAL_PRICE)
                    .check(inRange(clientWorkCurrency.getMinPrice(), clientWorkCurrency.getMaxPrice()), When.notNull());
            return vb.getResult();
        }
    }

    private static class AdGroupAddValidator implements Validator<GdAddMcBannerAdGroup, Defect> {
        private final AdGroupAddItemsValidator adGroupAddItemsValidator;

        AdGroupAddValidator(Currency clientWorkCurrency) {
            this.adGroupAddItemsValidator = new AdGroupAddItemsValidator(clientWorkCurrency);
        }

        @Override
        public ValidationResult<GdAddMcBannerAdGroup, Defect> apply(
                GdAddMcBannerAdGroup gdAdGroupUpdateInput) {
            ModelItemValidationBuilder<GdAddMcBannerAdGroup> vb =
                    ModelItemValidationBuilder.of(gdAdGroupUpdateInput);
            vb.list(GdAddMcBannerAdGroup.ADD_ITEMS)
                    .checkEachBy(adGroupAddItemsValidator);
            return vb.getResult();
        }
    }

    private static class AdGroupUpdateItemsValidator implements Validator<GdUpdateMcBannerAdGroupItem, Defect> {
        private final Currency clientWorkCurrency;

        AdGroupUpdateItemsValidator(Currency clientWorkCurrency) {
            this.clientWorkCurrency = clientWorkCurrency;
        }

        @Override
        public ValidationResult<GdUpdateMcBannerAdGroupItem, Defect> apply(
                GdUpdateMcBannerAdGroupItem gdUpdateMcBannerAdGroupItem) {
            ModelItemValidationBuilder<GdUpdateMcBannerAdGroupItem> vb =
                    ModelItemValidationBuilder.of(gdUpdateMcBannerAdGroupItem);
            vb.item(GdUpdateMcBannerAdGroupItem.GENERAL_PRICE)
                    .check(inRange(clientWorkCurrency.getMinPrice(), clientWorkCurrency.getMaxPrice()), When.notNull());
            return vb.getResult();
        }
    }

    private static class AdGroupUpdateValidator implements Validator<GdUpdateMcBannerAdGroup, Defect> {
        private final Map<Long, AdGroupType> adGroupTypeMap;
        private final AdGroupUpdateItemsValidator adGroupUpdateItemsValidator;

        AdGroupUpdateValidator(Map<Long, AdGroupType> adGroupTypeMap, Currency clientWorkCurrency) {
            this.adGroupTypeMap = adGroupTypeMap;
            this.adGroupUpdateItemsValidator = new AdGroupUpdateItemsValidator(clientWorkCurrency);
        }

        @Override
        public ValidationResult<GdUpdateMcBannerAdGroup, Defect> apply(
                GdUpdateMcBannerAdGroup gdAdGroupUpdateInput) {
            ModelItemValidationBuilder<GdUpdateMcBannerAdGroup> vb =
                    ModelItemValidationBuilder.of(gdAdGroupUpdateInput);
            vb.list(GdUpdateMcBannerAdGroup.UPDATE_ITEMS)
                    .checkEach(isMcBannerAdGroups())
                    .checkEachBy(adGroupUpdateItemsValidator);
            return vb.getResult();
        }

        private ListConstraint<GdUpdateMcBannerAdGroupItem, Defect> isMcBannerAdGroups() {
            return adGroupUpdateItems -> {
                Map<Integer, Defect> result = new HashMap<>();
                for (int i = 0; i < adGroupUpdateItems.size(); i++) {
                    if (!AdGroupType.MCBANNER.equals(adGroupTypeMap.get(adGroupUpdateItems.get(i).getAdGroupId()))) {
                        result.put(i, GridDefectDefinitions.unsupportedAdGroupType());
                    }
                }
                return result;
            };
        }
    }
}
