package ru.yandex.direct.grid.processing.service.group.validation;

import java.util.HashMap;
import java.util.Map;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.container.UntypedAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.MobileContentAdGroup;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.retargeting.model.TargetingCategory;
import ru.yandex.direct.core.entity.retargeting.repository.TargetingCategoriesCache;
import ru.yandex.direct.core.entity.retargeting.service.validation2.RetargetingDefects;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.processing.model.api.GdValidationResult;
import ru.yandex.direct.grid.processing.model.group.mutation.GdAddMobileContentAdGroup;
import ru.yandex.direct.grid.processing.model.group.mutation.GdAddMobileContentAdGroupItem;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateAdGroupInterestItem;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateMobileContentAdGroup;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateMobileContentAdGroupItem;
import ru.yandex.direct.grid.processing.service.validation.GridDefectDefinitions;
import ru.yandex.direct.grid.processing.service.validation.GridValidationResultConversionService;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;
import ru.yandex.direct.validation.builder.ListConstraint;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.DefaultPathNodeConverterProvider;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.Path;
import ru.yandex.direct.validation.result.PathNodeConverterProvider;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.grid.processing.service.validation.presentation.AdGroupConverters.MOBILE_CONTENT_AD_GROUP_PATH_CONVERTER;
import static ru.yandex.direct.grid.processing.service.validation.presentation.AdGroupConverters.UPDATE_AD_GROUP_PATH_CONVERTER;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.notEmptyCollection;
import static ru.yandex.direct.validation.constraint.CommonConstraints.eachNotNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.constraint.NumberConstraints.inRange;

@Service
@ParametersAreNonnullByDefault
public class MobileContentAdGroupValidationService {
    private GridValidationService gridValidationService;
    private final PathNodeConverterProvider pathNodeConverterProvider;
    private final ClientService clientService;
    private final TargetingCategoriesCache targetingCategoriesCache;

    @Autowired
    public MobileContentAdGroupValidationService(GridValidationService gridValidationService,
                                                 TargetingCategoriesCache targetingCategoriesCache,
                                                 ClientService clientService) {
        this.gridValidationService = gridValidationService;
        this.clientService = clientService;
        this.pathNodeConverterProvider = DefaultPathNodeConverterProvider.builder()
                .register(UntypedAdGroup.class, UPDATE_AD_GROUP_PATH_CONVERTER)
                .register(MobileContentAdGroup.class, MOBILE_CONTENT_AD_GROUP_PATH_CONVERTER)
                .build();
        this.targetingCategoriesCache = targetingCategoriesCache;
    }

    /**
     * Первичная валидация входных параметров
     */
    public void validateAddAdGroups(GdAddMobileContentAdGroup gdAddRequest) {
        gridValidationService.applyValidator(this::validateAddRequest, gdAddRequest, false);
    }

    public void validateUpdateAdGroups(GdUpdateMobileContentAdGroup gdUpdateRequest) {
        gridValidationService.applyValidator(this::validateUpdateRequest, gdUpdateRequest, false);
    }

    /**
     * Расширенная валидация, требующая чтения дополнительных данных из БД
     *
     * @param clientId              идентификатор клиента
     * @param adGroupTypeMap        мапа соответсвия идентификаторов групп и типов групп
     * @param adGroupsUpdateRequest запрос на обновление групп
     */
    public void additionalValidationOfUpdateAdGroup(ClientId clientId, Map<Long, AdGroupType> adGroupTypeMap,
                                                    GdUpdateMobileContentAdGroup adGroupsUpdateRequest) {
        MobileContentAdGroupValidationService.AdGroupValidator adGroupValidator =
                new MobileContentAdGroupValidationService
                        .AdGroupValidator(adGroupTypeMap, clientService.getWorkCurrency(clientId));
        gridValidationService.applyValidator(adGroupValidator, adGroupsUpdateRequest, false);
    }

    @Nullable
    public GdValidationResult getValidationResult(ValidationResult<?, Defect> vr, Path path) {
        return GridValidationResultConversionService
                .buildGridValidationResultIfErrors(vr, path, pathNodeConverterProvider);
    }

    private static final Validator<GdAddMobileContentAdGroupItem, Defect> AD_GROUP_ADD_ITEMS_VALIDATOR =
            items -> {
                ModelItemValidationBuilder<GdAddMobileContentAdGroupItem> vb = ModelItemValidationBuilder.of(items);
                vb.item(GdAddMobileContentAdGroupItem.CAMPAIGN_ID)
                        .check(validId(), When.isValid());
                vb.item(GdAddMobileContentAdGroupItem.AD_GROUP_MINUS_KEYWORDS)
                        .check(eachNotNull());
                vb.list(GdAddMobileContentAdGroupItem.LIBRARY_MINUS_KEYWORDS_IDS)
                        .checkEach(validId());
                vb.item(GdAddMobileContentAdGroupItem.KEYWORDS)
                        .check(eachNotNull());
                vb.item(GdAddMobileContentAdGroupItem.BID_MODIFIERS)
                        .check(notNull());
                vb.item(GdAddMobileContentAdGroupItem.RETARGETINGS)
                        .check(eachNotNull());
                vb.list(GdAddMobileContentAdGroupItem.RETARGETINGS)
                        .checkEachBy(AdGroupValidationUtils.RETARGETING_ITEM_VALIDATOR);
                return vb.getResult();
            };

    private static final Validator<GdUpdateMobileContentAdGroupItem, Defect> AD_GROUP_UPDATE_ITEMS_VALIDATOR =
            items -> {
                ModelItemValidationBuilder<GdUpdateMobileContentAdGroupItem> vb = ModelItemValidationBuilder.of(items);

                vb.item(GdUpdateMobileContentAdGroupItem.AD_GROUP_ID)
                        .check(validId());
                vb.item(GdUpdateMobileContentAdGroupItem.AD_GROUP_MINUS_KEYWORDS)
                        .check(eachNotNull());
                vb.list(GdUpdateMobileContentAdGroupItem.LIBRARY_MINUS_KEYWORDS_IDS)
                        .checkEach(validId());
                vb.item(GdUpdateMobileContentAdGroupItem.KEYWORDS)
                        .check(eachNotNull());
                vb.item(GdUpdateMobileContentAdGroupItem.RETARGETINGS)
                        .check(eachNotNull());
                vb.list(GdUpdateMobileContentAdGroupItem.RETARGETINGS)
                        .checkEachBy(AdGroupValidationUtils.RETARGETING_ITEM_VALIDATOR);
                return vb.getResult();
            };

    private ValidationResult<GdAddMobileContentAdGroup, Defect> validateAddRequest(
            GdAddMobileContentAdGroup gdAddMobileContentAdGroup) {

        Map<Long, TargetingCategory> targetingCategoryByCategoryId = getTargetingCategoryByCategoryId();

        ModelItemValidationBuilder<GdAddMobileContentAdGroup> vb = ModelItemValidationBuilder
                .of(gdAddMobileContentAdGroup);
        vb.list(GdAddMobileContentAdGroup.ADD_ITEMS)
                .check(notEmptyCollection(), When.notNull())
                .check(AdGroupValidationUtils.checkKeywordsUniformityForAddMobileContent())
                .check(AdGroupValidationUtils.checkRelevanceMatchUniformityForAddMobileContent())
                .checkEachBy(AD_GROUP_ADD_ITEMS_VALIDATOR)
                .checkEachBy(item -> validateAddItem(item, targetingCategoryByCategoryId));
        return vb.getResult();
    }

    private ValidationResult<GdUpdateMobileContentAdGroup, Defect> validateUpdateRequest(
            GdUpdateMobileContentAdGroup gdAddMobileContentAdGroup) {

        Map<Long, TargetingCategory> targetingCategoryByCategoryId = getTargetingCategoryByCategoryId();

        ModelItemValidationBuilder<GdUpdateMobileContentAdGroup> vb = ModelItemValidationBuilder
                .of(gdAddMobileContentAdGroup);
        vb.list(GdUpdateMobileContentAdGroup.UPDATE_ITEMS)
                .check(notEmptyCollection(), When.notNull())
                .check(AdGroupValidationUtils.checkKeywordsUniformityForUpdateMobileContent())
                .check(AdGroupValidationUtils.checkRelevanceMatchUniformityForUpdateMobileContent())
                .checkEachBy(AD_GROUP_UPDATE_ITEMS_VALIDATOR)
                .checkEachBy(item -> validateUpdateItem(item, targetingCategoryByCategoryId));
        return vb.getResult();
    }

    private ValidationResult<GdAddMobileContentAdGroupItem, Defect> validateAddItem(
            GdAddMobileContentAdGroupItem item,
            Map<Long, TargetingCategory> targetingCategoryByCategoryId) {
        ModelItemValidationBuilder<GdAddMobileContentAdGroupItem> vb = ModelItemValidationBuilder.of(item);
        vb.list(GdAddMobileContentAdGroupItem.INTERESTS)
                .checkEachBy(interest -> validateInterestItem(interest, targetingCategoryByCategoryId));
        return vb.getResult();
    }

    private ValidationResult<GdUpdateMobileContentAdGroupItem, Defect> validateUpdateItem(
            GdUpdateMobileContentAdGroupItem item,
            Map<Long, TargetingCategory> targetingCategoryByCategoryId) {
        ModelItemValidationBuilder<GdUpdateMobileContentAdGroupItem> vb = ModelItemValidationBuilder.of(item);
        vb.list(GdUpdateMobileContentAdGroupItem.INTERESTS)
                .checkEachBy(interest -> validateInterestItem(interest, targetingCategoryByCategoryId));
        return vb.getResult();
    }

    /**
     * Проверка, что условие нацеливания по интересам можно использовать.
     * Доступны только те категории, которые являются листьями (не имеют дочерних категорий)
     */
    private ValidationResult<GdUpdateAdGroupInterestItem, Defect> validateInterestItem(
            GdUpdateAdGroupInterestItem item,
            Map<Long, TargetingCategory> targetingCategoryByCategoryId) {
        ModelItemValidationBuilder<GdUpdateAdGroupInterestItem> vb = ModelItemValidationBuilder.of(item);
        vb.item(GdUpdateAdGroupInterestItem.CATEGORY_ID)
                .check(notNull())
                .check(fromPredicate(categoryId -> targetingCategoryByCategoryId.containsKey(categoryId)
                                && targetingCategoryByCategoryId.get(categoryId).isAvailable(),
                        RetargetingDefects.inconsistentStateTargetingCategoryUnavailable()), When.isValid());
        return vb.getResult();
    }

    private Map<Long, TargetingCategory> getTargetingCategoryByCategoryId() {
        return StreamEx.of(targetingCategoriesCache.getTargetingCategories())
                .mapToEntry(TargetingCategory::getTargetingCategoryId)
                .invert()
                .toMap();
    }

    private class AdGroupItemsValidator implements Validator<GdUpdateMobileContentAdGroupItem, Defect> {

        private final Currency clientWorkCurrency;

        AdGroupItemsValidator(Currency clientWorkCurrency) {
            this.clientWorkCurrency = clientWorkCurrency;
        }

        @Override
        public ValidationResult<GdUpdateMobileContentAdGroupItem, Defect> apply(
                GdUpdateMobileContentAdGroupItem gdUpdateMobileContentAdGroupItem) {
            ModelItemValidationBuilder<GdUpdateMobileContentAdGroupItem> vb =
                    ModelItemValidationBuilder.of(gdUpdateMobileContentAdGroupItem);
            vb.item(GdUpdateMobileContentAdGroupItem.GENERAL_PRICE)
                    .check(inRange(clientWorkCurrency.getMinPrice(), clientWorkCurrency.getMaxPrice()), When.notNull());
            return vb.getResult();
        }
    }

    private class AdGroupValidator implements Validator<GdUpdateMobileContentAdGroup, Defect> {

        private final Map<Long, AdGroupType> adGroupTypeMap;
        private final MobileContentAdGroupValidationService.AdGroupItemsValidator adGroupItemsValidator;

        AdGroupValidator(Map<Long, AdGroupType> adGroupTypeMap, Currency clientWorkCurrency) {
            this.adGroupTypeMap = adGroupTypeMap;
            this.adGroupItemsValidator =
                    new MobileContentAdGroupValidationService.AdGroupItemsValidator(clientWorkCurrency);
        }

        @Override
        public ValidationResult<GdUpdateMobileContentAdGroup, Defect> apply(
                GdUpdateMobileContentAdGroup gdAdGroupUpdateInput) {
            ModelItemValidationBuilder<GdUpdateMobileContentAdGroup> vb =
                    ModelItemValidationBuilder.of(gdAdGroupUpdateInput);
            vb.list(GdUpdateMobileContentAdGroup.UPDATE_ITEMS)
                    .checkEach(isMobileContentAdGroups())
                    .checkEachBy(adGroupItemsValidator);
            return vb.getResult();
        }

        private ListConstraint<GdUpdateMobileContentAdGroupItem, Defect> isMobileContentAdGroups() {
            return adGroupUpdateItems -> {
                Map<Integer, Defect> result = new HashMap<>();
                for (int i = 0; i < adGroupUpdateItems.size(); i++) {
                    if (!AdGroupType.MOBILE_CONTENT.equals(adGroupTypeMap.get(adGroupUpdateItems.get(i).getAdGroupId()))) {
                        result.put(i, GridDefectDefinitions.unsupportedAdGroupType());
                    }
                }
                return result;
            };
        }
    }
}
