package ru.yandex.direct.grid.processing.service.group.validation;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.container.UntypedAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.TextAdGroup;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.feed.model.BusinessType;
import ru.yandex.direct.core.entity.feed.model.Feed;
import ru.yandex.direct.core.entity.feed.service.FeedService;
import ru.yandex.direct.core.entity.feed.validation.FeedDefects;
import ru.yandex.direct.core.entity.feedfilter.model.FeedFilter;
import ru.yandex.direct.core.entity.feedfilter.model.FeedFilterCondition;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.processing.model.api.GdValidationResult;
import ru.yandex.direct.grid.processing.model.group.mutation.GdAddAdGroupDynamicFeedAdTargetItem;
import ru.yandex.direct.grid.processing.model.group.mutation.GdAddTextAdGroup;
import ru.yandex.direct.grid.processing.model.group.mutation.GdAddTextAdGroupItem;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateAdGroupOfferRetargetingItem;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateAdGroupRelevanceMatchItem;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateTextAdGroup;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateTextAdGroupItem;
import ru.yandex.direct.grid.processing.service.validation.GridDefectDefinitions;
import ru.yandex.direct.grid.processing.service.validation.GridValidationResultConversionService;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;
import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListConstraint;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.defect.CollectionDefects;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.DefaultPathNodeConverterProvider;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.Path;
import ru.yandex.direct.validation.result.PathNodeConverterProvider;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.feature.FeatureName.ALLOW_USE_FEEDS_IN_TGO_GROUPS;
import static ru.yandex.direct.feature.FeatureName.CUSTOM_AUDIENCE_ENABLED;
import static ru.yandex.direct.feature.FeatureName.ENABLED_DYNAMIC_FEED_AD_TARGET_IN_TEXT_AD_GROUP;
import static ru.yandex.direct.feature.FeatureName.ENABLE_OFFER_RETARGETINGS_IN_TEXT_AD_GROUP;
import static ru.yandex.direct.grid.processing.service.validation.presentation.AdGroupConverters.FEED_FILTER_CONDITION_CONVERTER;
import static ru.yandex.direct.grid.processing.service.validation.presentation.AdGroupConverters.FEED_FILTER_CONVERTER;
import static ru.yandex.direct.grid.processing.service.validation.presentation.AdGroupConverters.TEMP_UPDATE_AD_GROUP_PATH_CONVERTER;
import static ru.yandex.direct.grid.processing.service.validation.presentation.AdGroupConverters.UPDATE_AD_GROUP_PATH_CONVERTER;
import static ru.yandex.direct.grid.processing.service.validation.presentation.AdGroupConverters.UPDATE_TEXT_AD_GROUP_PATH_CONVERTER;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.notEmptyCollection;
import static ru.yandex.direct.validation.constraint.CommonConstraints.eachNotNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.isNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.constraint.NumberConstraints.inRange;

@Service
@ParametersAreNonnullByDefault
public class TextAdGroupValidationService {
    private final GridValidationService gridValidationService;
    private final PathNodeConverterProvider pathNodeConverterProvider;
    private final ClientService clientService;
    private final FeedService feedService;
    private final FeatureService featureService;

    @Autowired
    public TextAdGroupValidationService(GridValidationService gridValidationService, ClientService clientService,
                                        FeedService feedService, FeatureService featureService) {
        this.gridValidationService = gridValidationService;
        this.clientService = clientService;
        this.feedService = feedService;
        this.featureService = featureService;
        this.pathNodeConverterProvider = DefaultPathNodeConverterProvider.builder()
                .register(UntypedAdGroup.class, UPDATE_AD_GROUP_PATH_CONVERTER)
                .register(TextAdGroup.class, UPDATE_TEXT_AD_GROUP_PATH_CONVERTER)
                .register(FeedFilter.class, FEED_FILTER_CONVERTER)
                .register(FeedFilterCondition.class, FEED_FILTER_CONDITION_CONVERTER)
                .register(AdGroup.class, TEMP_UPDATE_AD_GROUP_PATH_CONVERTER)
                .build();
    }

    /**
     * Первичная валидация входных параметров
     *
     * @param adGroupsUpdateRequest запрос на обновление групп
     */
    public void validateUpdateRequest(GdUpdateTextAdGroup adGroupsUpdateRequest, ClientId clientId) {
        gridValidationService.applyValidator(gdUpdateTextAdGroup -> validateUpdateAdGroups(gdUpdateTextAdGroup,
                        clientId),
                adGroupsUpdateRequest, false);
    }

    private ValidationResult<GdUpdateTextAdGroup, Defect> validateUpdateAdGroups(GdUpdateTextAdGroup adGroupsUpdateRequest,
                                                                                 ClientId clientId) {
        ModelItemValidationBuilder<GdUpdateTextAdGroup> vb = ModelItemValidationBuilder.of(adGroupsUpdateRequest);

        boolean tgoFeedsEnabled = featureService.isEnabledForClientId(clientId, ALLOW_USE_FEEDS_IN_TGO_GROUPS);
        boolean isCustomAudienceEnabled = featureService.isEnabledForClientId(clientId, CUSTOM_AUDIENCE_ENABLED);
        boolean offerRetargetingsEnabled = featureService.isEnabledForClientId(clientId,
                ENABLE_OFFER_RETARGETINGS_IN_TEXT_AD_GROUP);

        vb.list(GdUpdateTextAdGroup.UPDATE_ITEMS)
                .check(notEmptyCollection(), When.notNull())
                .check(checkKeywordsUniformity())
                .check(checkRelevanceMatchUniformity())
                .check(checkOfferRetargetingsUniformity(offerRetargetingsEnabled))
                .checkEachBy(updateTextAdGroupItem -> validateUpdateAdGroup(updateTextAdGroupItem,
                        tgoFeedsEnabled, isCustomAudienceEnabled));

        return vb.getResult();
    }

    private ValidationResult<GdUpdateTextAdGroupItem, Defect> validateUpdateAdGroup(GdUpdateTextAdGroupItem updateTextAdGroupItem,
                                                                                    boolean tgoFeedsAllowed,
                                                                                    boolean isCustomAudienceEnabled) {
        ModelItemValidationBuilder<GdUpdateTextAdGroupItem> vb = ModelItemValidationBuilder.of(updateTextAdGroupItem);

        vb.item(GdUpdateTextAdGroupItem.AD_GROUP_ID)
                .check(validId());
        vb.item(GdUpdateTextAdGroupItem.AD_GROUP_MINUS_KEYWORDS)
                .check(eachNotNull());
        vb.list(GdUpdateTextAdGroupItem.LIBRARY_MINUS_KEYWORDS_IDS)
                .checkEach(validId());
        vb.item(GdUpdateTextAdGroupItem.KEYWORDS)
                .check(eachNotNull());
        vb.item(GdUpdateTextAdGroupItem.FEED_ID)
                .check(isNull(), When.isFalse(tgoFeedsAllowed));
        vb.item(GdUpdateTextAdGroupItem.FEED_ID)
                .check(validId(), When.notNull());
        vb.item(GdUpdateTextAdGroupItem.SMART_FEED_FILTER)
                .check(isNull(), When.isTrue(updateTextAdGroupItem.getSmartFeedId() == null));
        vb.item(GdUpdateTextAdGroupItem.FEED_FILTER_CATEGORIES)
                .check(isNull(), When.isFalse(tgoFeedsAllowed));
        vb.item(GdUpdateTextAdGroupItem.RETARGETINGS)
                .check(eachNotNull());
        vb.list(GdUpdateTextAdGroupItem.RETARGETINGS)
                .checkEachBy(AdGroupValidationUtils.RETARGETING_ITEM_VALIDATOR);
        vb.item(GdUpdateTextAdGroupItem.CA_RETARGETING_CONDITION)
                .checkBy(
                        AdGroupValidationUtils.CA_RETARGETING_ITEM_VALIDATOR,
                        When.notNullAnd(When.isTrue(isCustomAudienceEnabled))
                );

        return vb.getResult();
    }

    /**
     * Расширенная валидация, требующая чтения дополнительных данных из БД
     *
     * @param clientId              идентификатор клиента
     * @param adGroupTypeMap        мапа соответсвия идентификаторов групп и типов групп
     * @param adGroupsUpdateRequest запрос на обновление групп
     */
    public void additionalValidationOfUpdateAdGroup(ClientId clientId, Map<Long, AdGroupType> adGroupTypeMap,
                                                    GdUpdateTextAdGroup adGroupsUpdateRequest) {
        var feedIds = StreamEx.of(adGroupsUpdateRequest.getUpdateItems())
                .flatMap(updateItem -> Stream.of(updateItem.getFeedId(), updateItem.getSmartFeedId()))
                .nonNull()
                .toList();
        var feedsByFeedId = listToMap(feedService.getFeeds(clientId, feedIds), Feed::getId);

        AdGroupValidator adGroupValidator =
                new AdGroupValidator(adGroupTypeMap, clientService.getWorkCurrency(clientId), feedsByFeedId);
        gridValidationService.applyValidator(adGroupValidator, adGroupsUpdateRequest, false);
    }

    public void additionalValidationOfAddAdGroup(ClientId clientId, GdAddTextAdGroup adGroupsAddRequest) {
        var feedIds = StreamEx.of(adGroupsAddRequest.getAddItems())
                .flatMap(addItem -> Stream.of(addItem.getFeedId(), addItem.getSmartFeedId()))
                .nonNull()
                .toList();
        var feedsByFeedId = listToMap(feedService.getFeeds(clientId, feedIds), Feed::getId);

        var validator = new AdGroupAddValidator(clientId, feedsByFeedId);
        gridValidationService.applyValidator(validator, adGroupsAddRequest, false);
    }

    public void validateCreateTextAdGroups(ClientId clientId, GdAddTextAdGroup gdAddRequest) {
        var isCustomAudienceEnabled = featureService.isEnabledForClientId(clientId, CUSTOM_AUDIENCE_ENABLED);
        var isDynamicFeedAdTargetEnabled =
                featureService.isEnabledForClientId(clientId, ENABLED_DYNAMIC_FEED_AD_TARGET_IN_TEXT_AD_GROUP);
        var offerRetargetingIsEnabled =
                featureService.isEnabledForClientId(clientId, ENABLE_OFFER_RETARGETINGS_IN_TEXT_AD_GROUP);
        var validator = new AdGroupCreateValidator(isCustomAudienceEnabled, isDynamicFeedAdTargetEnabled, offerRetargetingIsEnabled);

        gridValidationService.applyValidator(validator, gdAddRequest, false);
    }

    @Nullable
    public GdValidationResult getValidationResult(ValidationResult<?, Defect> vr, Path path) {
        return GridValidationResultConversionService
                .buildGridValidationResultIfErrors(vr, path, pathNodeConverterProvider);
    }

    private class AdGroupCreateValidator implements Validator<GdAddTextAdGroup, Defect> {

        private boolean isCustomAudienceEnabled;
        private boolean isDynamicFeedAdTargetEnabled;
        private boolean isOfferRetargetingsEnabled;

        AdGroupCreateValidator(boolean isCustomAudienceEnabled,
                               boolean isDynamicFeedAdGroupEnabled,
                               boolean isOfferRetargetingsEnabled) {
            this.isCustomAudienceEnabled = isCustomAudienceEnabled;
            this.isDynamicFeedAdTargetEnabled = isDynamicFeedAdGroupEnabled;
            this.isOfferRetargetingsEnabled = isOfferRetargetingsEnabled;
        }

        @Override
        public ValidationResult<GdAddTextAdGroup, Defect> apply(GdAddTextAdGroup gdAddTextAdGroup) {
            ModelItemValidationBuilder<GdAddTextAdGroup> vb = ModelItemValidationBuilder.of(gdAddTextAdGroup);

            var itemsValidator = new AdGroupCreateItemsValidator(isCustomAudienceEnabled);

            vb.list(GdAddTextAdGroup.ADD_ITEMS)
                    .check(notEmptyCollection(), When.notNull())
                    .check(checkKeywords())
                    .check(checkRelevanceMatch())
                    .check(checkDynamicFeedAdTargets(isDynamicFeedAdTargetEnabled))
                    .check(checkOfferRetargetings(isOfferRetargetingsEnabled))
                    .checkEachBy(itemsValidator);
            return vb.getResult();
        }
    }

    private class AdGroupCreateItemsValidator implements Validator<GdAddTextAdGroupItem, Defect> {

        private boolean isCustomAudienceEnabled;

        AdGroupCreateItemsValidator(boolean isCustomAudienceEnabled) {
            this.isCustomAudienceEnabled = isCustomAudienceEnabled;
        }

        @Override
        public ValidationResult<GdAddTextAdGroupItem, Defect> apply(GdAddTextAdGroupItem item) {
            ModelItemValidationBuilder<GdAddTextAdGroupItem> vb = ModelItemValidationBuilder.of(item);

            boolean isHyperLocalAdGroup = item.getHyperGeoId() != null;
            vb.item(GdAddTextAdGroupItem.REGION_IDS)
                    .check(notNull(), When.isFalse(isHyperLocalAdGroup))
                    .check(isNull(), When.isTrue(isHyperLocalAdGroup));
            vb.item(GdAddTextAdGroupItem.HYPER_GEO_ID)
                    .check(validId());
            vb.item(GdAddTextAdGroupItem.BID_MODIFIERS)
                    .check(notNull());
            vb.item(GdAddTextAdGroupItem.CAMPAIGN_ID)
                    .check(validId(), When.isValid());
            vb.item(GdAddTextAdGroupItem.AD_GROUP_MINUS_KEYWORDS)
                    .check(notNull())
                    .check(eachNotNull());
            vb.list(GdAddTextAdGroupItem.LIBRARY_MINUS_KEYWORDS_IDS)
                    .check(notNull())
                    .checkEach(validId());
            vb.item(GdAddTextAdGroupItem.RETARGETINGS)
                    .check(eachNotNull());
            vb.list(GdAddTextAdGroupItem.RETARGETINGS)
                    .checkEachBy(AdGroupValidationUtils.RETARGETING_ITEM_VALIDATOR);
            vb.item(GdAddTextAdGroupItem.CA_RETARGETING_CONDITION)
                    .checkBy(
                            AdGroupValidationUtils.CA_RETARGETING_ITEM_VALIDATOR,
                            When.notNullAnd(When.isTrue(isCustomAudienceEnabled))
                    );
            vb.item(GdAddTextAdGroupItem.SMART_FEED_FILTER)
                    .check(isNull(), When.isTrue(item.getSmartFeedId() == null));

            return vb.getResult();
        }
    }

    private static Constraint<List<GdAddTextAdGroupItem>, Defect> checkKeywords() {
        return checkAdGroupItemPropertyUniformity(GdAddTextAdGroupItem.KEYWORDS);
    }

    private static Constraint<List<GdAddTextAdGroupItem>, Defect> checkRelevanceMatch() {
        return checkAdGroupItemPropertyUniformity(GdAddTextAdGroupItem.RELEVANCE_MATCH);
    }

    private static Constraint<List<GdAddTextAdGroupItem>, Defect> checkAddOfferRetargetingsAreDisabled() {
        return Constraint.fromPredicate(
                list -> list.stream().allMatch(item -> item.getOfferRetargeting() == null),
                CommonDefects.invalidValue());
    }

    private static Constraint<List<GdAddTextAdGroupItem>, Defect> checkOfferRetargetings(boolean offerRetargetingsIsEnabled) {
        if (offerRetargetingsIsEnabled) {
            return checkAdGroupItemPropertyUniformity(GdAddTextAdGroupItem.OFFER_RETARGETING);
        } else {
            return checkAddOfferRetargetingsAreDisabled();
        }
    }

    private static Constraint<List<GdUpdateTextAdGroupItem>, Defect> checkUpdateOfferRetargetingsAreDisabled() {
        return Constraint.fromPredicate(
                list -> list.stream().allMatch(item -> item.getOfferRetargeting() == null),
                CommonDefects.invalidValue());
    }

    private static Constraint<List<GdUpdateTextAdGroupItem>, Defect> checkOfferRetargetingsUniformity(boolean offerRetargetingsIsEnabled) {
        if (offerRetargetingsIsEnabled) {
            return checkAdGroupItemPropertyUniformity(GdUpdateTextAdGroupItem.OFFER_RETARGETING);
        } else {
            return checkUpdateOfferRetargetingsAreDisabled();
        }
    }

    private static Constraint<List<GdAddTextAdGroupItem>, Defect> checkDynamicFeedAdTargets(boolean isEnabledDynamicTextAdTarget) {
        return list -> {
            var allItemsAreNulls = list.stream().allMatch(item -> item.getDynamicFeedAdTarget() == null);
            if (!isEnabledDynamicTextAdTarget && !allItemsAreNulls) {
                return CommonDefects.invalidValue();
            } else if (isEnabledDynamicTextAdTarget &&
                    !allItemsAreNulls &&
                    !list.stream().allMatch(item -> item.getDynamicFeedAdTarget() != null)) {
                return CollectionDefects.notContainNulls();
            }
            return null;
        };
    }

    private static Constraint<List<GdUpdateTextAdGroupItem>, Defect> checkKeywordsUniformity() {
        return checkAdGroupItemPropertyUniformity(GdUpdateTextAdGroupItem.KEYWORDS);
    }

    private static Constraint<List<GdUpdateTextAdGroupItem>, Defect> checkRelevanceMatchUniformity() {
        return checkAdGroupItemPropertyUniformity(GdUpdateTextAdGroupItem.RELEVANCE_MATCH);
    }

    private static <I extends Model, M> Constraint<List<I>, Defect> checkAdGroupItemPropertyUniformity(
            ModelProperty<I, M> modelProperty) {
        return Constraint.fromPredicate(
                list -> list.stream().allMatch(item -> modelProperty.get(item) != null)
                        || list.stream().allMatch(item -> modelProperty.get(item) == null),
                CollectionDefects.notContainNulls());
    }

    private static Constraint<Long, Defect> checkAssociatedFeedBusinessType(Map<Long, Feed> feedsByFeedId) {
        return Constraint.fromPredicate(aLong -> {
            var feed = feedsByFeedId.get(aLong);
            return feed != null && feed.getBusinessType() == BusinessType.RETAIL;
        }, CommonDefects.invalidValue());
    }

    private static Constraint<GdUpdateAdGroupOfferRetargetingItem, Defect> checkRelevanceMatchIsDisabled(GdUpdateAdGroupRelevanceMatchItem relevanceMatchItem) {
        return Constraint.fromPredicate(offerRetargeting ->
                        !offerRetargeting.getIsActive() || relevanceMatchItem == null || !relevanceMatchItem.getIsActive(),
                CommonDefects.invalidValue());
    }

    private static Constraint<Long, Defect> checkFeedExists(Map<Long, Feed> feedsByFeedId) {
        return Constraint.fromPredicate(feedsByFeedId::containsKey, FeedDefects.feedNotExist());
    }

    /**
     * проверяет, что не включен одновременно автотаргетинг и динамическое нацеливание,
     * а также, что передан id фида для динамического нацеливания
     */
    private static Constraint<GdAddAdGroupDynamicFeedAdTargetItem, Defect> checkDynamicAdTargets(
            @Nullable GdUpdateAdGroupRelevanceMatchItem relevanceMatchItem,
            @Nullable Long feedId) {
        return dynamicAdTarget -> {
            if (dynamicAdTarget.getIsActive() && relevanceMatchItem != null && relevanceMatchItem.getIsActive()) {
                return CommonDefects.invalidValue();
            } else if (dynamicAdTarget.getIsActive() && feedId == null) {
                return FeedDefects.feedIsNotSet();
            }
            return null;
        };
    }

    private class AdGroupItemsValidator implements Validator<GdUpdateTextAdGroupItem, Defect> {

        private final Currency clientWorkCurrency;

        private final Map<Long, Feed> feedsByFeedId;

        AdGroupItemsValidator(Currency clientWorkCurrency, Map<Long, Feed> feedsByFeedId) {
            this.clientWorkCurrency = clientWorkCurrency;
            this.feedsByFeedId = feedsByFeedId;
        }

        @Override
        public ValidationResult<GdUpdateTextAdGroupItem, Defect> apply(
                GdUpdateTextAdGroupItem gdUpdateTextAdGroupItem) {
            ModelItemValidationBuilder<GdUpdateTextAdGroupItem> vb =
                    ModelItemValidationBuilder.of(gdUpdateTextAdGroupItem);
            vb.item(GdUpdateTextAdGroupItem.GENERAL_PRICE)
                    .check(inRange(clientWorkCurrency.getMinPrice(), clientWorkCurrency.getMaxPrice()), When.notNull());
            vb.item(GdUpdateTextAdGroupItem.FEED_ID)
                    .check(validId(), When.notNull())
                    .check(checkAssociatedFeedBusinessType(feedsByFeedId), When.notNull());
            vb.item(GdUpdateTextAdGroupItem.SMART_FEED_ID)
                    .check(validId(), When.notNull())
                    .check(checkFeedExists(feedsByFeedId), When.isValid());
            vb.item(GdUpdateTextAdGroupItem.OFFER_RETARGETING)
                    .check(checkRelevanceMatchIsDisabled(gdUpdateTextAdGroupItem.getRelevanceMatch()), When.notNull());
            return vb.getResult();
        }
    }

    private class AdGroupValidator implements Validator<GdUpdateTextAdGroup, Defect> {


        private final Map<Long, AdGroupType> adGroupTypeMap;
        private final AdGroupItemsValidator adGroupItemsValidator;

        AdGroupValidator(Map<Long, AdGroupType> adGroupTypeMap, Currency clientWorkCurrency,
                         Map<Long, Feed> feedsByFeedId) {
            this.adGroupTypeMap = adGroupTypeMap;
            this.adGroupItemsValidator = new AdGroupItemsValidator(clientWorkCurrency, feedsByFeedId);
        }

        @Override
        public ValidationResult<GdUpdateTextAdGroup, Defect> apply(
                GdUpdateTextAdGroup gdAdGroupUpdateInput) {
            ModelItemValidationBuilder<GdUpdateTextAdGroup> vb = ModelItemValidationBuilder.of(gdAdGroupUpdateInput);
            vb.list(GdUpdateTextAdGroup.UPDATE_ITEMS)
                    .checkEach(isTextAdGroups())
                    .checkEachBy(adGroupItemsValidator);
            return vb.getResult();
        }

        private ListConstraint<GdUpdateTextAdGroupItem, Defect> isTextAdGroups() {
            return adGroupUpdateItems -> {
                Map<Integer, Defect> result = new HashMap<>();
                for (int i = 0; i < adGroupUpdateItems.size(); i++) {
                    if (!AdGroupType.BASE.equals(adGroupTypeMap.get(adGroupUpdateItems.get(i).getAdGroupId()))) {
                        result.put(i, GridDefectDefinitions.unsupportedAdGroupType());
                    }
                }
                return result;
            };
        }
    }

    private class AdGroupAddValidator implements Validator<GdAddTextAdGroup, Defect> {

        private final AdGroupAddItemsValidator adGroupItemsValidator;

        AdGroupAddValidator(ClientId clientId, Map<Long, Feed> feedsByFeedId) {
            this.adGroupItemsValidator = new AdGroupAddItemsValidator(clientId, feedsByFeedId);
        }

        @Override
        public ValidationResult<GdAddTextAdGroup, Defect> apply(
                GdAddTextAdGroup gdAdGroupAddInput) {
            ModelItemValidationBuilder<GdAddTextAdGroup> vb = ModelItemValidationBuilder.of(gdAdGroupAddInput);
            vb.list(GdAddTextAdGroup.ADD_ITEMS)
                    .checkEachBy(adGroupItemsValidator);
            return vb.getResult();
        }
    }

    private class AdGroupAddItemsValidator implements Validator<GdAddTextAdGroupItem, Defect> {

        private final Currency clientWorkCurrency;
        private final Map<Long, Feed> feedsByFeedId;

        AdGroupAddItemsValidator(ClientId clientId, Map<Long, Feed> feedsByFeedId) {
            this.clientWorkCurrency = clientService.getWorkCurrency(clientId);
            this.feedsByFeedId = feedsByFeedId;
        }

        @Override
        public ValidationResult<GdAddTextAdGroupItem, Defect> apply(GdAddTextAdGroupItem gdAddTextAdGroupItem) {
            ModelItemValidationBuilder<GdAddTextAdGroupItem> vb = ModelItemValidationBuilder.of(gdAddTextAdGroupItem);
            vb.item(GdAddTextAdGroupItem.GENERAL_PRICE)
                    .check(inRange(clientWorkCurrency.getMinPrice(), clientWorkCurrency.getMaxPrice()), When.notNull());
            vb.item(GdAddTextAdGroupItem.FEED_ID)
                    .check(validId(), When.notNull())
                    .check(checkAssociatedFeedBusinessType(feedsByFeedId), When.notNull());
            vb.item(GdAddTextAdGroupItem.SMART_FEED_ID)
                    .check(validId(), When.notNull())
                    .check(checkFeedExists(feedsByFeedId), When.isValid());
            vb.item(GdAddTextAdGroupItem.DYNAMIC_FEED_AD_TARGET)
                    .check(checkDynamicAdTargets(
                            gdAddTextAdGroupItem.getRelevanceMatch(),
                            gdAddTextAdGroupItem.getSmartFeedId()), When.notNull());
            vb.item(GdAddTextAdGroupItem.OFFER_RETARGETING)
                    .check(checkRelevanceMatchIsDisabled(gdAddTextAdGroupItem.getRelevanceMatch()), When.notNull());
            return vb.getResult();
        }
    }
}
