package ru.yandex.direct.grid.processing.service.hypergeo.service;

import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.hypergeo.model.HyperGeoSegment;
import ru.yandex.direct.core.entity.hypergeo.model.HyperGeoSegmentDetails;
import ru.yandex.direct.core.entity.hypergeo.model.HyperGeoSimple;
import ru.yandex.direct.core.entity.hypergeo.service.CreateHyperGeoResultContainer;
import ru.yandex.direct.core.entity.hypergeo.service.HyperGeoService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.processing.model.api.GdValidationResult;
import ru.yandex.direct.grid.processing.model.hypergeo.mutation.createhypergeo.GdCreateHyperGeos;
import ru.yandex.direct.grid.processing.model.hypergeo.mutation.createhypergeo.GdCreateHyperGeosPayload;
import ru.yandex.direct.grid.processing.model.hypergeo.mutation.createhypergeo.GdCreateHyperGeosPayloadItem;
import ru.yandex.direct.grid.processing.model.hypergeo.mutation.createhypergeosegment.GdCreateHyperGeoSegment;
import ru.yandex.direct.grid.processing.model.hypergeo.mutation.createhypergeosegment.GdCreateHyperGeoSegments;
import ru.yandex.direct.grid.processing.model.hypergeo.mutation.createhypergeosegment.GdCreateHyperGeoSegmentsPayload;
import ru.yandex.direct.grid.processing.model.hypergeo.mutation.createhypergeosegment.GdCreateHyperGeoSegmentsPayloadItem;
import ru.yandex.direct.grid.processing.service.hypergeo.HyperGeoUtils;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.PathHelper.path;

@Service
@ParametersAreNonnullByDefault
public class HyperGeoMutationService {

    private final HyperGeoService hyperGeoService;
    private final GridValidationService gridValidationService;
    private final HyperGeoMutationValidationService hyperGeoMutationValidationService;

    @Autowired
    public HyperGeoMutationService(HyperGeoService hyperGeoService,
                                   HyperGeoMutationValidationService hyperGeoMutationValidationService,
                                   GridValidationService gridValidationService) {
        this.hyperGeoService = hyperGeoService;
        this.hyperGeoMutationValidationService = hyperGeoMutationValidationService;
        this.gridValidationService = gridValidationService;
    }

    public GdCreateHyperGeoSegmentsPayload createHyperGeoSegments(ClientId clientId,
                                                                  String login,
                                                                  GdCreateHyperGeoSegments input) {
        List<GdCreateHyperGeoSegment> hyperGeoSegmentItems = input.getCreateHyperGeoSegmentItems();
        if (hyperGeoSegmentItems.isEmpty()) {
            return new GdCreateHyperGeoSegmentsPayload()
                    .withCreatedHyperGeoSegments(emptyList())
                    .withValidationResult(null);
        }

        hyperGeoMutationValidationService.validateCreateHyperGeoSegments(input);

        List<HyperGeoSegmentDetails> hyperGeoSegmentDetails = mapList(hyperGeoSegmentItems, HyperGeoUtils::convert);

        CreateHyperGeoResultContainer results =
                hyperGeoService.createHyperGeoSegments(clientId, login, hyperGeoSegmentDetails);
        Map<Long, List<Long>> coveringGeoByHyperGeoSegmentId = StreamEx.of(results.getModels())
                .mapToEntry(HyperGeoSegment::getId, HyperGeoSegment::getCoveringGeo)
                .nonNullKeys()
                .toMap();

        List<GdCreateHyperGeoSegmentsPayloadItem> payloadItems = StreamEx.of(results.getMassResult().getResult())
                .filter(Result::isSuccessful)
                .map(Result::getResult)
                .map(hyperGeoSegmentId -> new GdCreateHyperGeoSegmentsPayloadItem()
                        .withId(hyperGeoSegmentId)
                        .withCoveringGeo(coveringGeoByHyperGeoSegmentId.get(hyperGeoSegmentId)))
                .toList();

        GdValidationResult gdValidationResult =
                gridValidationService.getValidationResult(results.getMassResult(),
                        path(field(GdCreateHyperGeoSegments.CREATE_HYPER_GEO_SEGMENT_ITEMS)));

        return new GdCreateHyperGeoSegmentsPayload()
                .withCreatedHyperGeoSegments(payloadItems)
                .withValidationResult(gdValidationResult);
    }

    public GdCreateHyperGeosPayload createHyperGeos(ClientId clientId, GdCreateHyperGeos input) {
        if (input.getCreateHyperGeoItems().isEmpty()) {
            return new GdCreateHyperGeosPayload()
                    .withCreatedHyperGeos(emptyList())
                    .withValidationResult(null);
        }

        hyperGeoMutationValidationService.validateCreateHyperGeos(input);

        List<HyperGeoSimple> hyperGeos = mapList(input.getCreateHyperGeoItems(), HyperGeoUtils::convert);

        MassResult<Long> results = hyperGeoService.createHyperGeos(clientId, hyperGeos);
        List<GdCreateHyperGeosPayloadItem> payloadItems = StreamEx.of(results.getResult())
                .filter(Result::isSuccessful)
                .map(result -> new GdCreateHyperGeosPayloadItem().withId(result.getResult()))
                .toList();

        GdValidationResult gdValidationResult =
                gridValidationService.getValidationResult(results,
                        path(field(GdCreateHyperGeos.CREATE_HYPER_GEO_ITEMS)));

        return new GdCreateHyperGeosPayload()
                .withCreatedHyperGeos(payloadItems)
                .withValidationResult(gdValidationResult);
    }
}
