package ru.yandex.direct.grid.processing.service.internalad;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import io.leangen.graphql.annotations.GraphQLArgument;
import io.leangen.graphql.annotations.GraphQLContext;
import io.leangen.graphql.annotations.GraphQLNonNull;
import io.leangen.graphql.annotations.GraphQLQuery;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.internalads.model.InternalTemplateInfo;
import ru.yandex.direct.core.entity.internalads.model.ResourceChoice;
import ru.yandex.direct.core.entity.internalads.model.ResourceInfo;
import ru.yandex.direct.core.entity.internalads.model.ResourceRestriction;
import ru.yandex.direct.core.entity.internalads.model.ResourceType;
import ru.yandex.direct.core.entity.internalads.restriction.Restriction;
import ru.yandex.direct.core.entity.internalads.restriction.RestrictionImageDimensionsEq;
import ru.yandex.direct.core.entity.internalads.restriction.RestrictionImageDimensionsMax;
import ru.yandex.direct.core.entity.internalads.restriction.RestrictionImageDimensionsMin;
import ru.yandex.direct.core.entity.internalads.restriction.RestrictionImageDimensionsSquare;
import ru.yandex.direct.core.entity.internalads.restriction.RestrictionImageFormat;
import ru.yandex.direct.core.entity.internalads.restriction.RestrictionImageFormatIn;
import ru.yandex.direct.core.entity.internalads.restriction.RestrictionImageSizeKbMax;
import ru.yandex.direct.core.entity.internalads.restriction.RestrictionRegexp;
import ru.yandex.direct.core.entity.internalads.restriction.RestrictionTextLengthMax;
import ru.yandex.direct.core.entity.internalads.restriction.RestrictionTextLengthMin;
import ru.yandex.direct.core.entity.internalads.restriction.RestrictionUrlCorrect;
import ru.yandex.direct.core.entity.internalads.service.TemplateInfoService;
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService;
import ru.yandex.direct.grid.processing.model.client.GdClient;
import ru.yandex.direct.grid.processing.model.internalad.GdImageDimensionsValueRestriction;
import ru.yandex.direct.grid.processing.model.internalad.GdImageFormatValueRestriction;
import ru.yandex.direct.grid.processing.model.internalad.GdImageSizeKbMaxValueRestriction;
import ru.yandex.direct.grid.processing.model.internalad.GdInternalTemplateInfo;
import ru.yandex.direct.grid.processing.model.internalad.GdInternalTemplateInfoRequest;
import ru.yandex.direct.grid.processing.model.internalad.GdInternalTemplateInfoRequestEntry;
import ru.yandex.direct.grid.processing.model.internalad.GdLengthValueRestriction;
import ru.yandex.direct.grid.processing.model.internalad.GdRegexpRestriction;
import ru.yandex.direct.grid.processing.model.internalad.GdResourceInfo;
import ru.yandex.direct.grid.processing.model.internalad.GdResourceRestriction;
import ru.yandex.direct.grid.processing.model.internalad.GdResourceType;
import ru.yandex.direct.grid.processing.model.internalad.GdResourceValueChoice;
import ru.yandex.direct.grid.processing.model.internalad.GdResourceValueRestriction;
import ru.yandex.direct.grid.processing.model.internalad.GdResourceValueRestrictionType;
import ru.yandex.direct.grid.processing.model.internalad.GdRestrictionImageFormat;
import ru.yandex.direct.grid.processing.model.internalad.GdUrlCorrectRestriction;
import ru.yandex.direct.utils.converter.Converters;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.utils.FunctionalUtils.flatMapToSet;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@GridGraphQLService
@ParametersAreNonnullByDefault
public class InternalAdTemplateInfoGraphQlService {
    private static final String INTERNAL_AD_TEMPLATE_INFO_RESOLVER_NAME = "internalAdTemplateInfo";

    private static final Map<RestrictionImageFormat, GdRestrictionImageFormat> FORMAT_MAP = Map.of(
            RestrictionImageFormat.PNG, GdRestrictionImageFormat.PNG,
            RestrictionImageFormat.GIF, GdRestrictionImageFormat.GIF,
            RestrictionImageFormat.JPEG, GdRestrictionImageFormat.JPEG,
            RestrictionImageFormat.SVG, GdRestrictionImageFormat.SVG
    );

    private static final Map<ResourceType, GdResourceType> RESOURCE_TYPE_MAP = Map.of(
            ResourceType.URL, GdResourceType.URL,
            ResourceType.IMAGE, GdResourceType.IMAGE,
            ResourceType.AGE, GdResourceType.AGE,
            ResourceType.TEXT, GdResourceType.TEXT,
            ResourceType.CLOSE_COUNTER, GdResourceType.TEXT
    );

    private final TemplateInfoService templateInfoService;

    @Autowired
    public InternalAdTemplateInfoGraphQlService(TemplateInfoService templateInfoService) {
        this.templateInfoService = templateInfoService;
    }

    /**
     * GraphQL подзапрос. Выдаёт информацию о шаблоне, его ресурсах и ограничениях.
     */
    @GraphQLNonNull
    @GraphQLQuery(name = INTERNAL_AD_TEMPLATE_INFO_RESOLVER_NAME)
    public List<GdInternalTemplateInfo> internalAdTemplateInfo(
            @GraphQLContext GdClient client,
            @GraphQLNonNull @GraphQLArgument(name = "request") GdInternalTemplateInfoRequest request
    ) {
        Set<Long> templateIds = flatMapToSet(request.getEntries(), GdInternalTemplateInfoRequestEntry::getTemplateIds);
        Map<Long, InternalTemplateInfo> templateInfoMap = listToMap(
                templateInfoService.getByTemplateIds(templateIds),
                InternalTemplateInfo::getTemplateId);
        return StreamEx.of(request.getEntries())
                .mapToEntry(GdInternalTemplateInfoRequestEntry::getCampaignId,
                        GdInternalTemplateInfoRequestEntry::getTemplateIds)
                .flatMapValues(Collection::stream)
                .mapValues(templateInfoMap::get)
                .nonNullValues()
                .mapKeyValue(this::convertTemplateInfo)
                .toList();
    }

    private GdInternalTemplateInfo convertTemplateInfo(Long campaignId, InternalTemplateInfo templateInfo) {
        return new GdInternalTemplateInfo()
                .withCampaignId(campaignId)
                .withTemplateId(templateInfo.getTemplateId())
                .withResources(mapList(templateInfo.getResources(), this::convertResource))
                .withResourceRestrictionsErrorMessage(templateInfo.getResourceRestrictionsErrorMessage())
                .withResourceRestrictions(mapList(templateInfo.getResourceRestrictions(),
                        this::convertResourceRestriction));
    }

    private GdResourceInfo convertResource(ResourceInfo resource) {
        return new GdResourceInfo()
                .withId(resource.getId())
                .withLabel(resource.getLabel())
                .withType(convertResourceType(resource.getType()))
                .withValueRestrictions(mapList(resource.getValueRestrictions(), this::convertValueRestriction))
                .withChoices(mapList(resource.getChoices(), this::convertChoice))
                .withHidden(resource.isHidden());
    }

    private static GdResourceType convertResourceType(ResourceType resourceType) {
        checkArgument(RESOURCE_TYPE_MAP.containsKey(resourceType), "unexpected resourceType %s", resourceType);
        return RESOURCE_TYPE_MAP.get(resourceType);
    }

    private GdResourceValueChoice convertChoice(ResourceChoice choice) {
        return new GdResourceValueChoice().withValue(choice.getValue()).withDisplay(choice.getDisplay());
    }

    private GdResourceRestriction convertResourceRestriction(ResourceRestriction resourceRestriction) {
        return new GdResourceRestriction()
                .withRequired(resourceRestriction.getRequired())
                .withAbsent(resourceRestriction.getAbsent());
    }

    private GdResourceValueRestriction convertValueRestriction(Restriction valueRestriction) {
        GdResourceValueRestriction gdValueRestriction;
        if (valueRestriction instanceof RestrictionTextLengthMin) {
            gdValueRestriction = new GdLengthValueRestriction()
                    .withType(GdResourceValueRestrictionType.TEXT_LENGTH_MIN)
                    .withLength(((RestrictionTextLengthMin) valueRestriction).getLength());
        } else if (valueRestriction instanceof RestrictionTextLengthMax) {
            gdValueRestriction = new GdLengthValueRestriction()
                    .withType(GdResourceValueRestrictionType.TEXT_LENGTH_MAX)
                    .withLength(((RestrictionTextLengthMax) valueRestriction).getLength());
        } else if (valueRestriction instanceof RestrictionImageFormatIn) {
            gdValueRestriction = new GdImageFormatValueRestriction()
                    .withType(GdResourceValueRestrictionType.IMAGE_FORMAT_IN)
                    .withFormat(convertFormat(((RestrictionImageFormatIn) valueRestriction).getFormats()));
        } else if (valueRestriction instanceof RestrictionImageDimensionsEq) {
            gdValueRestriction = new GdImageDimensionsValueRestriction()
                    .withType(GdResourceValueRestrictionType.IMAGE_DIMENSIONS_EQ)
                    .withWidth(((RestrictionImageDimensionsEq) valueRestriction).getWidth())
                    .withHeight(((RestrictionImageDimensionsEq) valueRestriction).getHeight())
                    .withFormatsForCheck(convertFormat((
                            (RestrictionImageDimensionsEq) valueRestriction).getFormatsForCheck()));
        } else if (valueRestriction instanceof RestrictionImageDimensionsSquare) {
            gdValueRestriction = new GdImageDimensionsValueRestriction()
                    .withType(GdResourceValueRestrictionType.IMAGE_DIMENSIONS_SQUARE)
                    .withFormatsForCheck(convertFormat((
                            (RestrictionImageDimensionsSquare) valueRestriction).getFormatsForCheck()));
        } else if (valueRestriction instanceof RestrictionImageDimensionsMin) {
            gdValueRestriction = new GdImageDimensionsValueRestriction()
                    .withType(GdResourceValueRestrictionType.IMAGE_DIMENSIONS_MIN)
                    .withWidth(((RestrictionImageDimensionsMin) valueRestriction).getWidth())
                    .withHeight(((RestrictionImageDimensionsMin) valueRestriction).getHeight())
                    .withFormatsForCheck(convertFormat((
                            (RestrictionImageDimensionsMin) valueRestriction).getFormatsForCheck()));
        } else if (valueRestriction instanceof RestrictionImageDimensionsMax) {
            gdValueRestriction = new GdImageDimensionsValueRestriction()
                    .withType(GdResourceValueRestrictionType.IMAGE_DIMENSIONS_MAX)
                    .withWidth(((RestrictionImageDimensionsMax) valueRestriction).getWidth())
                    .withHeight(((RestrictionImageDimensionsMax) valueRestriction).getHeight())
                    .withFormatsForCheck(convertFormat(
                            ((RestrictionImageDimensionsMax) valueRestriction).getFormatsForCheck()));
        } else if (valueRestriction instanceof RestrictionImageSizeKbMax) {
            gdValueRestriction = new GdImageSizeKbMaxValueRestriction()
                    .withType(GdResourceValueRestrictionType.IMAGE_SIZE_KB_MAX)
                    .withSize(((RestrictionImageSizeKbMax) valueRestriction).getSize())
                    .withFormatsForCheck(convertFormat(
                            ((RestrictionImageSizeKbMax) valueRestriction).getFormatsForCheck()));
        } else if (valueRestriction instanceof RestrictionUrlCorrect) {
            gdValueRestriction = new GdUrlCorrectRestriction()
                    .withType(GdResourceValueRestrictionType.URL_CORRECT)
                    .withIsHttpsOnly(((RestrictionUrlCorrect) valueRestriction).isHttpsOnly());
        } else if (valueRestriction instanceof RestrictionRegexp) {
            gdValueRestriction = new GdRegexpRestriction()
                    .withType(GdResourceValueRestrictionType.REGEXP)
                    .withRegexp(((RestrictionRegexp) valueRestriction).regexp());
        } else {
            throw new IllegalStateException(String.format("unknown restriction type %s", valueRestriction));
        }
        checkState(gdValueRestriction.getType() != null);
        return gdValueRestriction
                .withStrict(valueRestriction.isStrict())
                .withDefectCode(valueRestriction.getDefectId().getCode());
    }

    @Nullable
    private List<GdRestrictionImageFormat> convertFormat(@Nullable Collection<RestrictionImageFormat> formats) {
        if (formats == null) {
            return null;
        }

        return Converters.mappingValueConverter(FORMAT_MAP).convertList(formats);
    }
}
