package ru.yandex.direct.grid.processing.service.inventori;

import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Set;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.exception.GridValidationException;
import ru.yandex.direct.grid.processing.model.api.GdDefect;
import ru.yandex.direct.grid.processing.model.api.GdValidationResult;
import ru.yandex.direct.grid.processing.model.inventori.GdIndoorReachRequest;
import ru.yandex.direct.grid.processing.model.inventori.GdOutdoorReachRequest;
import ru.yandex.direct.grid.processing.model.inventori.GdReachBudgetInfo;
import ru.yandex.direct.grid.processing.model.inventori.GdReachIndoorResult;
import ru.yandex.direct.grid.processing.model.inventori.GdReachMultiBudgetRequest;
import ru.yandex.direct.grid.processing.model.inventori.GdReachMultiBudgetsResult;
import ru.yandex.direct.grid.processing.model.inventori.GdReachOutdoorResult;
import ru.yandex.direct.grid.processing.model.inventori.GdReachRecommendationResult;
import ru.yandex.direct.grid.processing.model.inventori.GdReachRequest;
import ru.yandex.direct.grid.processing.model.inventori.GdReachResult;
import ru.yandex.direct.grid.processing.model.inventori.GdUacForecastResponse;
import ru.yandex.direct.grid.processing.model.inventori.GdUacReachRequest;
import ru.yandex.direct.grid.processing.model.inventori.GdUacRecommendationRequest;
import ru.yandex.direct.grid.processing.model.inventori.GdUacRecommendationResponse;
import ru.yandex.direct.validation.defect.params.NumberDefectParams;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.web.core.entity.inventori.model.GeneralCampaignPredictionResult;
import ru.yandex.direct.web.core.entity.inventori.model.GeneralCpmRecommendationRequest;
import ru.yandex.direct.web.core.entity.inventori.model.GeneralCpmRecommendationResult;
import ru.yandex.direct.web.core.entity.inventori.model.ReachIndoorRequest;
import ru.yandex.direct.web.core.entity.inventori.model.ReachIndoorResult;
import ru.yandex.direct.web.core.entity.inventori.model.ReachMultiBudgetResult;
import ru.yandex.direct.web.core.entity.inventori.model.ReachOutdoorRequest;
import ru.yandex.direct.web.core.entity.inventori.model.ReachOutdoorResult;
import ru.yandex.direct.web.core.entity.inventori.model.ReachRecommendationResult;
import ru.yandex.direct.web.core.entity.inventori.model.ReachRequest;
import ru.yandex.direct.web.core.entity.inventori.model.ReachResult;
import ru.yandex.direct.web.core.entity.inventori.service.InventoriService;
import ru.yandex.direct.web.core.entity.inventori.service.InventoriWebService;
import ru.yandex.direct.web.core.entity.inventori.service.InventoriWebValidationService;

import static java.util.Collections.emptyList;
import static java.util.Collections.min;
import static ru.yandex.direct.utils.CollectionUtils.isEmpty;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;
import static ru.yandex.direct.web.core.entity.inventori.validation.InventoriDefectIds.Number.LOW_REACH;

@Service
public class GridInventoriService {
    public static final long MIN_REACH = 1000L;
    private final InventoriWebService inventoriWebService;
    private final InventoriWebValidationService inventoriValidationService;
    private final InventoriDataConverter inventoriDataConverter;
    private final ClientService clientService;
    private final InventoriService inventoriService;

    @Autowired
    public GridInventoriService(InventoriWebValidationService inventoriValidationService,
                                InventoriWebService inventoriWebService,
                                InventoriDataConverter inventoriDataConverter,
                                ClientService clientService,
                                InventoriService inventoriService) {
        this.inventoriWebService = inventoriWebService;
        this.inventoriValidationService = inventoriValidationService;
        this.inventoriDataConverter = inventoriDataConverter;
        this.clientService = clientService;
        this.inventoriService = inventoriService;
    }

    public GdReachResult getReach(GdReachRequest input) {
        ReachRequest request = inventoriDataConverter.convertReachRequestFromGd(input);
        ValidationResult<ReachRequest, Defect> preValidationResult = inventoriValidationService.validate(request);
        checkErrors(preValidationResult);

        ReachResult result = inventoriWebService.getReachForecast(request);
        return inventoriDataConverter.convertReachResultToGd(result);
    }

    public GdReachResult getReach(GdUacReachRequest input) {
        ReachRequest request = inventoriDataConverter.convertReachRequestFromGd(input);
        ValidationResult<ReachRequest, Defect> preValidationResult = inventoriValidationService.validate(request);
        checkErrors(preValidationResult);

        ReachResult result = inventoriWebService.getGeneralReachForecast(request);
        return inventoriDataConverter.convertReachResultToGd(result);
    }

    public GdUacRecommendationResponse getRecommendation(GridGraphQLContext context,
                                                         GdUacRecommendationRequest input) {

        ClientId clientId = context.getSubjectUser().getClientId();
        User operator = context.getOperator();
        Client client = clientService.getClient(clientId);

        GeneralCpmRecommendationRequest request = inventoriDataConverter.convertUacRecommendationRequest(input);
        ValidationResult<GeneralCpmRecommendationRequest, Defect>
                preValidationResult = inventoriValidationService.validate(request, operator.getUid(), clientId);
        checkErrors(preValidationResult);

        CurrencyCode currency = client.getWorkCurrency();

        GeneralCpmRecommendationResult response = inventoriWebService.forecast(request, currency);
        List<Defect> errors = response.getErrors().getErrors();
        if (!isEmpty(errors)) {
            if (errors.size() == 1 && errors.get(0).defectId() == LOW_REACH) {
                NumberDefectParams defect = (NumberDefectParams) errors.get(0).params();
                return new GdUacRecommendationResponse().withReachLessThan(defect.getMin().longValue());
            } else {
                throw convertToGridValidationException(errors);
            }
        } else {
            return inventoriDataConverter.convertResultToGd(response.getResult(), response.getRequestId());
        }
    }

    public GdUacForecastResponse getUacCampaignForecast(GridGraphQLContext context,
                                                      GdUacRecommendationRequest input) {
        ClientId clientId = context.getSubjectUser().getClientId();
        User operator = context.getOperator();
        Client client = clientService.getClient(clientId);
        GeneralCpmRecommendationRequest request = inventoriDataConverter.convertUacRecommendationRequest(input);
        ValidationResult<GeneralCpmRecommendationRequest, Defect>
                preValidationResult = inventoriValidationService.validate(request, operator.getUid(), clientId);
        checkErrors(preValidationResult);

        CurrencyCode currency = client.getWorkCurrency();

        GeneralCampaignPredictionResult response = inventoriWebService.getGeneralCampaignPrediction(request, currency);
        List<Defect> errors = response.getErrors().getErrors();
        if (!errors.isEmpty()) {
            throw convertToGridValidationException(errors);
        } else {
            return inventoriDataConverter.convertGeneralCampaignPredictionResultToGd(response.getResult(), response.getRequestId());
        }

    }

    public GdReachMultiBudgetsResult getReachMultiBudgets(GdReachMultiBudgetRequest input, ClientId clientId) {
        ReachRequest request = inventoriDataConverter.convertReachRequestFromGd(input, clientId);
        ValidationResult<ReachRequest, Defect> preValidationResult = inventoriValidationService.validate(request);
        checkErrors(preValidationResult);

        ReachMultiBudgetResult result = inventoriWebService.getReachMultiBudgetsForecast(request,
                mapList(input.getBudgets(), sum -> sum * 1_000_000));

        List<Defect> errors = result.getErrors();
        if (!isEmpty(errors)) {
            if (errors.size() == 1 && errors.get(0).defectId() == LOW_REACH) {
                return getEmptyMultiBudgetsResult(result.getRequestId());
            }
            throw convertToGridValidationException(errors);
        }

        GdReachMultiBudgetsResult gdResult =
                inventoriDataConverter.convertParametrisedCampaignPredictionResponseToGd(result);

        /*
         *    Могут быть ситуации, когда по всем бюджетам ожидается один охват.
         *    Нам выгодно стимулировать больший бюджет, поэтому когда прогноз показов по всем бюджетам округленный
         *    до тысячи совпадает, то берем максимум и для меньших бюджетов показываем охват пропорционально бюджету
         */
        Set<Long> reaches = listToSet(gdResult.getReaches(), GdReachBudgetInfo::getReach);
        Set<Long> roundedReaches = mapSet(reaches, r -> Math.round(r / 1000.0) * 1000); // округляем до 1000
        if (!isEmpty(reaches) && min(reaches) >= MIN_REACH && roundedReaches.size() == 1) {
            GdReachBudgetInfo maxBudgetInfo = Collections.max(gdResult.getReaches(),
                    Comparator.comparing(GdReachBudgetInfo::getBudget));

            gdResult.getReaches()
                    .forEach(budgetInfo -> decreaseReachBudgetInfo(budgetInfo, maxBudgetInfo,
                            maxBudgetInfo.getBudget(), maxBudgetInfo.getReach()));
        }

        // Если охват меньше 1000, то информацию об охвате не выводим
        for (GdReachBudgetInfo r : gdResult.getReaches()) {
            if (r.getReach() < MIN_REACH) {
                r.withShowReachLessThan(true)
                        .withReach(MIN_REACH)
                        .withLeftIntervalReach(0L)
                        .withRightIntervalReach(0L);
            }
        }

        return gdResult;
    }

    private GdReachMultiBudgetsResult getEmptyMultiBudgetsResult(String requestId) {
        return new GdReachMultiBudgetsResult()
                .withRequestId(requestId)
                .withTotalReach(0L)
                .withReaches(emptyList());
    }

    private void decreaseReachBudgetInfo(GdReachBudgetInfo budgetInfo, GdReachBudgetInfo maxBudgetInfo, Long maxBudget,
                                         Long maxReach) {
        double coef = 1.0 * budgetInfo.getBudget() / maxBudget;
        long reach = (long) (maxReach * coef);
        Long leftInterval = (long) ((maxBudgetInfo.getReach() - maxBudgetInfo.getLeftIntervalReach()) * coef);
        Long rightInterval = (long) ((maxBudgetInfo.getRightIntervalReach() - maxBudgetInfo.getReach()) * coef);
        budgetInfo.withReach(reach);
        budgetInfo.withLeftIntervalReach(reach - leftInterval);
        budgetInfo.withRightIntervalReach(reach + rightInterval);
    }

    public GdReachOutdoorResult getOutdoorReach(GdOutdoorReachRequest input, ClientId clientId) {
        ReachOutdoorRequest request = inventoriDataConverter.convertReachRequestFromGd(input);
        ValidationResult<ReachOutdoorRequest, Defect> preValidationResult =
                inventoriValidationService.validate(request, clientId);
        checkErrors(preValidationResult);

        ReachOutdoorResult result = inventoriWebService.getReachOutdoorForecast(request);
        return inventoriDataConverter.convertReachResultToGd(result);
    }

    public GdReachIndoorResult getIndoorReach(GdIndoorReachRequest input, ClientId clientId) {
        ReachIndoorRequest request = inventoriDataConverter.convertReachRequestFromGd(input);
        ValidationResult<ReachIndoorRequest, Defect> preValidationResult =
                inventoriValidationService.validate(request, clientId);
        checkErrors(preValidationResult);

        ReachIndoorResult result = inventoriWebService.getReachIndoorForecast(request);
        return inventoriDataConverter.convertReachResultToGd(result);
    }

    private void checkErrors(ValidationResult<?, Defect> preValidationResult) {
        if (preValidationResult.hasAnyErrors()) {
            throw convertToGridValidationException(preValidationResult.getErrors());
        }
    }

    private GridValidationException convertToGridValidationException(List<Defect> errors) {
        return new GridValidationException(new GdValidationResult()
                .withErrors(mapList(errors,
                        e -> new GdDefect()
                                .withCode(e.defectId().toString())
                                .withParams(e.params())
                )));
    }

    public GdReachRecommendationResult getReachRecommendation(GdReachRequest input) {
        ReachRecommendationResult result =
                inventoriWebService.getReachRecommendation(inventoriDataConverter.convertReachRequestFromGd(input));
        return inventoriDataConverter.convertReachRecommendationResultToGd(result);
    }

}
