package ru.yandex.direct.grid.processing.service.inventori;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import com.google.common.collect.ImmutableMap;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.crypta.repository.CryptaSegmentRepository;
import ru.yandex.direct.core.entity.hypergeo.model.HyperGeo;
import ru.yandex.direct.core.entity.hypergeo.model.HyperGeoSegment;
import ru.yandex.direct.core.entity.hypergeo.model.HyperPoint;
import ru.yandex.direct.core.entity.hypergeo.service.HyperGeoService;
import ru.yandex.direct.core.entity.uac.model.DeviceType;
import ru.yandex.direct.core.entity.uac.model.InventoryType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.geobasehelper.GeoBaseHelper;
import ru.yandex.direct.grid.processing.model.inventori.GdBannerFormatIncreasePercent;
import ru.yandex.direct.grid.processing.model.inventori.GdBidModifierDemographic;
import ru.yandex.direct.grid.processing.model.inventori.GdBlockSize;
import ru.yandex.direct.grid.processing.model.inventori.GdError;
import ru.yandex.direct.grid.processing.model.inventori.GdImpressionLimit;
import ru.yandex.direct.grid.processing.model.inventori.GdIncomeGrade;
import ru.yandex.direct.grid.processing.model.inventori.GdIndoorReachRequest;
import ru.yandex.direct.grid.processing.model.inventori.GdMobileOsType;
import ru.yandex.direct.grid.processing.model.inventori.GdOutdoorReachRequest;
import ru.yandex.direct.grid.processing.model.inventori.GdPageBlock;
import ru.yandex.direct.grid.processing.model.inventori.GdPlatformCorrectionsWeb;
import ru.yandex.direct.grid.processing.model.inventori.GdReachBudgetInfo;
import ru.yandex.direct.grid.processing.model.inventori.GdReachIndoorResult;
import ru.yandex.direct.grid.processing.model.inventori.GdReachInfo;
import ru.yandex.direct.grid.processing.model.inventori.GdReachMultiBudgetRequest;
import ru.yandex.direct.grid.processing.model.inventori.GdReachMultiBudgetsResult;
import ru.yandex.direct.grid.processing.model.inventori.GdReachOutdoorResult;
import ru.yandex.direct.grid.processing.model.inventori.GdReachRecommendationResult;
import ru.yandex.direct.grid.processing.model.inventori.GdReachRequest;
import ru.yandex.direct.grid.processing.model.inventori.GdReachResult;
import ru.yandex.direct.grid.processing.model.inventori.GdUacDeviceType;
import ru.yandex.direct.grid.processing.model.inventori.GdUacForecastResponse;
import ru.yandex.direct.grid.processing.model.inventori.GdUacInventoryType;
import ru.yandex.direct.grid.processing.model.inventori.GdUacReachRequest;
import ru.yandex.direct.grid.processing.model.inventori.GdUacRecommendationRequest;
import ru.yandex.direct.grid.processing.model.inventori.GdUacRecommendationResponse;
import ru.yandex.direct.grid.processing.model.inventori.GdUacStrategy;
import ru.yandex.direct.grid.processing.model.inventori.GdVideoCreative;
import ru.yandex.direct.grid.processing.model.retargeting.GdGoalMinimal;
import ru.yandex.direct.grid.processing.model.retargeting.GdRetargetingConditionRuleItemReq;
import ru.yandex.direct.grid.processing.model.retargeting.GdRetargetingConditionRuleType;
import ru.yandex.direct.grid.processing.model.touchsocdem.GdTouchSocdemAgePoint;
import ru.yandex.direct.grid.processing.model.touchsocdem.GdTouchSocdemGender;
import ru.yandex.direct.inventori.model.request.GroupType;
import ru.yandex.direct.inventori.model.response.MultiBudgetsPredictionResponse;
import ru.yandex.direct.utils.CollectionUtils;
import ru.yandex.direct.web.core.entity.inventori.model.BidModifierDemographicWeb;
import ru.yandex.direct.web.core.entity.inventori.model.BlockSize;
import ru.yandex.direct.web.core.entity.inventori.model.CampaignStrategy;
import ru.yandex.direct.web.core.entity.inventori.model.Condition;
import ru.yandex.direct.web.core.entity.inventori.model.GeneralCampaignPredictionSuccessResult;
import ru.yandex.direct.web.core.entity.inventori.model.GeneralCpmRecommendationRequest;
import ru.yandex.direct.web.core.entity.inventori.model.GeneralCpmRecommendationSuccessResult;
import ru.yandex.direct.web.core.entity.inventori.model.Goal;
import ru.yandex.direct.web.core.entity.inventori.model.ImpressionLimit;
import ru.yandex.direct.web.core.entity.inventori.model.MobileOsTypeWeb;
import ru.yandex.direct.web.core.entity.inventori.model.PageBlockWeb;
import ru.yandex.direct.web.core.entity.inventori.model.PlatformCorrectionsWeb;
import ru.yandex.direct.web.core.entity.inventori.model.ReachIndoorRequest;
import ru.yandex.direct.web.core.entity.inventori.model.ReachIndoorResult;
import ru.yandex.direct.web.core.entity.inventori.model.ReachInfo;
import ru.yandex.direct.web.core.entity.inventori.model.ReachMultiBudgetResult;
import ru.yandex.direct.web.core.entity.inventori.model.ReachOutdoorRequest;
import ru.yandex.direct.web.core.entity.inventori.model.ReachOutdoorResult;
import ru.yandex.direct.web.core.entity.inventori.model.ReachRecommendationResult;
import ru.yandex.direct.web.core.entity.inventori.model.ReachRequest;
import ru.yandex.direct.web.core.entity.inventori.model.ReachResult;
import ru.yandex.direct.web.core.entity.inventori.model.VideoCreativeWeb;
import ru.yandex.direct.web.core.model.retargeting.RetargetingConditionRuleType;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.ObjectUtils.defaultIfNull;
import static ru.yandex.direct.core.entity.crypta.utils.CryptaSegmentBrandSafetyUtils.makeBrandSafetyCategories;
import static ru.yandex.direct.grid.processing.model.inventori.GdIncomeGrade.HIGH;
import static ru.yandex.direct.grid.processing.model.inventori.GdIncomeGrade.LOW;
import static ru.yandex.direct.grid.processing.model.inventori.GdIncomeGrade.MIDDLE;
import static ru.yandex.direct.grid.processing.model.inventori.GdIncomeGrade.PREMIUM;
import static ru.yandex.direct.grid.processing.model.touchsocdem.GdTouchSocdemAgePoint.AGE_0;
import static ru.yandex.direct.grid.processing.model.touchsocdem.GdTouchSocdemAgePoint.AGE_18;
import static ru.yandex.direct.grid.processing.model.touchsocdem.GdTouchSocdemAgePoint.AGE_25;
import static ru.yandex.direct.grid.processing.model.touchsocdem.GdTouchSocdemAgePoint.AGE_35;
import static ru.yandex.direct.grid.processing.model.touchsocdem.GdTouchSocdemAgePoint.AGE_45;
import static ru.yandex.direct.grid.processing.model.touchsocdem.GdTouchSocdemAgePoint.AGE_55;
import static ru.yandex.direct.grid.processing.model.touchsocdem.GdTouchSocdemAgePoint.AGE_INF;
import static ru.yandex.direct.utils.CollectionUtils.isEmpty;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;

@Component
public class InventoriDataConverter {

    public static final Map<GdTouchSocdemGender, Long> GENDER_TO_CRYPTA =
            ImmutableMap.<GdTouchSocdemGender, Long>builder()
                    .put(GdTouchSocdemGender.MALE, 2499000001L)
                    .put(GdTouchSocdemGender.FEMALE, 2499000002L)
                    .build();

    public static final LinkedHashMap<GdTouchSocdemAgePoint, Long> AGE_TO_CRYPTA = new LinkedHashMap<>();

    static {
        // важен порядок добавления элементов в мапу
        AGE_TO_CRYPTA.put(AGE_0, 2499000003L);
        AGE_TO_CRYPTA.put(AGE_18, 2499000004L);
        AGE_TO_CRYPTA.put(AGE_25, 2499000005L);
        AGE_TO_CRYPTA.put(AGE_35, 2499000006L);
        AGE_TO_CRYPTA.put(AGE_45, 2499000007L);
        AGE_TO_CRYPTA.put(AGE_55, 2499000008L);
    }

    public static final LinkedHashMap<GdIncomeGrade, Long> INCOME_TO_CRYPTA = new LinkedHashMap<>();

    static {
        // важен порядок добавления элементов в мапу
        INCOME_TO_CRYPTA.put(LOW, 2499000009L);
        INCOME_TO_CRYPTA.put(MIDDLE, 2499000010L);
        INCOME_TO_CRYPTA.put(HIGH, 2499000012L);
        INCOME_TO_CRYPTA.put(PREMIUM, 2499000013L);
    }

    private final GeoBaseHelper geoBaseHelper;
    private final HyperGeoService hyperGeoService;
    private final CryptaSegmentRepository cryptaSegmentRepository;

    @Autowired
    public InventoriDataConverter(GeoBaseHelper geoBaseHelper, HyperGeoService hyperGeoService, CryptaSegmentRepository cryptaSegmentRepository) {
        this.geoBaseHelper = geoBaseHelper;
        this.hyperGeoService = hyperGeoService;
        this.cryptaSegmentRepository = cryptaSegmentRepository;
    }

    private PlatformCorrectionsWeb convertPlatformCorrectionsWebFromGd(GdPlatformCorrectionsWeb model) {
        if (model == null) {
            return null;
        }

        /*
         * Формат значения корректировки на фронте(-100..1200) отличается от принимаемого инвентори(0..1300)
         * Поэтому прибавляем +100 к значению корректировки при конвертации
         *
         * Ссылка на вики инвентори с описанием входных данных
         * https://wiki.yandex-team.ru/inventori/api/campaign/group/group_corrections/
         */
        return new PlatformCorrectionsWeb(
                Optional.ofNullable(model.getDesktop()).map(i -> i + 100).orElse(null),
                Optional.ofNullable(model.getMobile()).map(i -> i + 100).orElse(null),
                convertMobileOsTypeFromGd(model.getMobileOsType()));
    }

    private MobileOsTypeWeb convertMobileOsTypeFromGd(GdMobileOsType osType) {
        if (osType == null) {
            return null;
        }
        switch (osType) {
            case IOS:
                return MobileOsTypeWeb.IOS;
            case ANDROID:
                return MobileOsTypeWeb.ANDROID;
            default:
                throw new IllegalArgumentException("Unexpected enum value " + osType);
        }
    }

    private RetargetingConditionRuleType convertRetargetingConditionRuleTypeFromGd(GdRetargetingConditionRuleType type) {
        switch (type) {
            case OR:
                return RetargetingConditionRuleType.or;
            case NOT:
                return RetargetingConditionRuleType.not;
            case ALL:
                return RetargetingConditionRuleType.all;
            default:
                throw new IllegalArgumentException("Unexpected enum value " + type);
        }
    }

    private List<GdError> convertErrorListToGd(List<ru.yandex.direct.web.core.entity.inventori.model.Error> list) {
        return mapList(list, this::convertErrorToGd);
    }

    private GdError convertErrorToGd(ru.yandex.direct.web.core.entity.inventori.model.Error model) {
        return new GdError()
                .withGoalId(model.getGoalId())
                .withType(model.getType());
    }

    private List<VideoCreativeWeb> convertVideoCreativeListFromGd(List<GdVideoCreative> list) {
        return mapList(list, this::convertVideoCreativeFromGd);
    }

    private VideoCreativeWeb convertVideoCreativeFromGd(GdVideoCreative model) {
        return new VideoCreativeWeb(model.getDuration());
    }

    private List<Goal> convertGoalListFromGd(List<GdGoalMinimal> list) {
        return mapList(list, this::convertGoalFromGd);
    }

    private Goal convertGoalFromGd(GdGoalMinimal model) {
        return new Goal(model.getId(), model.getTime());
    }

    private BlockSize convertBlockSizeFromGd(GdBlockSize b) {
        return new BlockSize(b.getWidth(), b.getHeight());
    }

    private List<Condition> converConditionListFromGd(List<GdRetargetingConditionRuleItemReq> skills) {
        return mapList(skills, this::converConditionFromGd);
    }

    private Condition converConditionFromGd(GdRetargetingConditionRuleItemReq model) {
        return new Condition(convertRetargetingConditionRuleTypeFromGd(model.getType()),
                convertGoalListFromGd(model.getGoals()))
                .withInterestType(model.getInterestType());
    }

    public ReachRequest convertReachRequestFromGd(GdReachRequest input) {
        return new ReachRequest(input.getGeo(), input.getAdgroupId(),
                mapList(input.getBlockSizes(), this::convertBlockSizeFromGd),
                convertVideoCreativeListFromGd(input.getVideoCreatives()), null,
                converConditionListFromGd(input.getConditions()),
                input.getCampaignId(), input.getExcludedDomains(),
                convertPlatformCorrectionsWebFromGd(input.getPlatformCorrectionsWeb()), input.getGroupType(),
                input.getTargetTags());
    }

    public ReachRequest convertReachRequestFromGd(GdReachMultiBudgetRequest input, ClientId clientId) {
        List<Condition> conditions = toConditions(input.getGenders(), input.getAgeLower(), input.getAgeUpper(),
                input.getConditions());

        return new ReachRequest()
                .withCampaignId(defaultIfNull(input.getCampaignId(), 0L))
                .withAdgroupId(input.getAdgroupId())
                .withGroupType(GroupType.BANNER)
                .withHasAdaptiveCreative(true)
                .withGeo(convertGeo(input, clientId))
                .withConditions(conditions);
    }

    public List<Condition> toConditions(List<GdTouchSocdemGender> genders, GdTouchSocdemAgePoint ageLower,
            GdTouchSocdemAgePoint ageUpper, List<GdRetargetingConditionRuleItemReq> gdConditions)
    {
        List<Condition> conditions = converConditionListFromGd(gdConditions);
        // Если есть оба пола, то их не передаем
        if (!isEmpty(genders) && genders.size() == 1) {
            conditions.add(toGenderCondition(genders));
        }

        // Если возраст от 0 до бесконечности, то его не передаем
        if (ageLower != AGE_0 || ageUpper != AGE_INF) {
            conditions.add(toAgeCondition(ageLower, ageUpper));
        }
        return conditions;
    }

    public GeneralCpmRecommendationRequest convertUacRecommendationRequest(GdUacRecommendationRequest input) {
        List<Condition> conditions = getConditions(input);

        GdUacStrategy uacStrategyData = input.getStrategy();


        List<String> excludedBsCategories = null;
        if (input.getBrandSafety() != null && nvl(input.getBrandSafety().getEnabled(), false)) {
            excludedBsCategories =
                    makeExcludedBsCategories(input.getBrandSafety().getAdditionalCategories());
        }

        GdImpressionLimit impressionLimit = uacStrategyData.getImpressionLimit();
        return new GeneralCpmRecommendationRequest()
                .withStrategy(new CampaignStrategy()
                        .withImpressionLimit(new ImpressionLimit()
                                .withDays(impressionLimit.getDays())
                                .withImpressions(impressionLimit.getImpressions()))
                        .withBudget(ifNotNull(uacStrategyData.getBudget(), BigDecimal::doubleValue))
                        .withCpm(ifNotNull(uacStrategyData.getAvgCpm(), BigDecimal::doubleValue))
                        .withCpv(ifNotNull(uacStrategyData.getAvgCpv(), BigDecimal::doubleValue))
                        .withStartDate(uacStrategyData.getStartDate())
                        .withEndDate(uacStrategyData.getFinishDate())
                        .withType(input.getStrategyName().getTypedValue().name())
                )
                .withCpmCampaignType(input.getCampaignType())
                .withGroupType(input.getAdGroupType())
                .withCampaignId(input.getCampaignId())
                .withVideoCreatives(convertVideoCreativeListFromGd(input.getVideoCreatives()))
                .withBlockSizes(mapList(input.getBlockSizes(), this::convertBlockSizeFromGd))
                .withGeo(input.getGeo())
                .withHasBrandlift(input.getHasBrandlift())
                .withConditions(conditions)
                .withInventoryTypes(mapSet(nvl(input.getInventoryTypes(), emptySet()), this::toInventoryTypeFromGd))
                .withDeviceTypes(mapSet(nvl(input.getDeviceTypes(), emptySet()), this::toDeviceTypeFromGd))
                .withExcludedDomains(input.getBlackListDomains())
                .withExcludedBsCategories(excludedBsCategories);
    }

    private List<String> makeExcludedBsCategories(List<Long> additionalCategories) {
        var brandSafetyCategories = makeBrandSafetyCategories(true, nvl(additionalCategories, emptyList()));
        var cryptaGoals = cryptaSegmentRepository.getBrandSafety();
        return StreamEx.of(brandSafetyCategories)
                .filter(cryptaGoals::containsKey)
                .map(cryptaGoals::get)
                .map(goal -> goal.getKeyword() + ":" + goal.getKeywordValue())
                .toList();
    }

    public ReachRequest convertReachRequestFromGd(GdUacReachRequest input) {
        return new ReachRequest(input.getGeo(), input.getAdgroupId(),
                mapList(input.getBlockSizes(), this::convertBlockSizeFromGd),
                convertVideoCreativeListFromGd(input.getVideoCreatives()), null,
                getConditions(input),
                input.getCampaignId(), null,
                null, input.getAdGroupType(),
                null);
    }

    public List<Condition> getConditions(GdUacReachRequest input) {
        List<Condition> conditions = toConditions(input.getGenders(), input.getAgeLower(), input.getAgeUpper(),
                input.getConditions());

        GdIncomeGrade incomeGradeLower = input.getIncomeGradeLower();
        GdIncomeGrade incomeGradeUpper = input.getIncomeGradeUpper();
        if (incomeGradeLower != LOW || incomeGradeUpper != PREMIUM) {
            Condition incomeCondition = toIncomeCondition(incomeGradeLower, incomeGradeUpper);
            conditions.add(incomeCondition);
        }
        return conditions;
    }

    private Set<Integer> convertGeo(GdReachMultiBudgetRequest input, ClientId clientId) {
        Set<Integer> geo = input.getGeo();
        Long hyperGeoId = input.getHyperGeoId();

        if (!CollectionUtils.isEmpty(geo)) {
            return geo;
        }

        return convertHyperGeoToGeo(hyperGeoId, clientId).orElse(geo);
    }

    private Optional<Set<Integer>> convertHyperGeoToGeo(Long hyperGeoId, ClientId clientId) {
        if (hyperGeoId == null) {
            return Optional.empty();
        }

        HyperGeo hyperGeo = hyperGeoService.getHyperGeoById(clientId, List.of(hyperGeoId)).get(hyperGeoId);
        Collection<HyperGeoSegment> hyperGeoSegments = hyperGeo == null ? emptyList() : hyperGeo.getHyperGeoSegments();
        if (hyperGeoSegments.isEmpty()) {
            return Optional.empty();
        }

        // В тачах сейчас можно создавать только одну окружность в сегменте и только один сегмент на группу
        checkState(hyperGeoSegments.size() == 1);
        HyperGeoSegment hyperGeoSegment = hyperGeoSegments.iterator().next();
        List<HyperPoint> hyperPoints = hyperGeoSegment.getSegmentDetails().getPoints();
        checkState(hyperPoints.size() == 1);
        HyperPoint hyperPoint = hyperPoints.get(0);

        Optional<Long> mostSuitableRegionId =
                geoBaseHelper.getDirectRegionId(hyperPoint.getLatitude(), hyperPoint.getLongitude());

        return mostSuitableRegionId
                .map(Long::intValue)
                .map(Set::of);
    }

    private Condition toIncomeCondition(GdIncomeGrade incomeLower, GdIncomeGrade incomeUpper) {
        return getCondition(incomeLower, incomeUpper, INCOME_TO_CRYPTA);
    }

    private Condition toAgeCondition(GdTouchSocdemAgePoint ageLower, GdTouchSocdemAgePoint ageUpper) {
        return getCondition(ageLower, ageUpper, AGE_TO_CRYPTA);
    }

    private <T> Condition getCondition(T lower, T upper, LinkedHashMap<T, Long> xToCrypta) {
        List<Goal> goals = new ArrayList<>();
        boolean lowerFound = false;
        for (Map.Entry<T, Long> entry : xToCrypta.entrySet()) {
            if (entry.getKey() == upper) {
                break;
            }

            if (entry.getKey() == lower) {
                lowerFound = true;
            }

            if (lowerFound) {
                goals.add(new Goal().withId(entry.getValue()));
            }
        }

        return new Condition().withType(RetargetingConditionRuleType.or).withGoals(goals);
    }

    private Condition toGenderCondition(List<GdTouchSocdemGender> genders) {
        return new Condition().withType(RetargetingConditionRuleType.or).withGoals(mapList(genders,
                g -> new Goal().withId(GENDER_TO_CRYPTA.get(g))));
    }

    public ReachOutdoorRequest convertReachRequestFromGd(GdOutdoorReachRequest input) {
        return new ReachOutdoorRequest()
                .withAdgroupId(input.getAdGroupId())
                .withCampaignId(input.getCampaignId())
                .withVideoCreativeIds(input.getVideoCreativeIds())
                .withPageBlocks(ifNotNull(input.getPageBlocks(), pageBlocks -> mapList(pageBlocks,
                        this::toPageBlockWeb)));
    }

    public ReachIndoorRequest convertReachRequestFromGd(GdIndoorReachRequest input) {
        return new ReachIndoorRequest()
                .withAdgroupId(input.getAdGroupId())
                .withCampaignId(input.getCampaignId())
                .withVideoCreativeIds(input.getVideoCreativeIds())
                .withPageBlocks(ifNotNull(input.getPageBlocks(), pageBlocks -> mapList(pageBlocks,
                        this::toPageBlockWeb)))
                .withBidModifierDemographics(ifNotNull(input.getBidModifierDemographic(),
                        bmd -> mapList(bmd, this::toBidModifierDemographicWeb)));
    }

    private BidModifierDemographicWeb toBidModifierDemographicWeb(GdBidModifierDemographic item) {
        return new BidModifierDemographicWeb(item.getGender(), item.getAge(), item.getMultiplier());
    }

    private PageBlockWeb toPageBlockWeb(GdPageBlock pb) {
        return new PageBlockWeb(pb.getPageId(), pb.getBlockIds());
    }

    private GdReachInfo convertReachInfoToGd(ReachInfo model) {
        return model == null ? null : new GdReachInfo()
                .withReach(model.getReach())
                .withReachLessThan(model.getReachLessThan())
                .withErrors(convertErrorListToGd(model.getErrors()));
    }

    public GdReachResult convertReachResultToGd(ReachResult model) {
        return new GdReachResult()
                .withRequestId(model.getRequestId())
                .withBasic(convertReachInfoToGd(model.getBasic()))
                .withDetailed(convertReachInfoToGd(model.getDetailed()));
    }

    public GdUacRecommendationResponse convertResultToGd(GeneralCpmRecommendationSuccessResult result, String requestId) {
        return new GdUacRecommendationResponse()
                .withInventoriRequestId(requestId)
                .withTrafficLightColor(result.getTrafficLightColor())
                .withRecommendedPrice(result.getRecommendedPrice())
                .withTargetEventsLeftBorder(result.getTargetEventsLeftBorder())
                .withTargetEventsRightBorder(result.getTargetEventsRightBorder())
                .withReachLeftBorder(result.getReachLeftBorder())
                .withReachRightBorder(result.getReachRightBorder());
    }

    public GdUacForecastResponse convertGeneralCampaignPredictionResultToGd(
            GeneralCampaignPredictionSuccessResult result, String requestId) {
        return new GdUacForecastResponse()
                .withInventoriRequestId(requestId)
                .withTargetReach(result.getTargetReach())
                .withForecastReach(result.getForecastReach())
                .withForecastShows(result.getForecastShows())
                .withSovByShows(result.getSovByShows())
                .withSovByReach(result.getSovByReach())
                .withForecastBudget(result.getForecastBudget())
                .withRecommendedPrice(result.getRecommendedPrice())
                .withTrafficLightColour(result.getTrafficLightColour());
    }

    public GdReachMultiBudgetsResult convertParametrisedCampaignPredictionResponseToGd(
            ReachMultiBudgetResult model) {
        MultiBudgetsPredictionResponse response = model.getMultiBudgetsPredictionResponse();
        return new GdReachMultiBudgetsResult()
                .withRequestId(model.getRequestId())
                .withTotalReach(response.getPredictionTargetReach())
                .withReaches(mapList(response.getIntervalPredictionResults(),
                        interval -> new GdReachBudgetInfo()
                                .withShowReachLessThan(false)
                                .withBudget(interval.getBudget() / 1_000_000)
                                .withReach(interval.getPredictionReach())
                                .withLeftIntervalReach(interval.getLeftPredictionReachBorder())
                                .withRightIntervalReach(interval.getRightPredictionReachBorder())
                ));
    }

    public GdReachOutdoorResult convertReachResultToGd(ReachOutdoorResult model) {
        return new GdReachOutdoorResult()
                .withRequestId(model.getRequestId())
                .withReach(model.getReach())
                .withReachLessThan(model.getReachLessThan())
                .withOtsCapacity(model.getOtsCapacity());
    }

    public GdReachIndoorResult convertReachResultToGd(ReachIndoorResult model) {
        return new GdReachIndoorResult()
                .withRequestId(model.getRequestId())
                .withReach(model.getReach())
                .withReachLessThan(model.getReachLessThan())
                .withOtsCapacity(model.getOtsCapacity());
    }

    public GdReachRecommendationResult convertReachRecommendationResultToGd(ReachRecommendationResult model) {
        return new GdReachRecommendationResult()
                .withRequestId(model.getRequestId())
                .withBannerFormats(model.getBannerFormatIncreasePercent() == null ? null :
                        model.getBannerFormatIncreasePercent().stream().map(
                                it -> new GdBannerFormatIncreasePercent()
                                        .withFormat(it.getFormat())
                                        .withIncreasePercent(it.getIncreasePercent()))
                                .collect(toList()));
    }

    public InventoryType toInventoryTypeFromGd(GdUacInventoryType gdUacInventoryType) {
        switch (gdUacInventoryType) {
            case INAPP:
                return InventoryType.INAPP;
            case INPAGE:
                return InventoryType.INPAGE;
            case INSTREAM:
                return InventoryType.INSTREAM;
            case REWARDED:
                return InventoryType.REWARDED;
        }
        return null;
    }

    public DeviceType toDeviceTypeFromGd(GdUacDeviceType gdUacDeviceType) {
        switch (gdUacDeviceType) {
            case PHONE:
                return DeviceType.PHONE;
            case PHONE_IOS:
                return DeviceType.PHONE_IOS;
            case PHONE_ANDROID:
                return DeviceType.PHONE_ANDROID;
            case DESKTOP:
                return DeviceType.DESKTOP;
            case SMART_TV:
                return DeviceType.SMART_TV;
            case TABLET:
                return DeviceType.TABLET;
        }
        return null;
    }
}
