package ru.yandex.direct.grid.processing.service.jsonsettings;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import io.leangen.graphql.annotations.GraphQLArgument;
import io.leangen.graphql.annotations.GraphQLMutation;
import io.leangen.graphql.annotations.GraphQLNonNull;
import io.leangen.graphql.annotations.GraphQLQuery;
import io.leangen.graphql.annotations.GraphQLRootContext;

import ru.yandex.direct.core.security.authorization.PreAuthorizeRead;
import ru.yandex.direct.grid.core.frontdb.jsonsettings.service.JsonSettingsService;
import ru.yandex.direct.grid.model.jsonsettings.GdGetJsonSettings;
import ru.yandex.direct.grid.model.jsonsettings.GdSetJsonSettings;
import ru.yandex.direct.grid.model.jsonsettings.IdType;
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.jsonsettings.GdGetJsonSettingsPayload;
import ru.yandex.direct.grid.processing.model.jsonsettings.GdSetJsonSettingsPayload;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;

@GridGraphQLService
@ParametersAreNonnullByDefault
public class JsonSettingsGraphQlService {

    private final JsonSettingsService jsonSettingsService;
    private final GridValidationService gridValidationService;

    public JsonSettingsGraphQlService(JsonSettingsService jsonSettingsService,
                                      GridValidationService gridValidationService) {
        this.jsonSettingsService = jsonSettingsService;
        this.gridValidationService = gridValidationService;
    }

    @GraphQLNonNull
    @PreAuthorizeRead
    @GraphQLMutation(name = "setJsonSettings")
    public GdSetJsonSettingsPayload setJsonSettings(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdSetJsonSettings input) {
        gridValidationService.validateSetJsonSettings(input);

        Long operatorUid = context.getOperator().getUid();
        Long clientId = context.getSubjectUser() != null ? context.getSubjectUser().getClientId().asLong() : null;
        if (clientId == null) {
            gridValidationService.validateSetJsonSettingsClientIdIsNull(input);
        }
        String result;
        try (TraceProfile profile = Trace.current().profile("json_settings_service:set_json_settings")) {
            result = jsonSettingsService.setJsonSettings(clientId, operatorUid, input);
        }


        return new GdSetJsonSettingsPayload()
                .withClientId(input.getIdType() == IdType.UID ? null : clientId)
                .withUid(input.getIdType() == IdType.CLIENT_ID ? null : operatorUid)
                .withSettings(result);
    }

    @GraphQLNonNull
    @PreAuthorizeRead
    @GraphQLQuery(name = "getJsonSettings")
    public GdGetJsonSettingsPayload getJsonSettings(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdGetJsonSettings input) {
        gridValidationService.validateGetJsonSettings(input);

        Long operatorUid = context.getOperator().getUid();
        Long clientId = context.getSubjectUser() != null ? context.getSubjectUser().getClientId().asLong() : null;
        if (clientId == null) {
            gridValidationService.validateGetJsonSettingsClientIdIsNull(input);
        }
        List<String> result;
        try (TraceProfile profile = Trace.current().profile("json_settings_service:get_json_settings")) {
            result = jsonSettingsService.getJsonSettings(clientId, operatorUid, input);
        }

        return new GdGetJsonSettingsPayload()
                .withClientId(input.getIdType() == IdType.UID ? null : clientId)
                .withUid(input.getIdType() == IdType.CLIENT_ID ? null : operatorUid)
                .withSettings(result);
    }
}
