package ru.yandex.direct.grid.processing.service.landing

import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.landing.BizLandingService
import ru.yandex.direct.core.entity.landing.model.BizLanding
import ru.yandex.direct.grid.processing.model.landing.GdAddBizLanding
import ru.yandex.direct.grid.processing.model.landing.GdAddBizLandingPayload
import ru.yandex.direct.grid.processing.model.landing.GdBizLanding
import ru.yandex.direct.grid.processing.model.landing.GdGetBizLanding
import ru.yandex.direct.grid.processing.model.landing.GdUpdateBizLanding
import ru.yandex.direct.grid.processing.model.landing.GdUpdateBizLandingPayload
import ru.yandex.direct.grid.processing.service.landing.converter.LandingImageType
import ru.yandex.direct.grid.processing.service.landing.converter.patchImageUrlFromLandlord
import ru.yandex.direct.grid.processing.service.landing.converter.patchImageUrlToLandlord
import ru.yandex.direct.grid.processing.service.landing.converter.toContacts
import ru.yandex.direct.grid.processing.service.landing.converter.toCta
import ru.yandex.direct.grid.processing.service.landing.converter.toGdContacts
import ru.yandex.direct.grid.processing.service.landing.converter.toGdCta

@Service
class BizLandingDataService(
    private val bizLandingService: BizLandingService,
    private val gdBizLandingValidationService: GdBizLandingValidationService
) {

    fun getBizLanding(operatorUid: Long, input: GdGetBizLanding): GdBizLanding? {
        gdBizLandingValidationService.validateGetInput(input)
        val bizLanding = bizLandingService.getBizLanding(operatorUid, input.id, input.permalink) ?: return null
        return GdBizLanding().apply {
            id = bizLanding.id
            permalink = bizLanding.permalink
            url = bizLanding.url
            name = bizLanding.name
            description = bizLanding.description
            categories = bizLanding.categories
            logoUrl = patchImageUrlFromLandlord(bizLanding.logoUrl, LandingImageType.LOGO)
            coverUrl = patchImageUrlFromLandlord(bizLanding.coverUrl, LandingImageType.COVER)
            cta = toGdCta(bizLanding.cta)
            contacts = toGdContacts(bizLanding.contacts)
        }
    }

    fun updateBizLanding(
        operatorUid: Long,
        input: GdUpdateBizLanding
    ): GdUpdateBizLandingPayload {
        val landing = BizLanding().apply {
            id = input.id
            permalink = input.permalink
            name = input.name
            description = input.description
            categories = input.categories
            logoUrl = patchImageUrlToLandlord(input.logoUrl, LandingImageType.LOGO)
            coverUrl = patchImageUrlToLandlord(input.coverUrl, LandingImageType.COVER)
            cta = toCta(input.cta)
            contacts = toContacts(input.contacts)
        }
        val result = bizLandingService.updateBizLanding(operatorUid, landing)
        if (!result.isSuccessful) {
            return GdUpdateBizLandingPayload()
                .withValidationResult(gdBizLandingValidationService.toGdValidationResult(result.validationResult))
        }
        return GdUpdateBizLandingPayload().withId(result.result)
    }

    fun addBizLanding(
        operatorUid: Long,
        input: GdAddBizLanding
    ): GdAddBizLandingPayload {
        val landing = BizLanding().apply {
            permalink = input.permalink
            name = input.name
            description = input.description
            categories = input.categories
            logoUrl = patchImageUrlToLandlord(input.logoUrl, LandingImageType.LOGO)
            coverUrl = patchImageUrlToLandlord(input.coverUrl, LandingImageType.COVER)
            cta = toCta(input.cta)
            contacts = toContacts(input.contacts)
        }
        val result = bizLandingService.addBizLanding(operatorUid, landing)
        if (!result.isSuccessful) {
            return GdAddBizLandingPayload()
                .withValidationResult(gdBizLandingValidationService.toGdValidationResult(result.validationResult))
        }
        return GdAddBizLandingPayload().withId(result.result)
    }

}
