package ru.yandex.direct.grid.processing.service.landing.converter

import org.slf4j.LoggerFactory
import ru.yandex.direct.core.entity.image.model.AvatarsImage
import ru.yandex.direct.core.entity.landing.model.BizContact
import ru.yandex.direct.core.entity.landing.model.BizCta
import ru.yandex.direct.grid.processing.model.cliententity.image.GdAdImageNamespace
import ru.yandex.direct.grid.processing.model.landing.GdBizContact
import ru.yandex.direct.grid.processing.model.landing.GdBizContactType
import ru.yandex.direct.grid.processing.model.landing.GdBizCta
import ru.yandex.direct.grid.processing.model.landing.GdBizCtaAction
import ru.yandex.direct.grid.processing.service.landing.BizLandingDataService

private val logger = LoggerFactory.getLogger(BizLandingDataService::class.java)

private const val LANDLORD_NAMESPACE = "maps-adv-crm"
private const val LANDLORD_IMAGE_URL_TEMPLATE = "%s"

enum class LandingImageType(
    val landlordSizeAlias: String,
    val directSizeAlias: String
) {
    LOGO("landing_logo", "x450"),
    COVER("landing_background", "x1200"),
}

fun toGdCta(cta: BizCta): GdBizCta {
    return GdBizCta().apply {
        action = GdBizCtaAction.fromSource(cta.action)
        custom = cta.custom
        value = cta.value
    }
}

fun toGdContacts(contacts: List<BizContact>) =
    contacts.mapNotNull {
        if (it.value.isNullOrEmpty()) {
            null
        } else {
            GdBizContact().apply {
                this.type = GdBizContactType.fromSource(it.type)
                this.value = it.value
            }
        }
    }

fun toCta(cta: GdBizCta) =
    BizCta().apply {
        action = GdBizCtaAction.toSource(cta.action)
        custom = cta.custom
        value = cta.value
    }

fun toContacts(contacts: List<GdBizContact>) =
    contacts.map { c ->
        BizContact().apply {
            this.type = GdBizContactType.toSource(c.type)
            this.value = c.value
        }
    }

fun patchImageUrlFromLandlord(url: String?, imageType: LandingImageType): String? {
    if (url.isNullOrEmpty()) {
        return null
    }
    val avatarsUrl = AvatarsImage.fromString(url)
    return when (avatarsUrl.namespace) {
        LANDLORD_NAMESPACE -> {
            if (avatarsUrl.size == LANDLORD_IMAGE_URL_TEMPLATE) {
                avatarsUrl.copy(size = imageType.landlordSizeAlias).toString()
            } else {
                url
            }
        }
        GdAdImageNamespace.DIRECT.name.lowercase() -> {
            url
        }
        else -> {
            // На случай если в Бизнесе картинки начнут загружаться в другой неймспейс
            logger.error("Unknown namespace from Landlord image url: $url")
            url
        }
    }
}

fun patchImageUrlToLandlord(url: String?, imageType: LandingImageType): String? {
    if (url.isNullOrEmpty()) {
        return null
    }
    val avatarsUrl = AvatarsImage.fromString(url)
    return when (avatarsUrl.namespace) {
        LANDLORD_NAMESPACE -> {
            if (avatarsUrl.size == imageType.landlordSizeAlias) {
                avatarsUrl.copy(size = LANDLORD_IMAGE_URL_TEMPLATE).toString()
            } else {
                url
            }
        }
        GdAdImageNamespace.DIRECT.name.lowercase() -> {
            avatarsUrl.copy(size = imageType.directSizeAlias).toString()
        }
        else -> {
            url
        }
    }
}
