package ru.yandex.direct.grid.processing.service.menuheader;

import java.util.HashSet;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.grid.processing.model.client.GdMenuItem;
import ru.yandex.direct.rbac.RbacClientsRelations;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.rbac.model.ClientsRelation;
import ru.yandex.direct.rbac.model.ClientsRelationType;

import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.hasOneOfRoles;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isAgency;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isClient;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isDeveloper;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isInternalAdAdmin;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isInternalAdRole;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isLimitedAgency;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isManager;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isSuper;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isSuperReader;

/**
 * Сервис, возвращающий объекты меню, которые нужно отобразить в шапке.
 */
@Service
@ParametersAreNonnullByDefault
public class AllowedMenuItemsService {
    private final RbacService rbacService;
    private final RbacClientsRelations rbacClientsRelations;
    private final ClientService clientService;
    private final FeatureService featureService;

    @Autowired
    public AllowedMenuItemsService(RbacService rbacService,
                                   RbacClientsRelations rbacClientsRelations,
                                   ClientService clientService,
                                   FeatureService featureService) {
        this.rbacService = rbacService;
        this.rbacClientsRelations = rbacClientsRelations;
        this.clientService = clientService;
        this.featureService = featureService;
    }

    /**
     * Получить доступные объекты меню в шапке по включенным фичам.
     * Если оператор управляет клиентом, то доступные объекты меню определяются по объединению множества фичей
     * оператора и клиента
     *
     * @param operator - оператор
     * @param clientId - id пользователя, которым управляет оператор (в случае работы через оператора)
     * @return - доступные объекты меню в шапке
     */
    public Set<GdMenuItem> getAllowedMenuItems(User operator, ClientId clientId) {
        Set<String> operatorFeatures = featureService.getEnabledForClientId(operator.getClientId());
        Set<String> availableFeatures = new HashSet<>(operatorFeatures);

        if (!operator.getClientId().equals(clientId)) {
            Set<String> userFeatures = featureService.getEnabledForClientId(clientId);
            availableFeatures.addAll(userFeatures);
        }

        boolean isInternalAdProduct = rbacService.isInternalAdProduct(clientId);
        boolean isUnderAgency = rbacService.isUnderAgency(operator.getChiefUid());
        boolean isOperatorLinkedClient = operator.getRole().equals(RbacRole.CLIENT) && !operator.getClientId().equals(clientId);

        return Stream.of(GdMenuItem.values())
                .filter(gdMenuItem -> isAllowed(operator, clientId, gdMenuItem, availableFeatures, isUnderAgency,
                        isInternalAdProduct, isOperatorLinkedClient))
                .collect(toSet());
    }

    private boolean isAllowed(User operator, ClientId clientId, GdMenuItem gdMenuItem, Set<String> availableFeatures,
                              boolean isUnderAgency, boolean clientIsInternalAdProduct, boolean isOperatorLinkedClient) {
        switch (gdMenuItem) {
            case ADMIN:
                return isSuper(operator) || isDeveloper(operator);

            case ADMIN_AGENCY:
                return isAgency(operator);

            case ADMIN_MANAGER:
                return isManager(operator);

            case ADVANCED_FORECAST:
                return !isInternalAdRole(operator);

            case AGENCY_SEARCH:
                return isAgency(operator);

            case CREATE:
                return canCreateCampaigns(operator, clientId, isUnderAgency);

            case DEALS_LIST:
                boolean hasDealsFeature = availableFeatures.contains(FeatureName.CPM_DEALS.getName());
                return hasDealsFeature && !isLimitedAgency(operator)
                        && hasOneOfRoles(operator, RbacRole.SUPER, RbacRole.SUPERREADER, RbacRole.AGENCY);

            case DOCUMENTS_AND_PAYMENTS:
                return hasOneOfRoles(operator, RbacRole.CLIENT, RbacRole.SUPPORT, RbacRole.LIMITED_SUPPORT)
                        && !isUnderAgency && !isOperatorLinkedClient && !operator.getIsReadonlyRep();

            case INTERNAL_AD_MANAGE_MARKETERS:
                return isInternalAdAdmin(operator) ||
                        (clientIsInternalAdProduct && (isSuper(operator) || isSuperReader(operator)));

            case INTERNAL_AD_MANAGE_PRODUCTS:
                return clientIsInternalAdProduct || isInternalAdRole(operator);

            case INTERNAL_AD_SEARCH:
                return clientIsInternalAdProduct || isInternalAdRole(operator);

            case RECOMMENDATIONS:
                return !isInternalAdRole(operator)
                        && !clientIsInternalAdProduct;

            case SHOW_CAMPS:
                return isClient(operator);

            case SHOW_AGENCY_CLIENTS:
                return isAgency(operator);

            case SHOW_MANAGER_CLIENTS:
                return isManager(operator);

            case SHOW_SEARCH_PAGE:
                return hasOneOfRoles(operator, RbacRole.SUPER, RbacRole.SUPERREADER, RbacRole.SUPPORT, RbacRole.MANAGER,
                        RbacRole.PLACER, RbacRole.MEDIA, RbacRole.LIMITED_SUPPORT);

            case WORDSTAT:
                return !isInternalAdRole(operator);

            case TURBO_LANDING_CONSTRUCTOR:
                return true;

            case VIDEO_CONSTRUCTOR:
                return !isInternalAdRole(operator)
                        && availableFeatures.contains(FeatureName.VIDEO_CONSTRUCTOR_ENABLED.getName());

            case SHOW_MINUS_KEYWORDS_LIB:
                return !isInternalAdRole(operator)
                        && !clientIsInternalAdProduct
                        && availableFeatures.contains(FeatureName.MINUS_WORDS_LIB.getName());

            case MULTI_CLIENTS_IN_STAT:
                return availableFeatures.contains(FeatureName.MULTI_CLIENTS_IN_STAT_ALLOWED.getName());

            default:
                throw new IllegalArgumentException("Unknown menu item: " + gdMenuItem.name());
        }
    }

    private boolean canCreateCampaigns(User operator, ClientId clientId, boolean isUnderAgency) {
        if (operator.getIsReadonlyRep()) {
            return false;
        }

        if (isInternalAdRole(operator)) {
            if (!rbacService.isInternalAdProduct(clientId)) {
                return false;
            }

            switch (operator.getRole()) {
                case INTERNAL_AD_MANAGER:
                    Optional<ClientsRelationType> relationType = rbacClientsRelations
                            .getInternalAdProductRelation(operator.getClientId(), clientId)
                            .map(ClientsRelation::getRelationType);

                    return relationType.isPresent() &&
                            relationType.get() == ClientsRelationType.INTERNAL_AD_PUBLISHER;

                case INTERNAL_AD_ADMIN:
                    return true;

                case INTERNAL_AD_SUPERREADER:
                    return false;

                default:
                    throw new IllegalStateException("unknown internal ad role: " + operator.getRole());
            }
        }

        if (!hasOneOfRoles(operator, RbacRole.CLIENT, RbacRole.MANAGER, RbacRole.AGENCY)) {
            return false;
        }

        if (isUnderAgency) {
            return clientService.isSuperSubclient(operator.getClientId());
        }

        return true;
    }
}
