package ru.yandex.direct.grid.processing.service.minuskeywordspack;

import java.util.List;
import java.util.Optional;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseValidator;
import ru.yandex.direct.core.entity.minuskeywordspack.container.AddedMinusKeywordsPackInfo;
import ru.yandex.direct.core.entity.minuskeywordspack.container.UpdatedMinusKeywordsPackInfo;
import ru.yandex.direct.core.entity.minuskeywordspack.model.MinusKeywordsPack;
import ru.yandex.direct.core.entity.minuskeywordspack.repository.MinusKeywordsPackRepository;
import ru.yandex.direct.core.entity.minuskeywordspack.service.MinusKeywordsPacksAddOperation;
import ru.yandex.direct.core.entity.minuskeywordspack.service.MinusKeywordsPacksAddOperationFactory;
import ru.yandex.direct.core.entity.minuskeywordspack.service.MinusKeywordsPacksDeleteOperation;
import ru.yandex.direct.core.entity.minuskeywordspack.service.MinusKeywordsPacksDeleteOperationFactory;
import ru.yandex.direct.core.entity.minuskeywordspack.service.MinusKeywordsPacksUpdateOperation;
import ru.yandex.direct.core.entity.minuskeywordspack.service.MinusKeywordsPacksUpdateOperationFactory;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.grid.processing.model.api.GdValidationResult;
import ru.yandex.direct.grid.processing.model.minuskeywordspack.GdGetMinusKeywordsPacks;
import ru.yandex.direct.grid.processing.model.minuskeywordspack.GdGetMinusKeywordsPacksItem;
import ru.yandex.direct.grid.processing.model.minuskeywordspack.GdGetMinusKeywordsPacksPayload;
import ru.yandex.direct.grid.processing.model.minuskeywordspack.mutation.GdAddMinusKeywordsPacks;
import ru.yandex.direct.grid.processing.model.minuskeywordspack.mutation.GdAddMinusKeywordsPacksPayload;
import ru.yandex.direct.grid.processing.model.minuskeywordspack.mutation.GdAddMinusKeywordsPacksPayloadItem;
import ru.yandex.direct.grid.processing.model.minuskeywordspack.mutation.GdDeleteMinusKeywordsPacks;
import ru.yandex.direct.grid.processing.model.minuskeywordspack.mutation.GdDeleteMinusKeywordsPacksPayload;
import ru.yandex.direct.grid.processing.model.minuskeywordspack.mutation.GdUpdateMinusKeywordsPacks;
import ru.yandex.direct.grid.processing.model.minuskeywordspack.mutation.GdUpdateMinusKeywordsPacksPayload;
import ru.yandex.direct.grid.processing.model.minuskeywordspack.mutation.GdUpdateMinusKeywordsPacksPayloadItem;
import ru.yandex.direct.grid.processing.service.cache.GridCacheService;
import ru.yandex.direct.grid.processing.service.minuskeywordspack.container.GetMinusKeywordsPacksCacheRecordInfo;
import ru.yandex.direct.grid.processing.service.minuskeywordspack.converter.MinusKeywordsPackDataConverter;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.result.MassResult;

import static java.util.function.Function.identity;
import static ru.yandex.direct.grid.processing.service.cache.util.CacheUtils.normalizeLimitOffset;
import static ru.yandex.direct.grid.processing.service.minuskeywordspack.converter.MinusKeywordsPackDataConverter.getCacheRecordInfo;
import static ru.yandex.direct.grid.processing.service.minuskeywordspack.converter.MinusKeywordsPackDataConverter.toCoreMinusKeywordsPackChanges;
import static ru.yandex.direct.grid.processing.service.minuskeywordspack.converter.MinusKeywordsPackDataConverter.toCoreMinusKeywordsPacks;
import static ru.yandex.direct.grid.processing.service.minuskeywordspack.converter.MinusKeywordsPackDataConverter.toGdGetMinusKeywordsPacksItemList;
import static ru.yandex.direct.grid.processing.util.ResponseConverter.getResults;
import static ru.yandex.direct.grid.processing.util.ResultConverterHelper.getSuccessfullyUpdatedIds;
import static ru.yandex.direct.operation.Applicability.PARTIAL;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.PathHelper.path;

@Service
@ParametersAreNonnullByDefault
public class MinusKeywordsPackDataService {

    private final MinusKeywordsPacksAddOperationFactory minusKeywordsPacksAddOperationFactory;
    private final MinusKeywordsPacksUpdateOperationFactory minusKeywordsPacksUpdateOperationFactory;
    private final MinusKeywordsPacksDeleteOperationFactory minusKeywordsPacksDeleteOperationFactory;
    private final GridValidationService gridValidationService;
    private final ShardHelper shardHelper;
    private final MinusKeywordsPackRepository packRepository;
    private final GridCacheService cacheService;

    @Autowired
    public MinusKeywordsPackDataService(MinusKeywordsPacksAddOperationFactory minusKeywordsPacksAddOperationFactory,
                                        MinusKeywordsPacksUpdateOperationFactory minusKeywordsPacksUpdateOperationFactory,
                                        MinusKeywordsPacksDeleteOperationFactory minusKeywordsPacksDeleteOperationFactory,
                                        GridValidationService gridValidationService,
                                        ShardHelper shardHelper,
                                        MinusKeywordsPackRepository packRepository,
                                        GridCacheService cacheService) {
        this.minusKeywordsPacksAddOperationFactory = minusKeywordsPacksAddOperationFactory;
        this.minusKeywordsPacksUpdateOperationFactory = minusKeywordsPacksUpdateOperationFactory;
        this.minusKeywordsPacksDeleteOperationFactory = minusKeywordsPacksDeleteOperationFactory;
        this.gridValidationService = gridValidationService;
        this.shardHelper = shardHelper;
        this.packRepository = packRepository;
        this.cacheService = cacheService;
    }

    public GdAddMinusKeywordsPacksPayload addMinusKeywordsPacks(GdAddMinusKeywordsPacks input, ClientId clientId) {
        int shard = shardHelper.getShardByClientId(clientId);
        List<MinusKeywordsPack> minusKeywordsPacksToAdd = toCoreMinusKeywordsPacks(input);

        MinusKeywordsPacksAddOperation addOperation = minusKeywordsPacksAddOperationFactory
                .newInstance(PARTIAL, minusKeywordsPacksToAdd,
                        MinusPhraseValidator.ValidationMode.ONE_ERROR_PER_TYPE_AND_KEYWORD, clientId, shard, false);
        MassResult<AddedMinusKeywordsPackInfo> result = addOperation.prepareAndApply();

        GdValidationResult validationResult = getGdValidationResult(result, GdAddMinusKeywordsPacks.ADD_ITEMS);
        List<GdAddMinusKeywordsPacksPayloadItem> addedItems =
                getResults(result, MinusKeywordsPackDataConverter::toGdAddMinusKeywordsPacksPayloadItem);
        return new GdAddMinusKeywordsPacksPayload()
                .withAddedItems(addedItems)
                .withValidationResult(validationResult);
    }

    public GdUpdateMinusKeywordsPacksPayload updateMinusKeywordsPacks(GdUpdateMinusKeywordsPacks input,
                                                                      ClientId clientId) {
        int shard = shardHelper.getShardByClientId(clientId);
        List<ModelChanges<MinusKeywordsPack>> minusKeywordsPacksChanges = toCoreMinusKeywordsPackChanges(input);

        MinusKeywordsPacksUpdateOperation updateOperation = minusKeywordsPacksUpdateOperationFactory
                .newInstance(PARTIAL, minusKeywordsPacksChanges,
                        MinusPhraseValidator.ValidationMode.ONE_ERROR_PER_TYPE_AND_KEYWORD, clientId, shard);
        MassResult<UpdatedMinusKeywordsPackInfo> result = updateOperation.prepareAndApply();

        GdValidationResult validationResult = getGdValidationResult(result, GdUpdateMinusKeywordsPacks.UPDATE_ITEMS);
        List<GdUpdateMinusKeywordsPacksPayloadItem> updatedItems =
                getResults(result, MinusKeywordsPackDataConverter::toGdUpdateMinusKeywordsPacksPayloadItem);
        return new GdUpdateMinusKeywordsPacksPayload()
                .withUpdatedItems(updatedItems)
                .withValidationResult(validationResult);
    }

    /**
     * Удалить наборы минус-фраз
     *
     * @param input    запрос со списком идентификаторов библиотечных наборов, которые нужно удалить.
     * @param clientId id клиента, над которым производится действие
     */
    public GdDeleteMinusKeywordsPacksPayload deleteMinusKeywordsPacks(GdDeleteMinusKeywordsPacks input,
                                                                      ClientId clientId) {
        int shard = shardHelper.getShardByClientId(clientId);

        MinusKeywordsPacksDeleteOperation operation = minusKeywordsPacksDeleteOperationFactory
                .newInstance(PARTIAL, input.getMinusKeywordPackIds(), clientId, shard);

        MassResult<Long> result = operation.prepareAndApply();
        GdValidationResult vr = getGdValidationResult(result, GdDeleteMinusKeywordsPacks.MINUS_KEYWORD_PACK_IDS);

        return new GdDeleteMinusKeywordsPacksPayload()
                .withDeletedPackIds(getSuccessfullyUpdatedIds(result, identity()))
                .withValidationResult(vr);
    }

    /**
     * Достать результат валидации из MassResult и сконвертировать в GdValidationResult
     *
     * @param result        - результат операции
     * @param modelProperty - имя modelProperty будет использоваться в качестве префикса для всех путей в валидации
     * @return сконвертированный результат валидации
     */
    private GdValidationResult getGdValidationResult(MassResult result, ModelProperty modelProperty) {
        return gridValidationService.getValidationResult(result, path(field(modelProperty)));
    }

    /**
     * Получить библиотечные наборы клиента вместе с количеством привязанных групп.
     * Данные кешируются.
     */
    public GdGetMinusKeywordsPacksPayload getLibraryMinusKeywordsPacks(GdGetMinusKeywordsPacks input, ClientId clientId) {
        LimitOffset range = normalizeLimitOffset(input.getLimitOffset());
        GetMinusKeywordsPacksCacheRecordInfo cacheInfo = getCacheRecordInfo(clientId, input);
        Optional<GdGetMinusKeywordsPacksPayload> res = cacheService.getFromCache(cacheInfo, range);
        if (res.isPresent()) {
            return res.get();
        }

        // если в кеше не нашли:
        int shard = shardHelper.getShardByClientId(clientId);
        List<Pair<MinusKeywordsPack, Long>> libraryPacksWithLinksCount = packRepository
                .getLibraryPacksWithLinksCount(shard, clientId, input.getPackIdIn(), LimitOffset.maxLimited());

        List<GdGetMinusKeywordsPacksItem> rowSetFull = toGdGetMinusKeywordsPacksItemList(libraryPacksWithLinksCount);

        GdGetMinusKeywordsPacksPayload payload = new GdGetMinusKeywordsPacksPayload();
        payload.setTotalCount(rowSetFull.size());
        return cacheService.getResultAndSaveToCache(cacheInfo, payload, rowSetFull, range);
    }
}
