package ru.yandex.direct.grid.processing.service.offer;

import java.math.BigDecimal;
import java.nio.ByteBuffer;
import java.time.LocalDate;
import java.util.Base64;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.model.CampaignAttributionModel;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithAttributionModel;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.campaign.service.CampMetrikaCountersService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.grid.core.entity.model.GdiOfferStats;
import ru.yandex.direct.grid.core.entity.offer.model.GdiOffer;
import ru.yandex.direct.grid.core.entity.offer.model.GdiOfferFilter;
import ru.yandex.direct.grid.core.entity.offer.model.GdiOfferId;
import ru.yandex.direct.grid.core.entity.offer.model.GdiOfferOrderBy;
import ru.yandex.direct.grid.core.entity.offer.service.GridOfferService;
import ru.yandex.direct.grid.model.GdStatRequirements;
import ru.yandex.direct.grid.model.offer.GdOffer;
import ru.yandex.direct.grid.processing.model.offer.GdOffersContainer;
import ru.yandex.direct.grid.processing.service.offer.converter.OfferDataConverter;
import ru.yandex.direct.metrika.client.model.response.GetExistentCountersResponseItem;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.utils.net.FastUrlBuilder;

import static java.util.Collections.emptySet;
import static ru.yandex.direct.grid.processing.service.offer.converter.OfferDataConverter.toInternalFilter;
import static ru.yandex.direct.grid.processing.service.offer.converter.OfferDataConverter.toOfferOrderByYtFields;
import static ru.yandex.direct.utils.FunctionalUtils.flatMapToSet;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class OfferDataService {
    private static final String LAST_CLICK_FILTER_FORMAT = "(ym:s:%sDirectOfferComplexID=='%s')";
    private static final String FIRST_CLICK_FILTER_FORMAT = "(EXISTS+ym:u:userID+WITH+" +
            "(ym:u:%sDirectOfferComplexID=='%s'))";

    /** Свойства, для сбора данных по которым необходимо подключение Ecommerce к счетчику Метрики. */
    private static final Set<ModelProperty<GdiOfferStats, BigDecimal>> ECOMMERCE_STATS_PROPERTIES = Set.of(
            GdiOfferStats.REVENUE, GdiOfferStats.CRR, GdiOfferStats.CARTS, GdiOfferStats.PURCHASES,
            GdiOfferStats.AVG_PRODUCT_PRICE, GdiOfferStats.AVG_PURCHASE_REVENUE
    );

    private final ShardHelper shardHelper;
    private final CampaignTypedRepository campaignTypedRepository;
    private final CampMetrikaCountersService campMetrikaCountersService;
    private final GridOfferService gridOfferService;

    @Autowired
    public OfferDataService(ShardHelper shardHelper,
                            CampaignTypedRepository campaignTypedRepository,
                            CampMetrikaCountersService campMetrikaCountersService,
                            GridOfferService gridOfferService) {
        this.shardHelper = shardHelper;
        this.campaignTypedRepository = campaignTypedRepository;
        this.campMetrikaCountersService = campMetrikaCountersService;
        this.gridOfferService = gridOfferService;
    }

    public List<GdOffer> getOffers(ClientId clientId, GdOffersContainer input) {
        GdiOfferFilter filter = toInternalFilter(input.getFilter());
        List<GdiOfferOrderBy> offerOrderByList = toOfferOrderByYtFields(input.getOrderBy());
        GdStatRequirements statRequirements = input.getStatRequirements();

        List<GdiOffer> offers = gridOfferService.getOffers(clientId, filter, offerOrderByList,
                statRequirements.getFrom(), statRequirements.getTo());
        enrichOffers(clientId, offers, filter.getCampaignIdIn(), statRequirements.getFrom(), statRequirements.getTo());

        return mapList(offers, OfferDataConverter::toGdOffer);
    }

    private void enrichOffers(ClientId clientId, Collection<GdiOffer> offers, Collection<Long> campaignIds,
                              LocalDate statStartDay, LocalDate statEndDay) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        Map<Long, List<Long>> counterIdsByCampaignId = campMetrikaCountersService.getCounterByCampaignIds(clientId,
                campaignIds);

        Set<Long> allCounterIds = flatMapToSet(counterIdsByCampaignId.values(), Function.identity());
        Map<Long, GetExistentCountersResponseItem> counterById = listToMap(campMetrikaCountersService
                        .getExistentCounters(allCounterIds),
                GetExistentCountersResponseItem::getCounterId);
        Set<Long> availableCounterIds = campMetrikaCountersService.getAvailableCounterIdsByClientId(clientId, allCounterIds);

        Map<Long, CampaignAttributionModel> attributionModelByCampaignId = StreamEx.of(campaignTypedRepository
                        .getSafely(shard, campaignIds, CampaignWithAttributionModel.class))
                .mapToEntry(CampaignWithAttributionModel::getId, CampaignWithAttributionModel::getAttributionModel)
                .nonNullValues()
                .toMap();

        Map<GdiOfferId, Set<Long>> campaignIdSetByOfferId = gridOfferService.getCampaignIdsByOfferId(clientId,
                campaignIds, mapList(offers, GdiOffer::getId), statStartDay, statEndDay);
        Set<Set<Long>> campaignIdSets = listToSet(campaignIdSetByOfferId.values());

        Map<Set<Long>, Set<Long>> counterIdsByCampaignIdSet = StreamEx.of(campaignIdSets)
                .mapToEntry(Function.identity())
                .flatMapValues(Collection::stream)
                .mapValues(counterIdsByCampaignId::get)
                .nonNullValues()
                .flatMapValues(Collection::stream)
                .grouping(Collectors.toSet());

        Map<Set<Long>, CampaignAttributionModel> attributionModelByCampaignIdSet = StreamEx.of(campaignIdSets)
                .mapToEntry(Function.identity())
                .flatMapValues(Collection::stream)
                .mapValues(attributionModelByCampaignId::get)
                .nonNullValues()
                .collapseKeys(Collectors.groupingBy(Function.identity(), Collectors.counting()))
                .mapValues(cardinalityMap -> EntryStream.of(cardinalityMap)
                        .max(Map.Entry.<CampaignAttributionModel, Long>comparingByValue()
                                .thenComparing(Map.Entry.comparingByKey()))
                        .map(Map.Entry::getKey)
                        .orElseThrow())
                .toMap();

        offers.forEach(offer -> {
            Set<Long> campaignIdSet = campaignIdSetByOfferId.get(offer.getId());
            Set<Long> counterIds = counterIdsByCampaignIdSet.getOrDefault(campaignIdSet, emptySet());
            Set<Long> ecommerceCounterIds = StreamEx.of(counterIds)
                    .map(counterById::get)
                    .nonNull()
                    .filterBy(GetExistentCountersResponseItem::getEcommerce, true)
                    .map(GetExistentCountersResponseItem::getCounterId)
                    .toSet();

            CampaignAttributionModel attributionModel = attributionModelByCampaignIdSet.getOrDefault(campaignIdSet,
                    CampaignAttributionModel.LAST_YANDEX_DIRECT_CLICK);
            Map<Long, String> metrikaStatUrlByCounterId = StreamEx.of(ecommerceCounterIds)
                    .filter(availableCounterIds::contains)
                    .mapToEntry(counterId -> getMetrikaStatUrl(counterId, statStartDay, statEndDay,
                            attributionModel, offer.getId()))
                    .toMap();
            offer.getStats().setMetrikaStatUrlByCounterId(metrikaStatUrlByCounterId);

            if (ecommerceCounterIds.isEmpty()) {
                ECOMMERCE_STATS_PROPERTIES.forEach(property -> {
                    @Nullable BigDecimal value = property.get(offer.getStats());
                    if (value != null && value.compareTo(BigDecimal.ZERO) == 0) {
                        property.set(offer.getStats(), null);
                    }
                });
            }

            if (!availableCounterIds.containsAll(counterIds)) { // проверка на доступ
                offer.getStats().setRevenue(null);
                offer.getStats().setCrr(null);
                offer.getStats().setAvgPurchaseRevenue(null);
            }
        });
    }

    static String getMetrikaStatUrl(Long counterId, LocalDate statStartDay, LocalDate statEndDay,
                                    CampaignAttributionModel attributionModel, GdiOfferId offerId) {
        byte[] offerIdBytes = ByteBuffer.allocate(16)
                .putInt(offerId.getBusinessId().intValue())
                .putInt(offerId.getShopId().intValue())
                .putLong(offerId.getOfferYabsId())
                .array();
        String offerComplexId = Base64.getEncoder().encodeToString(offerIdBytes);

        return new FastUrlBuilder("https://metrika.yandex.ru")
                .addPath("stat")
                .addPath("orders_products")
                .addParam("id", counterId)
                .addParam("period", String.format("%s:%s", statStartDay, statEndDay))
                .addParam("filter", getMetrikaStatUrlFilter(attributionModel, offerComplexId))
                .build();
    }

    private static String getMetrikaStatUrlFilter(CampaignAttributionModel attributionModel, String offerComplexId) {
        switch (attributionModel) {
            case LAST_CLICK:
                return String.format(LAST_CLICK_FILTER_FORMAT, "last", offerComplexId);
            case FIRST_CLICK:
                return String.format(FIRST_CLICK_FILTER_FORMAT, "first", offerComplexId);
            case LAST_SIGNIFICANT_CLICK:
                return String.format(LAST_CLICK_FILTER_FORMAT, "lastSign", offerComplexId);
            case LAST_YANDEX_DIRECT_CLICK:
                return String.format(LAST_CLICK_FILTER_FORMAT, "LAST_YANDEX_DIRECT_CLICK", offerComplexId);
            case FIRST_CLICK_CROSS_DEVICE:
                return String.format(FIRST_CLICK_FILTER_FORMAT, "CROSS_DEVICE_FIRST", offerComplexId);
            case LAST_SIGNIFICANT_CLICK_CROSS_DEVICE:
                return String.format(LAST_CLICK_FILTER_FORMAT, "CROSS_DEVICE_LAST_SIGNIFICANT", offerComplexId);
            case LAST_YANDEX_DIRECT_CLICK_CROSS_DEVICE:
                return String.format(LAST_CLICK_FILTER_FORMAT, "CROSS_DEVICE_LAST_YANDEX_DIRECT_CLICK", offerComplexId);
            default:
                throw new IllegalArgumentException("Unexpected attribution model: " + attributionModel);
        }
    }
}
