package ru.yandex.direct.grid.processing.service.offer;

import java.util.List;
import java.util.Objects;
import java.util.Optional;

import javax.annotation.ParametersAreNonnullByDefault;

import io.leangen.graphql.annotations.GraphQLArgument;
import io.leangen.graphql.annotations.GraphQLContext;
import io.leangen.graphql.annotations.GraphQLMutation;
import io.leangen.graphql.annotations.GraphQLNonNull;
import io.leangen.graphql.annotations.GraphQLQuery;
import io.leangen.graphql.annotations.GraphQLRootContext;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.security.authorization.PreAuthorizeWrite;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.model.GdStatRequirements;
import ru.yandex.direct.grid.model.offer.GdOffer;
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.client.GdClient;
import ru.yandex.direct.grid.processing.model.offer.GdOfferFilter;
import ru.yandex.direct.grid.processing.model.offer.GdOffersContainer;
import ru.yandex.direct.grid.processing.model.offer.GdOffersContext;
import ru.yandex.direct.grid.processing.model.offer.mutation.GdFilterOffers;
import ru.yandex.direct.grid.processing.model.offer.mutation.GdFilterOffersPayload;
import ru.yandex.direct.grid.processing.service.cache.GridCacheService;
import ru.yandex.direct.grid.processing.service.offer.container.OffersCacheRecordInfo;
import ru.yandex.direct.grid.processing.service.offer.validation.OfferValidationService;
import ru.yandex.direct.grid.processing.service.shortener.GridShortenerService;
import ru.yandex.direct.multitype.entity.LimitOffset;

import static java.util.Collections.emptySet;
import static ru.yandex.direct.grid.processing.service.cache.util.CacheUtils.normalizeLimitOffset;
import static ru.yandex.direct.grid.processing.service.offer.converter.OfferDataConverter.toGdOfferStatsColumnFlags;
import static ru.yandex.direct.grid.processing.service.offer.converter.OfferDataConverter.toOffersCacheRecordInfo;
import static ru.yandex.direct.grid.processing.util.StatHelper.normalizeStatRequirements;

@GridGraphQLService
@ParametersAreNonnullByDefault
public class OfferGraphQlService {
    public static final String OFFERS_RESOLVER_NAME = "offers";

    private final GridCacheService gridCacheService;
    private final GridShortenerService gridShortenerService;
    private final OfferDataService offerDataService;
    private final OfferMutationService offerMutationService;
    private final OfferValidationService offerValidationService;

    @Autowired
    public OfferGraphQlService(GridCacheService gridCacheService,
                               GridShortenerService gridShortenerService,
                               OfferDataService offerDataService,
                               OfferMutationService offerMutationService,
                               OfferValidationService offerValidationService) {
        this.gridCacheService = gridCacheService;
        this.gridShortenerService = gridShortenerService;
        this.offerDataService = offerDataService;
        this.offerMutationService = offerMutationService;
        this.offerValidationService = offerValidationService;
    }

    @GraphQLNonNull
    @GraphQLQuery(name = OFFERS_RESOLVER_NAME)
    public GdOffersContext getOffers(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdClient client,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdOffersContainer input) {
        offerValidationService.validateGdOffersContainer(input);

        var clientId = ClientId.fromLong(client.getInfo().getId());

        if (input.getFilterKey() != null) {
            GdOfferFilter savedFilter = gridShortenerService.getSavedFilter(input.getFilterKey(),
                    clientId,
                    GdOfferFilter.class,
                    () -> new GdOfferFilter().withCampaignIdIn(emptySet()));
            input.setFilter(savedFilter);
        }

        GdStatRequirements statRequirements = input.getStatRequirements();
        statRequirements = normalizeStatRequirements(statRequirements, context.getInstant());
        input.setStatRequirements(statRequirements);

        LimitOffset range = normalizeLimitOffset(input.getLimitOffset());

        OffersCacheRecordInfo recordInfo = toOffersCacheRecordInfo(clientId, input);
        Optional<GdOffersContext> cached = gridCacheService.getFromCache(recordInfo, range);
        if (cached.isPresent()) {
            return cached.get();
        }

        List<GdOffer> offers = offerDataService.getOffers(clientId, input);
        GdOffersContext offersContext = new GdOffersContext()
                .withTotalCount(offers.size())
                .withStatsColumnFlags(toGdOfferStatsColumnFlags(offers))
                .withFilter(input.getFilter());
        return gridCacheService.getResultAndSaveToCacheIfRequested(recordInfo, offersContext,
                offers, range, context.getFetchedFieldsReslover().getOffer().getCacheKey());
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @GraphQLMutation(name = "filterOffers")
    public GdFilterOffersPayload filterOffers(@GraphQLRootContext GridGraphQLContext context,
                                              @GraphQLNonNull @GraphQLArgument(name = "input") GdFilterOffers input) {
        var clientId = Objects.requireNonNull(context.getSubjectUser()).getClientId();
        return offerMutationService.filterOffers(clientId, context.getOperator().getUid(), input);
    }
}
