package ru.yandex.direct.grid.processing.service.offer;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Suppliers;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.dynamictextadtarget.container.DynamicTextAdTargetSelectionCriteria;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicFeedAdTarget;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicTextAdTarget;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicTextAdTargetState;
import ru.yandex.direct.core.entity.dynamictextadtarget.service.DynamicTextAdTargetService;
import ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.AddDynamicFeedAdTargetValidationService;
import ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.AddDynamicTextAdTargetValidationService;
import ru.yandex.direct.core.entity.performancefilter.container.PerformanceFiltersQueryFilter;
import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilter;
import ru.yandex.direct.core.entity.performancefilter.service.PerformanceFilterService;
import ru.yandex.direct.core.entity.performancefilter.service.PerformanceFilterStorage;
import ru.yandex.direct.core.entity.performancefilter.service.PerformanceFilterValidationService;
import ru.yandex.direct.dbutil.model.BusinessIdAndShopId;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.grid.core.entity.offer.converter.GridOfferConverter;
import ru.yandex.direct.grid.core.entity.offer.model.GdiOffer;
import ru.yandex.direct.grid.core.entity.offer.model.GdiOfferId;
import ru.yandex.direct.grid.core.entity.offer.service.GridOfferService;
import ru.yandex.direct.grid.processing.model.offer.mutation.GdFilterOffers;
import ru.yandex.direct.grid.processing.model.offer.mutation.GdFilterOffersPayload;
import ru.yandex.direct.grid.processing.service.offer.converter.OfferDataConverter;
import ru.yandex.direct.grid.processing.service.offer.util.OfferFilterDynamicFeedAdTargetBuilder;
import ru.yandex.direct.grid.processing.service.offer.util.OfferFilterDynamicTextAdTargetBuilder;
import ru.yandex.direct.grid.processing.service.offer.util.OfferFilterPerformanceFilterBuilder;
import ru.yandex.direct.grid.processing.service.offer.validation.OfferValidationService;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.operation.Applicability;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.utils.CollectionUtils.isEmpty;
import static ru.yandex.direct.utils.FunctionalUtils.flatten;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
class OfferMutationService {
    private static final Logger logger = LoggerFactory.getLogger(OfferMutationService.class);

    private final ShardHelper shardHelper;
    private final PerformanceFilterStorage performanceFilterStorage;
    private final DynamicTextAdTargetService dynamicTextAdTargetService;
    private final PerformanceFilterService performanceFilterService;
    private final GridOfferService gridOfferService;
    private final AddDynamicTextAdTargetValidationService dynamicTextAdTargetValidationService;
    private final AddDynamicFeedAdTargetValidationService dynamicFeedAdTargetValidationService;
    private final PerformanceFilterValidationService performanceFilterValidationService;
    private final OfferValidationService offerValidationService;

    @Autowired
    public OfferMutationService(ShardHelper shardHelper,
                                PerformanceFilterStorage performanceFilterStorage,
                                DynamicTextAdTargetService dynamicTextAdTargetService,
                                PerformanceFilterService performanceFilterService,
                                GridOfferService gridOfferService,
                                AddDynamicTextAdTargetValidationService dynamicTextAdTargetValidationService,
                                AddDynamicFeedAdTargetValidationService dynamicFeedAdTargetValidationService,
                                PerformanceFilterValidationService performanceFilterValidationService,
                                OfferValidationService offerValidationService) {
        this.shardHelper = shardHelper;
        this.performanceFilterStorage = performanceFilterStorage;
        this.dynamicTextAdTargetService = dynamicTextAdTargetService;
        this.performanceFilterService = performanceFilterService;
        this.gridOfferService = gridOfferService;
        this.dynamicTextAdTargetValidationService = dynamicTextAdTargetValidationService;
        this.dynamicFeedAdTargetValidationService = dynamicFeedAdTargetValidationService;
        this.performanceFilterValidationService = performanceFilterValidationService;
        this.offerValidationService = offerValidationService;
    }

    public GdFilterOffersPayload filterOffers(ClientId clientId, long operatorUid, GdFilterOffers input) {
        offerValidationService.validateFilterOffersRequest(input);

        var offerIds = listToSet(input.getOfferIds(), OfferDataConverter::toInternalOfferId);
        var offerById = gridOfferService.getOfferById(clientId, offerIds);
        var offersByBusinessIdAndShopId = EntryStream.of(offerById)
                .mapKeys(GridOfferConverter::extractBusinessIdAndShopId)
                .grouping();

        if (offerById.isEmpty() || input.getCampaignIdIn().isEmpty()
                || (input.getAdGroupIdIn() != null && input.getAdGroupIdIn().isEmpty())) {
            return getGdFilterOffersPayload(input, offerById.keySet(), true);
        }

        var dynamicTextAdTargetModelChanges = getOfferFilterDynamicTextAdTargetModelChanges(clientId, operatorUid,
                input.getCampaignIdIn(), input.getAdGroupIdIn(), offersByBusinessIdAndShopId);
        var dynamicFeedAdTargetModelChanges = getOfferFilterDynamicFeedAdTargetModelChanges(clientId, operatorUid,
                input.getCampaignIdIn(), input.getAdGroupIdIn(), offersByBusinessIdAndShopId);
        var performanceFilterModelChanges = getOfferFilterPerformanceFilterModelChanges(clientId,
                input.getCampaignIdIn(), input.getAdGroupIdIn(), offersByBusinessIdAndShopId);

        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        var dynTextVr = dynamicTextAdTargetValidationService
                .validateModelChanges(shard, clientId, dynamicTextAdTargetModelChanges);
        if (dynTextVr.hasAnyErrors()) {
            logger.warn("Dynamic text ad targets validation failed: {}", dynTextVr.flattenErrors());
        }
        var dynFeedVr = dynamicFeedAdTargetValidationService
                .validateModelChanges(shard, clientId, dynamicFeedAdTargetModelChanges);
        if (dynFeedVr.hasAnyErrors()) {
            logger.warn("Dynamic feed ad targets validation failed: {}", dynFeedVr.flattenErrors());
        }
        var perfVr = performanceFilterValidationService
                .validateModelChanges(clientId, performanceFilterModelChanges);
        if (perfVr.hasAnyErrors()) {
            logger.warn("Performance filters validation failed: {}", perfVr.flattenErrors());
        }

        if (dynTextVr.hasAnyErrors() || dynFeedVr.hasAnyErrors() || perfVr.hasAnyErrors()) {
            return getGdFilterOffersPayload(input, offerById.keySet(), false);
        }

        var dynTextResult = dynamicTextAdTargetService
                .updateDynamicTextAdTargets(clientId, operatorUid, dynamicTextAdTargetModelChanges);
        if (!isEmpty(dynTextResult.getErrors())) {
            logger.error("Failed to update dynamic text ad targets: {}", dynTextResult.getErrors());
        }
        var dynFeedResult = dynamicTextAdTargetService
                .updateDynamicFeedAdTargets(clientId, operatorUid, dynamicFeedAdTargetModelChanges);
        if (!isEmpty(dynFeedResult.getErrors())) {
            logger.error("Failed to update dynamic feed ad targets: {}", dynFeedResult.getErrors());
        }
        var perfResult = performanceFilterService
                .updatePerformanceFilters(clientId, operatorUid, performanceFilterModelChanges, Applicability.PARTIAL);
        if (!isEmpty(perfResult.getErrors())) {
            logger.error("Failed to update performance filters: {}", perfResult.getErrors());
        }

        if (!dynTextResult.isSuccessful() || !dynFeedResult.isSuccessful() || !perfResult.isSuccessful()) {
            throw new RuntimeException("Failed to filter offers");
        }

        return getGdFilterOffersPayload(input, offerById.keySet(), true);
    }

    private GdFilterOffersPayload getGdFilterOffersPayload(GdFilterOffers input, Set<GdiOfferId> existingOfferIds,
                                                           boolean successful) {
        return new GdFilterOffersPayload()
                .withFilteredOfferIds(mapList(successful ? existingOfferIds : emptyList(),
                        OfferDataConverter::toGdOfferId))
                .withValidationResult(offerValidationService
                        .getFilterOffersValidationResult(input, existingOfferIds, successful));
    }

    private List<ModelChanges<DynamicTextAdTarget>> getOfferFilterDynamicTextAdTargetModelChanges(
            ClientId clientId, long operatorUid, Set<Long> campaignIds, @Nullable Set<Long> adGroupIds,
            Map<BusinessIdAndShopId, List<GdiOffer>> offersByBusinessIdAndShopId) {
        var dynamicTextAdTargets = dynamicTextAdTargetService.getDynamicTextAdTargets(clientId, operatorUid,
                new DynamicTextAdTargetSelectionCriteria()
                        .withCampaignIds(campaignIds)
                        .withAdGroupIds(adGroupIds)
                        .withStates(DynamicTextAdTargetState.ON, DynamicTextAdTargetState.OFF,
                                DynamicTextAdTargetState.SUSPENDED), LimitOffset.maxLimited());

        // для всех ДО по сайту добавляем все товары в фильтры,
        // т.к. в случае ДО по сайту у нас нет фида и мы не можем понять, к какой именно группе относится оффер
        return StreamEx.of(dynamicTextAdTargets)
                .zipWith(StreamEx.generate(Suppliers.ofInstance(flatten(offersByBusinessIdAndShopId.values()))))
                .mapKeyValue(this::getOfferFilterModelChanges)
                .toList();
    }

    private List<ModelChanges<DynamicFeedAdTarget>> getOfferFilterDynamicFeedAdTargetModelChanges(
            ClientId clientId, long operatorUid, Set<Long> campaignIds, @Nullable Set<Long> adGroupIds,
            Map<BusinessIdAndShopId, List<GdiOffer>> offersByBusinessIdAndShopId) {
        var dynamicFeedAdTargetsByBusinessIdAndShopId = dynamicTextAdTargetService
                .getDynamicFeedAdTargetsByBusinessIdAndShopId(clientId, operatorUid,
                        new DynamicTextAdTargetSelectionCriteria()
                                .withCampaignIds(campaignIds)
                                .withAdGroupIds(adGroupIds)
                                .withBusinessIdsAndShopIds(offersByBusinessIdAndShopId.keySet())
                                .withStates(DynamicTextAdTargetState.ON, DynamicTextAdTargetState.OFF,
                                        DynamicTextAdTargetState.SUSPENDED));

        return EntryStream.of(dynamicFeedAdTargetsByBusinessIdAndShopId)
                .flatMapValues(Collection::stream)
                .invert()
                .mapValues(offersByBusinessIdAndShopId::get)
                .mapKeyValue(this::getOfferFilterModelChanges)
                .toList();
    }

    private List<ModelChanges<PerformanceFilter>> getOfferFilterPerformanceFilterModelChanges(
            ClientId clientId, Set<Long> campaignIds, @Nullable Set<Long> adGroupIds,
            Map<BusinessIdAndShopId, List<GdiOffer>> offersByBusinessIdAndShopId) {
        var performanceFiltersByBusinessIdAndShopId = performanceFilterService
                .getPerformanceFiltersByBusinessIdAndShopId(clientId, PerformanceFiltersQueryFilter.newBuilder()
                        .withClientId(clientId)
                        .withCampaignIds(campaignIds)
                        .withAdGroupIds(adGroupIds)
                        .withBusinessIdsAndShopIds(offersByBusinessIdAndShopId.keySet())
                        .build());

        return EntryStream.of(performanceFiltersByBusinessIdAndShopId)
                .flatMapValues(Collection::stream)
                .invert()
                .mapValues(offersByBusinessIdAndShopId::get)
                .mapKeyValue(this::getOfferFilterModelChanges)
                .toList();
    }

    ModelChanges<DynamicTextAdTarget> getOfferFilterModelChanges(DynamicTextAdTarget dynamicTextAdTarget,
                                                                 List<GdiOffer> offers) {
        return new OfferFilterDynamicTextAdTargetBuilder(dynamicTextAdTarget)
                .filterOffers(offers)
                .toModelChanges();
    }

    ModelChanges<DynamicFeedAdTarget> getOfferFilterModelChanges(DynamicFeedAdTarget dynamicFeedAdTarget,
                                                                 List<GdiOffer> offers) {
        var filterSchema = performanceFilterStorage.getFilterSchema(dynamicFeedAdTarget);

        return new OfferFilterDynamicFeedAdTargetBuilder(filterSchema, dynamicFeedAdTarget)
                .filterOffers(offers)
                .toModelChanges();
    }

    ModelChanges<PerformanceFilter> getOfferFilterModelChanges(PerformanceFilter performanceFilter,
                                                               List<GdiOffer> offers) {
        var filterSchema = performanceFilterStorage.getFilterSchema(performanceFilter);

        return new OfferFilterPerformanceFilterBuilder(filterSchema, performanceFilter)
                .filterOffers(offers)
                .toModelChanges();
    }
}
