package ru.yandex.direct.grid.processing.service.offer.converter;

import java.util.List;
import java.util.Map;
import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;

import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.core.entity.offer.model.GdiOffer;
import ru.yandex.direct.grid.core.entity.offer.model.GdiOfferFilter;
import ru.yandex.direct.grid.core.entity.offer.model.GdiOfferId;
import ru.yandex.direct.grid.core.entity.offer.model.GdiOfferOrderBy;
import ru.yandex.direct.grid.model.GdOfferStats;
import ru.yandex.direct.grid.model.offer.GdOffer;
import ru.yandex.direct.grid.model.offer.GdOfferId;
import ru.yandex.direct.grid.processing.model.offer.GdOfferFilter;
import ru.yandex.direct.grid.processing.model.offer.GdOfferOrderBy;
import ru.yandex.direct.grid.processing.model.offer.GdOfferOrderByField;
import ru.yandex.direct.grid.processing.model.offer.GdOfferStatsColumnFlags;
import ru.yandex.direct.grid.processing.model.offer.GdOffersContainer;
import ru.yandex.direct.grid.processing.service.offer.container.OffersCacheFilterData;
import ru.yandex.direct.grid.processing.service.offer.container.OffersCacheRecordInfo;
import ru.yandex.direct.grid.processing.util.StatHelper;
import ru.yandex.direct.model.ModelProperty;

import static ru.yandex.direct.utils.FunctionalUtils.mapAndFilterList;

@ParametersAreNonnullByDefault
public class OfferDataConverter {
    private static final Map<ModelProperty<GdOfferStatsColumnFlags, Boolean>, ModelProperty<GdOfferStats, ?>> STATS_COLUMN_FLAGS_PROPERTIES = Map.of(
            GdOfferStatsColumnFlags.HAS_REVENUE, GdOfferStats.REVENUE,
            GdOfferStatsColumnFlags.HAS_CRR, GdOfferStats.CRR,
            GdOfferStatsColumnFlags.HAS_CARTS, GdOfferStats.CARTS,
            GdOfferStatsColumnFlags.HAS_PURCHASES, GdOfferStats.PURCHASES,
            GdOfferStatsColumnFlags.HAS_AVG_PRODUCT_PRICE, GdOfferStats.AVG_PRODUCT_PRICE,
            GdOfferStatsColumnFlags.HAS_AVG_PURCHASE_REVENUE, GdOfferStats.AVG_PURCHASE_REVENUE
    );

    public static GdOfferStatsColumnFlags toGdOfferStatsColumnFlags(List<GdOffer> rowset) {
        List<GdOfferStats> stats = mapAndFilterList(rowset, GdOffer::getStats, Objects::nonNull);

        GdOfferStatsColumnFlags flags = new GdOfferStatsColumnFlags();
        STATS_COLUMN_FLAGS_PROPERTIES.forEach((flagProperty, columnValueProperty) -> flagProperty.set(flags,
                stats.stream().map(columnValueProperty::get).anyMatch(Objects::nonNull)));
        return flags;
    }

    public static OffersCacheRecordInfo toOffersCacheRecordInfo(ClientId clientId, GdOffersContainer input) {
        return new OffersCacheRecordInfo(clientId.asLong(), input.getCacheKey(),
                new OffersCacheFilterData()
                        .withFilter(input.getFilter())
                        .withOrderBy(input.getOrderBy()));
    }

    public static GdiOfferFilter toInternalFilter(GdOfferFilter filter) {
        return new GdiOfferFilter()
                .withCampaignIdIn(filter.getCampaignIdIn())
                .withAdGroupIdIn(filter.getAdGroupIdIn())
                .withNameContains(filter.getNameContains())
                .withNameNotContains(filter.getNameNotContains())
                .withUrlContains(filter.getUrlContains())
                .withUrlNotContains(filter.getUrlNotContains())
                .withStats(StatHelper.toInternalOfferStatsFilter(filter.getStats()));
    }

    public static List<GdiOfferOrderBy> toOfferOrderByYtFields(List<GdOfferOrderBy> orderBy) {
        return StreamEx.of(orderBy)
                .map(ob -> new GdiOfferOrderBy()
                        .withField(GdOfferOrderByField.toSource(ob.getField()))
                        .withOrder(ob.getOrder()))
                .toList();
    }

    public static GdOffer toGdOffer(GdiOffer offer) {
        return new GdOffer()
                .withId(toGdOfferId(offer.getId()))
                .withUrl(offer.getUrl())
                .withName(offer.getName())
                .withImageUrl(offer.getImageUrl())
                .withPrice(offer.getPrice())
                .withCurrencyIsoCode(offer.getCurrencyIsoCode())
                .withStats(StatHelper.internalOfferStatsToOuter(offer.getStats()));
    }

    public static GdOfferId toGdOfferId(GdiOfferId offerId) {
        return new GdOfferId()
                .withBusinessId(offerId.getBusinessId())
                .withShopId(offerId.getShopId())
                .withOfferYabsId(offerId.getOfferYabsId());
    }

    public static GdiOfferId toInternalOfferId(GdOfferId offerId) {
        return new GdiOfferId()
                .withBusinessId(offerId.getBusinessId())
                .withShopId(offerId.getShopId())
                .withOfferYabsId(offerId.getOfferYabsId());
    }
}
