package ru.yandex.direct.grid.processing.service.offer.util;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import org.jetbrains.annotations.Nullable;

import ru.yandex.direct.core.entity.performancefilter.model.Operator;
import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilterCondition;
import ru.yandex.direct.core.entity.performancefilter.schema.FilterSchema;
import ru.yandex.direct.core.entity.performancefilter.schema.parser.ObjectListParser;
import ru.yandex.direct.core.entity.performancefilter.service.PerformanceFilterConditionDBFormatParser;
import ru.yandex.direct.model.ModelWithId;
import ru.yandex.direct.utils.JsonUtils;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.utils.CommonUtils.safeCast;
import static ru.yandex.direct.utils.FunctionalUtils.selectList;

@SuppressWarnings("rawtypes")
@ParametersAreNonnullByDefault
public abstract class OfferFilterFeedConditionsBuilder<M extends ModelWithId, T extends PerformanceFilterCondition> extends OfferFilterBuilder<M, Integer> {
    protected final FilterSchema filterSchema;
    protected final List<T> conditions;

    private final Collection<Integer> urlNotContainsConditionIndexes;

    protected OfferFilterFeedConditionsBuilder(FilterSchema filterSchema, Collection<T> conditions) {
        this.filterSchema = filterSchema;
        this.conditions = new ArrayList<>(conditions);

        this.urlNotContainsConditionIndexes = EntryStream.of(this.conditions)
                .filterValues(condition -> condition.getFieldName().equals(FilterSchema.URL_FIELD_NAME))
                .filterValues(condition -> condition.getOperator().equals(Operator.NOT_CONTAINS))
                .keys()
                .toList();
    }

    @Override
    protected int maxItemCount() {
        return ObjectListParser.MAX_ITEM_COUNT;
    }

    @Override
    protected Collection<Integer> getConditions() {
        return urlNotContainsConditionIndexes;
    }

    @Override
    protected List<String> getItems(Integer index) {
        Collection<?> value = safeCast(conditions.get(index).getParsedValue(), Collection.class);
        checkState(value != null,
                "URL_NOT_CONTAINS condition's parsed value should be a collection");
        List<String> items = selectList(value, String.class);
        checkState(items.size() == value.size(),
                "URL_NOT_CONTAINS condition's parsed value should contain only strings");
        return items;
    }

    @Override
    public void setItems(@Nullable Integer index, List<String> items) {
        var condition = createCondition(FilterSchema.URL_FIELD_NAME, Operator.NOT_CONTAINS, JsonUtils.toJson(items));
        PerformanceFilterConditionDBFormatParser.setParsedValue(filterSchema, List.of(condition));

        if (index != null) {
            conditions.set(index, condition);
        } else {
            urlNotContainsConditionIndexes.add(conditions.size());
            conditions.add(condition);
        }
    }

    protected abstract T createCondition(String fieldName, Operator operator, String stringValue);
}
