package ru.yandex.direct.grid.processing.service.offer.validation;

import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Predicates;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.grid.core.entity.offer.model.GdiOfferId;
import ru.yandex.direct.grid.model.offer.GdOfferId;
import ru.yandex.direct.grid.processing.model.api.GdValidationResult;
import ru.yandex.direct.grid.processing.model.offer.GdOfferFilter;
import ru.yandex.direct.grid.processing.model.offer.GdOffersContainer;
import ru.yandex.direct.grid.processing.model.offer.mutation.GdFilterOffers;
import ru.yandex.direct.grid.processing.service.offer.converter.OfferDataConverter;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.grid.processing.service.validation.GridValidationService.IDS_COLLECTION_VALIDATOR;
import static ru.yandex.direct.grid.processing.service.validation.GridValidationService.LIMIT_OFFSET_VALIDATOR;
import static ru.yandex.direct.grid.processing.service.validation.GridValidationService.OFFER_STATS_FILTER_VALIDATOR;
import static ru.yandex.direct.grid.processing.service.validation.GridValidationService.STAT_REQUIREMENTS_VALIDATOR;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.maxListSize;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.unique;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notEqual;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.unconditional;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;
import static ru.yandex.direct.validation.result.PathHelper.path;

@Service
@ParametersAreNonnullByDefault
@SuppressWarnings("rawtypes")
public class OfferValidationService {
    public static final int MAX_OFFERS_TO_FILTER = 50;

    private static final Validator<GdOfferFilter, Defect> OFFER_FILTER_VALIDATOR = filter -> {
        ModelItemValidationBuilder<GdOfferFilter> vb = ModelItemValidationBuilder.of(filter);
        vb.item(GdOfferFilter.CAMPAIGN_ID_IN)
                .checkBy(IDS_COLLECTION_VALIDATOR);
        vb.item(GdOfferFilter.AD_GROUP_ID_IN)
                .checkBy(IDS_COLLECTION_VALIDATOR, When.notNull());
        vb.item(GdOfferFilter.STATS)
                .checkBy(OFFER_STATS_FILTER_VALIDATOR, When.notNull());
        return vb.getResult();
    };

    private static final Validator<GdOffersContainer, Defect> OFFERS_CONTAINER_VALIDATOR = req -> {
        ModelItemValidationBuilder<GdOffersContainer> vb = ModelItemValidationBuilder.of(req);

        vb.item(GdOffersContainer.FILTER)
                .check(notNull(), When.isTrue(req.getFilterKey() == null));
        vb.item(GdOffersContainer.FILTER_KEY)
                .check(notNull(), When.isTrue(req.getFilter() == null));

        vb.item(GdOffersContainer.FILTER)
                .checkBy(OFFER_FILTER_VALIDATOR, When.notNull());
        vb.item(GdOffersContainer.FILTER_KEY)
                .check(notBlank(), When.notNull());

        vb.item(GdOffersContainer.STAT_REQUIREMENTS)
                .checkBy(STAT_REQUIREMENTS_VALIDATOR);
        vb.item(GdOffersContainer.LIMIT_OFFSET)
                .checkBy(LIMIT_OFFSET_VALIDATOR);
        vb.item(GdOffersContainer.CACHE_KEY)
                .check(notBlank(), When.notNull());

        return vb.getResult();
    };

    private static final Validator<GdOfferId, Defect> OFFER_ID_VALIDATOR = id -> {
        ModelItemValidationBuilder<GdOfferId> vb = ModelItemValidationBuilder.of(id);
        vb.item(GdOfferId.BUSINESS_ID)
                .check(validId());
        vb.item(GdOfferId.SHOP_ID)
                .check(validId());
        vb.item(GdOfferId.OFFER_YABS_ID)
                .check(notEqual(0L, CommonDefects.validId()));
        return vb.getResult();
    };

    private static final Validator<GdFilterOffers, Defect> FILTER_OFFERS_VALIDATOR = req -> {
        ModelItemValidationBuilder<GdFilterOffers> vb = ModelItemValidationBuilder.of(req);
        vb.item(GdFilterOffers.CAMPAIGN_ID_IN)
                .checkBy(IDS_COLLECTION_VALIDATOR);
        vb.item(GdFilterOffers.AD_GROUP_ID_IN)
                .checkBy(IDS_COLLECTION_VALIDATOR, When.notNull());
        vb.list(GdFilterOffers.OFFER_IDS)
                .checkEachBy(OFFER_ID_VALIDATOR)
                .check(maxListSize(MAX_OFFERS_TO_FILTER))
                .checkEach(unique());
        return vb.getResult();
    };

    private final GridValidationService gridValidationService;

    @Autowired
    public OfferValidationService(GridValidationService gridValidationService) {
        this.gridValidationService = gridValidationService;
    }

    public void validateGdOffersContainer(GdOffersContainer input) {
        gridValidationService.applyValidator(OFFERS_CONTAINER_VALIDATOR, input, false);
    }

    public void validateFilterOffersRequest(GdFilterOffers input) {
        gridValidationService.applyValidator(FILTER_OFFERS_VALIDATOR, input, false);
    }

    public GdValidationResult getFilterOffersValidationResult(GdFilterOffers input, Set<GdiOfferId> existingOfferIds,
                                                              boolean successful) {
        ModelItemValidationBuilder<GdFilterOffers> vb = ModelItemValidationBuilder.of(input);
        vb.list(GdFilterOffers.OFFER_IDS)
                .checkEach(Constraint.fromPredicate(Predicates.compose(existingOfferIds::contains,
                        OfferDataConverter::toInternalOfferId), CommonDefects.objectNotFound()))
                .checkEach(unconditional(OfferDefects.unableToFilter()), When.isValidAnd(When.isFalse(successful)));
        return gridValidationService.toGdValidationResult(vb.getResult(), path());
    }
}
