package ru.yandex.direct.grid.processing.service.offlinereport;

import javax.annotation.ParametersAreNonnullByDefault;

import io.leangen.graphql.annotations.GraphQLArgument;
import io.leangen.graphql.annotations.GraphQLMutation;
import io.leangen.graphql.annotations.GraphQLNonNull;
import io.leangen.graphql.annotations.GraphQLQuery;
import io.leangen.graphql.annotations.GraphQLRootContext;

import ru.yandex.direct.core.entity.offlinereport.model.OfflineReportType;
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.offlinereport.GdOfflineReport;
import ru.yandex.direct.grid.processing.model.offlinereport.GdOfflineReportResult;
import ru.yandex.direct.grid.processing.model.offlinereport.GdOfflineReportResultRequest;
import ru.yandex.direct.grid.processing.model.offlinereport.GdOfflineReportWithDailyIntervalRequest;

/**
 * Сервис GraphQL для работы с оффлайн отчётами
 **/
@GridGraphQLService
@ParametersAreNonnullByDefault
public class OfflineReportGraphQlService {
    private final OfflineReportGridService service;

    public OfflineReportGraphQlService(OfflineReportGridService service) {
        this.service = service;
    }

    // Исторически метод использовался для получения списка отчетов по домену, поскольку других типов отчетов не было
    @GraphQLNonNull
    @GraphQLQuery(name = "offlineReportList", description = "список оффлайн отчётов по доменам")
    public GdOfflineReportResult getOfflineReportList(@GraphQLRootContext GridGraphQLContext context) {
        return service.getOfflineReportList(context.getOperator(), OfflineReportType.DOMAINS);
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "offlineReportListByType", description = "список оффлайн отчётов определенного типа")
    public GdOfflineReportResult getOfflineReportListByType(
            @GraphQLNonNull @GraphQLArgument(name = "reportType") OfflineReportType offlineReportType,
            @GraphQLRootContext GridGraphQLContext context) {
        return service.getOfflineReportList(context.getOperator(), offlineReportType);
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "offlineReportCreate",
            description = "запросить создание нового оффлайн отчёта с периодом с точностью до месяца")
    public GdOfflineReport offlineReportCreate(
            @GraphQLNonNull @GraphQLArgument(name = "input") GdOfflineReportResultRequest input,
            @GraphQLRootContext GridGraphQLContext context) {
        return service.createOfflineReportWithMonthlyInterval(input, context.getOperator(), context.getSubjectUser());
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "offlineReportWithDailyIntervalCreate",
            description = "запросить создание нового оффлайн отчёта с периодом с точностью до дня")
    public GdOfflineReport offlineReportWithDailyIntervalCreate(
            @GraphQLNonNull @GraphQLArgument(name = "input") GdOfflineReportWithDailyIntervalRequest input,
            @GraphQLRootContext GridGraphQLContext context) {
        return service.createOfflineReportWithDailyInterval(input, context.getOperator(), context.getSubjectUser());
    }

    /**
     * Запросить создание нового оффлайн отчёта.
     * PreAuthorizeWrite аннотация здесь не нужна, так как отчёт принадлежит запрашивающему. Данные клиента не меняются.
     * То есть суперридер может себе сформировать отчёт по какому-либо агенству.
     * И доступ к готовому отчёту будет только у этого суперридера.
     *
     * @param input   параметры фильтра
     * @param context контекст
     * @return новый заказ на отчёт с ID.
     */
    @GraphQLNonNull
    @GraphQLMutation(name = "createOfflineReport", description = "запросить создание нового оффлайн отчёта")
    public GdOfflineReport createOfflineReport(
            @GraphQLNonNull @GraphQLArgument(name = "input") GdOfflineReportResultRequest input,
            @GraphQLRootContext GridGraphQLContext context) {
        return service.createOfflineReportWithMonthlyInterval(input, context.getOperator(), context.getSubjectUser());
    }
}
