package ru.yandex.direct.grid.processing.service.offlinereport;

import java.time.LocalDate;
import java.time.Period;
import java.time.YearMonth;
import java.time.format.DateTimeFormatter;
import java.util.Set;
import java.util.function.Predicate;

import org.springframework.stereotype.Service;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.core.entity.offlinereport.model.OfflineReportType;
import ru.yandex.direct.grid.processing.model.offlinereport.GdOfflineReportResultRequest;
import ru.yandex.direct.grid.processing.model.offlinereport.GdOfflineReportWithDailyIntervalRequest;
import ru.yandex.direct.grid.processing.service.validation.GridDefectIds;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.common.db.PpcPropertyNames.AGENCY_KPI_OFFLINE_REPORT_MAX_DATE;
import static ru.yandex.direct.common.db.PpcPropertyNames.AGENCY_KPI_OFFLINE_REPORT_MIN_DATE;
import static ru.yandex.direct.common.db.PpcPropertyNames.OFFLINE_REPORT_MAX_REPORT_PERIOD_MONTH;
import static ru.yandex.direct.common.db.PpcPropertyNames.OFFLINE_REPORT_MIN_MONTH_FROM;
import static ru.yandex.direct.grid.processing.model.offlinereport.GdOfflineReportResultRequest.MONTH_FROM;
import static ru.yandex.direct.grid.processing.model.offlinereport.GdOfflineReportResultRequest.MONTH_TO;
import static ru.yandex.direct.grid.processing.model.offlinereport.GdOfflineReportWithDailyIntervalRequest.DATE_FROM;
import static ru.yandex.direct.grid.processing.model.offlinereport.GdOfflineReportWithDailyIntervalRequest.DATE_TO;
import static ru.yandex.direct.grid.processing.model.offlinereport.GdOfflineReportWithDailyIntervalRequest.REPORT_TYPE;
import static ru.yandex.direct.grid.processing.service.validation.GridDefectDefinitions.datesFromNotAfterTo;
import static ru.yandex.direct.grid.processing.service.validation.GridDefectDefinitions.invalidMonthFormat;
import static ru.yandex.direct.validation.Predicates.validYearMonth;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.DateConstraints.isNotAfterThan;
import static ru.yandex.direct.validation.constraint.DateConstraints.isNotBeforeThan;

/**
 * Сервис для валидации оффлайн отчёта.
 **/
@Service
public class OfflineReportValidationService {
    private static final DateTimeFormatter monthFormatter = DateTimeFormatter.ofPattern("yyyyMM");
    private static final String DEFAULT_DOMAIN_OFFLINE_REPORT_MIN_MONTH_FROM = "201903";
    private static final Integer DEFAULT_DOMAIN_OFFLINE_REPORT_MAX_REPORT_PERIOD_MONTH = 36;
    // Типы отчетов, которые формируются c точностью до месяца
    private static final Set<OfflineReportType> OFFLINE_REPORT_TYPES_WITH_MONTHLY_INTERVAL =
            Set.of(OfflineReportType.DOMAINS);

    // Типы отчетов, которые формируются c точностью до дня
    private static final Set<OfflineReportType> OFFLINE_REPORT_TYPES_WITH_DAILY_INTERVAL =
            Set.of(OfflineReportType.AGENCY_KPI);
    // Ограничения зафиксированы в DIRECT-141213
    // Дефолтная дата, начиная с которой есть исходные данные для рассчетов отчета по квартальной премии
    private static final LocalDate DEFAULT_AGENCY_KPI_OFFLINE_REPORT_MIN_DATE = LocalDate.of(2020, 12, 1);
    // Отступ в прошлое от текущей даты. Используется, как fallback для вычисления максимально-допустимой даты
    private static final int AGENCY_KPI_OFFLINE_REPORT_MAX_DATE_GAP = 3;
    // Максимальная длина периода, за который может быть заказан отчет
    private static final Period AGENCY_KPI_OFFLINE_REPORT_MAX_PERIOD_MONTHS = Period.ofMonths(3);

    private final GridValidationService gridValidationService;
    private final PpcPropertiesSupport ppcPropertiesSupport;

    public OfflineReportValidationService(GridValidationService gridValidationService,
                                          PpcPropertiesSupport ppcPropertiesSupport) {
        this.gridValidationService = gridValidationService;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
    }

    public void validateOfflineReportWithMonthlyIntervalRequest(GdOfflineReportResultRequest input) {
        gridValidationService.applyValidator(offlineReportWithMonthlyIntervalRequestValidator, input, false);
    }

    private final Validator<GdOfflineReportResultRequest, Defect> offlineReportWithMonthlyIntervalRequestValidator =
            req -> {
                ModelItemValidationBuilder<GdOfflineReportResultRequest> vb = ModelItemValidationBuilder.of(req);
                vb.item(GdOfflineReportResultRequest.REPORT_TYPE)
                        .check(notNull())
                        .check(inSet(OFFLINE_REPORT_TYPES_WITH_MONTHLY_INTERVAL), When.isValid());
                vb.item(MONTH_FROM)
                        .check(Constraint.fromPredicate(validYearMonth(monthFormatter), invalidMonthFormat()));
                vb.item(MONTH_TO)
                        .check(Constraint.fromPredicate(validYearMonth(monthFormatter), invalidMonthFormat()));
                vb
                        .check(monthFromIsNotAfterMonthTo(), When.isValid())
                        .check(monthFromMinValue(), When.isValid())
                        .check(maxReportPeriod(), When.isValid());
                return vb.getResult();
            };

    private Constraint<GdOfflineReportResultRequest, Defect> maxReportPeriod() {
        Period maxReportPeriodMonth = Period.ofMonths(ppcPropertiesSupport.get(OFFLINE_REPORT_MAX_REPORT_PERIOD_MONTH)
                .getOrDefault(DEFAULT_DOMAIN_OFFLINE_REPORT_MAX_REPORT_PERIOD_MONTH));
        Predicate<GdOfflineReportResultRequest> predicate =//диапазон не больше 3 лет
                input -> (parseMonth(input.getMonthFrom()).plus(maxReportPeriodMonth)).isAfter(parseMonth(input.getMonthTo()));
        return Constraint.fromPredicate(predicate, new Defect<>(GridDefectIds.OfflineReport.MAX_REPORT_PERIOD));
    }

    private Constraint<GdOfflineReportResultRequest, Defect> monthFromMinValue() {
        String minMonthFrom = ppcPropertiesSupport.get(OFFLINE_REPORT_MIN_MONTH_FROM)
                .getOrDefault(DEFAULT_DOMAIN_OFFLINE_REPORT_MIN_MONTH_FROM);
        Predicate<GdOfflineReportResultRequest> predicate =//начинаем с 1 марта 2019
                input -> !parseMonth(input.getMonthFrom()).isBefore(parseMonth(minMonthFrom));
        return Constraint.fromPredicate(predicate, new Defect<>(GridDefectIds.OfflineReport.MIN_MONTH_FROM));
    }

    private static Constraint<GdOfflineReportResultRequest, Defect> monthFromIsNotAfterMonthTo() {
        Predicate<GdOfflineReportResultRequest> predicate =//месяц по не раньше месяц с
                input -> !parseMonth(input.getMonthFrom()).isAfter(parseMonth(input.getMonthTo()));
        return Constraint.fromPredicate(predicate, datesFromNotAfterTo());
    }

    private static LocalDate parseMonth(String value) {
        return YearMonth.parse(value, monthFormatter).atDay(1);
    }


    public void validateOfflineReportWithDailyIntervalRequest(GdOfflineReportWithDailyIntervalRequest input) {
        gridValidationService.applyValidator(offlineReportWithDailyIntervalRequestValidator, input, false);
    }

    private final Validator<GdOfflineReportWithDailyIntervalRequest, Defect> offlineReportWithDailyIntervalRequestValidator =
            req -> {
                ModelItemValidationBuilder<GdOfflineReportWithDailyIntervalRequest> vb =
                        ModelItemValidationBuilder.of(req);
                var vbReportType = vb.item(REPORT_TYPE);
                var reportType = req.getReportType();
                var vbDateFrom = vb.item(DATE_FROM);
                var vbDateTo = vb.item(DATE_TO);
                vbReportType
                        .check(notNull())
                        .check(inSet(OFFLINE_REPORT_TYPES_WITH_DAILY_INTERVAL), When.isValid());
                vbDateFrom
                        .check(notNull())
                        .check(isNotAfterMaxDailyReportDate(reportType), When.isValid())
                        .check(isNotBeforeMinDailyReportDate(reportType), When.isValidBoth(vbReportType));
                vbDateTo
                        .check(notNull())
                        .check(isNotAfterMaxDailyReportDate(reportType), When.isValid())
                        .check(isNotBeforeThan(req.getDateFrom()), When.isValidBoth(vbDateFrom))
                        .check(isNotAfterThan(req.getDateFrom()
                                        .plus(AGENCY_KPI_OFFLINE_REPORT_MAX_PERIOD_MONTHS).minusDays(1)),
                                When.isValidBoth(vbDateFrom));
                return vb.getResult();
            };

    private Constraint<LocalDate, Defect> isNotBeforeMinDailyReportDate(OfflineReportType reportType) {
        switch (reportType) {
            case AGENCY_KPI:
                LocalDate minDate = getAgencyKpiMinimumAvailableDate();
                return isNotBeforeThan(minDate);
            default:
                throw new IllegalArgumentException("Forbidden for daily intervals report type " + reportType);
        }
    }

    private Constraint<LocalDate, Defect> isNotAfterMaxDailyReportDate(OfflineReportType reportType) {
        switch (reportType) {
            case AGENCY_KPI:
                LocalDate maxDate = getAgencyKpiMaximumAvailableDate();
                return isNotAfterThan(maxDate);
            default:
                throw new IllegalArgumentException("Forbidden for daily intervals report type " + reportType);
        }
    }

    public LocalDate getAgencyKpiMaximumAvailableDate() {
        return ppcPropertiesSupport.get(AGENCY_KPI_OFFLINE_REPORT_MAX_DATE)
                .getOrDefault(LocalDate.now().minusDays(AGENCY_KPI_OFFLINE_REPORT_MAX_DATE_GAP));
    }


    public String getAgencyKpiMaximumAvailableDateAsString() {
        return getAgencyKpiMaximumAvailableDate().format(DateTimeFormatter.ISO_LOCAL_DATE);
    }

    public LocalDate getAgencyKpiMinimumAvailableDate() {
        return ppcPropertiesSupport.get(AGENCY_KPI_OFFLINE_REPORT_MIN_DATE)
                .getOrDefault(DEFAULT_AGENCY_KPI_OFFLINE_REPORT_MIN_DATE);
    }

    public String getAgencyKpiMinimumAvailableDateAsString() {
        return getAgencyKpiMinimumAvailableDate().format(DateTimeFormatter.ISO_LOCAL_DATE);
    }
}
