package ru.yandex.direct.grid.processing.service.operator;

import java.math.BigDecimal;
import java.time.Instant;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.BooleanSupplier;
import java.util.function.Supplier;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds;
import ru.yandex.direct.core.entity.client.service.AgencyClientRelationService;
import ru.yandex.direct.core.entity.client.service.ClientLimitsService;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.freelancer.service.FreelancerService;
import ru.yandex.direct.core.entity.payment.service.AutopayService;
import ru.yandex.direct.core.entity.promoextension.PromoExtensionRepository;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.grid.model.campaign.GdiBaseCampaign;
import ru.yandex.direct.grid.model.campaign.GdiCampaignStrategyName;
import ru.yandex.direct.grid.processing.model.campaign.GdWallet;
import ru.yandex.direct.grid.processing.model.campaign.GdWalletAction;
import ru.yandex.direct.grid.processing.model.client.GdClientAccess;
import ru.yandex.direct.grid.processing.model.client.GdClientAutoOverdraftInfo;
import ru.yandex.direct.grid.processing.model.client.GdClientFeatures;
import ru.yandex.direct.grid.processing.model.client.GdClientInfo;
import ru.yandex.direct.grid.processing.service.campaign.CampaignInfoService;
import ru.yandex.direct.grid.processing.service.campaign.CampaignServiceUtils;
import ru.yandex.direct.grid.processing.service.client.ClientDataService;
import ru.yandex.direct.rbac.RbacClientsRelations;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.rbac.model.ClientsRelation;
import ru.yandex.direct.utils.NumberUtils;

import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds.UNDER_WALLET;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.hasOneOfRoles;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isAgency;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isDeveloper;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isInternalAdManager;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isManager;
import static ru.yandex.direct.grid.processing.service.autooverdraft.converter.AutoOverdraftDataConverter.toClientAutoOverdraftInfo;
import static ru.yandex.direct.rbac.RbacRole.AGENCY;
import static ru.yandex.direct.utils.CommonUtils.memoize;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.Predicates.not;

/**
 * Сервис для получения информации о правах и возможностях оператора
 */
@Service
@ParametersAreNonnullByDefault
public class OperatorAccessService {
    private final RbacService rbacService;
    private final RbacClientsRelations rbacClientsRelations;
    private final ClientService clientService;
    private final AgencyClientRelationService agencyClientRelationService;
    private final UserService userService;
    private final FeatureService featureService;
    private final FreelancerService freelancerService;
    private final AutopayService autopayService;
    private final CampaignInfoService campaignInfoService;
    private final ClientDataService clientDataService;
    private final PromoExtensionRepository promoExtensionRepository;
    private final ClientLimitsService clientLimitsService;

    @Autowired
    public OperatorAccessService(RbacService rbacService, RbacClientsRelations rbacClientsRelations,
                                 ClientService clientService, AgencyClientRelationService agencyClientRelationService,
                                 UserService userService, FreelancerService freelancerService,
                                 AutopayService autopayService, FeatureService featureService,
                                 CampaignInfoService campaignInfoService, ClientDataService clientDataService,
                                 PromoExtensionRepository promoExtensionRepository, ClientLimitsService clientLimitsService) {
        this.rbacService = rbacService;
        this.rbacClientsRelations = rbacClientsRelations;
        this.clientService = clientService;
        this.agencyClientRelationService = agencyClientRelationService;
        this.userService = userService;
        this.featureService = featureService;
        this.freelancerService = freelancerService;
        this.autopayService = autopayService;
        this.campaignInfoService = campaignInfoService;
        this.clientDataService = clientDataService;
        this.promoExtensionRepository = promoExtensionRepository;
        this.clientLimitsService = clientLimitsService;
    }

    public GdClientAccess getAccess(User operator, GdClientInfo clientInfo, User subjectUser, Instant instant) {
        Supplier<List<GdiBaseCampaign>> campaignsAndWalletsSupplier = memoize(() -> campaignInfoService
                .getAllBaseCampaigns(ClientId.fromLong(clientInfo.getId())));

        Supplier<Collection<GdiBaseCampaign>> nonWalletCampaignsSupplier = memoize(() ->
                filterList(campaignsAndWalletsSupplier.get(), not(CampaignServiceUtils::isWallet)));

        Supplier<List<GdWallet>> wallets = memoize(() -> campaignInfoService.extractWalletsList(operator,
                clientInfo,
                toClientAutoOverdraftInfo(clientInfo),
                campaignsAndWalletsSupplier.get(),
                instant));

        Supplier<GdClientFeatures> clientFeaturesSupplier = memoize(() -> clientDataService.getClientFeatures(
                ClientId.fromLong(clientInfo.getId()), operator, nonWalletCampaignsSupplier.get()));

        var container = new CalcClientAccessContainer(subjectUser, clientInfo,
                nonWalletCampaignsSupplier,
                memoize(() -> listToMap(wallets.get(), GdWallet::getId)),
                memoize(() -> getFirstWallet(wallets.get())),
                clientFeaturesSupplier
        );
        return getAccess(operator, container);
    }

    /**
     * Возвращает первый кошелек с валютой не YND_FIXED
     * Если такого нет, то первый из списка
     * Если список пустой, то null
     *
     * @param wallets список кошельков
     * @return кошелек, подходящий под условия
     */
    private static Optional<GdWallet> getFirstWallet(List<GdWallet> wallets) {
        if (wallets.isEmpty()) {
            return Optional.empty();
        }

        return wallets.stream()
                .filter(w -> w.getCurrency() != CurrencyCode.YND_FIXED)
                .findFirst()
                .or(() -> Optional.of(wallets.get(0)));
    }

    /**
     * Получить список возможностей оператора, по отношению к определенному клиенту
     *
     * @param operator  оператор
     * @param container контейнер с необходимыми для вычисления данными по клиенту
     */
    private GdClientAccess getAccess(User operator, CalcClientAccessContainer container) {
        ClientId clientId = ClientId.fromLong(container.getGdClientInfo().getId());
        Long subjectUserUid = container.getSubjectUser().getUid();
        var uidAndClientId = UidAndClientId.of(subjectUserUid, clientId);
        return getAccess(operator, Map.of(uidAndClientId, container)).get(uidAndClientId);
    }

    /**
     * Получить список возможностей оператора, по отношению к набору клиентов
     *
     * @param operator            оператор
     * @param calcAccessContainer мапа, содержащая для пар id клиента и uid subjectUser необходимые для вычисления
     *                            данные
     * @return мапа, в котором паре id клиента и uid subjectUser ставится в соответствие набор прав
     */
    private Map<UidAndClientId, GdClientAccess> getAccess(User operator,
                                                          Map<UidAndClientId, CalcClientAccessContainer>
                                                                  calcAccessContainer) {
        Set<ClientId> clientsWithApiEnabled = clientService
                .clientIdsWithApiEnabled(mapList(calcAccessContainer.keySet(), t -> t.getClientId().asLong()));
        boolean operatorDisallowMoneyTransfer = operator.getIsReadonlyRep()
                || userService.getUserAgencyDisallowMoneyTransfer(operator.getChiefUid());

        Set<ClientId> allowableToBindClients = agencyClientRelationService
                .getAllowableToBindClients(operator.getClientId(),
                        mapList(calcAccessContainer.keySet(), UidAndClientId::getClientId));
        Set<Long> unarchivedAgencyClients = agencyClientRelationService
                .getUnArchivedAgencyClients(operator.getClientId().asLong(),
                        mapList(calcAccessContainer.keySet(), t -> t.getClientId().asLong()));

        Set<String> operatorFeatures = featureService.getEnabledForClientId(operator.getClientId());

        return EntryStream.of(calcAccessContainer)
                .mapToValue((uidAndClientId, calcClientAccessContainer) -> getAccess(
                        operator, operatorFeatures, operatorDisallowMoneyTransfer,
                        allowableToBindClients, unarchivedAgencyClients,
                        calcClientAccessContainer,
                        clientsWithApiEnabled.contains(uidAndClientId.getClientId())
                ))
                .toMap();
    }

    private GdClientAccess getAccess(User operator, Set<String> operatorFeatures, boolean operatorDisallowMoneyTransfer,
                                     Set<ClientId> allowableToBindClients, Set<Long> unarchivedAgencyClients,
                                     CalcClientAccessContainer calcClientAccessContainer,
                                     boolean isApiEnabledFeature) {
        ClientId clientId = ClientId.fromLong(calcClientAccessContainer.getGdClientInfo().getId());
        Long agencyClientId = calcClientAccessContainer.getGdClientInfo().getAgencyClientId();
        boolean isOperatorMccControlClient = rbacService.isUserMccControlClient(operator.getUid());
        boolean isOperatorMccForClient = rbacService.isOperatorMccForClient(operator.getUid(), clientId.asLong());
        boolean operatorIsFreelancer = freelancerService.isFreelancer(operator.getClientId());
        boolean superSubclient = clientService.isSuperSubclient(clientId);
        boolean isOperatorManagerOfAgency = agencyClientId != null && isManager(operator) &&
                rbacService.isOperatorManagerOfAgency(operator.getUid(),
                        calcClientAccessContainer.getGdClientInfo().getAgencyUserId(), agencyClientId);

        boolean isSubclientManagedByRelatedClient =
                rbacService.isRelatedClient(operator.getClientId(), clientId);
        BooleanSupplier canUseXLS = () -> !operator.getIsReadonlyRep() && canUseXLS(operator.getUid(), calcClientAccessContainer);

        boolean isSocialAdvertising = featureService.isEnabledForClientId(clientId, FeatureName.SOCIAL_ADVERTISING);
        boolean socialPaymentCondition = !isSocialAdvertising || featureService.isEnabledForClientId(clientId,
                FeatureName.SOCIAL_ADVERTISING_PAYABLE);

        Supplier<GdWallet> firstWallet = () -> calcClientAccessContainer.getFirstWallet();
        BooleanSupplier walletCanEnable = () -> canEnableWallet(operator, firstWallet.get(),
                calcClientAccessContainer.getClientNonWalletCampaigns()) && socialPaymentCondition;
        BooleanSupplier walletAllowPay = () -> isOperatorAllowPayWallet(operator, firstWallet.get(), clientId)
                && socialPaymentCondition;
        BooleanSupplier walletCanUseOverdraftRest = () ->
                canUseOverdraftRestInWallet(calcClientAccessContainer.getGdClientInfo(),
                        superSubclient, firstWallet.get()) && socialPaymentCondition;
        BooleanSupplier walletIdLinkToBalanceDisabled = () -> isWalletIdLinkDisabled(operator, firstWallet.get())
                && socialPaymentCondition;
        BooleanSupplier canUseAutobudgetWeekBundle = () ->
                !featureService.isEnabledForClientId(clientId, FeatureName.DISABLE_AUTOBUDGET_WEEK_BUNDLE) &&
                        (!featureService.isEnabledForClientId(clientId, FeatureName.HIDE_AUTOBUDGET_WEEK_BUNDLE)
                                || hasNotArchivedAutobudgetWeekBundle(calcClientAccessContainer.getClientNonWalletCampaigns()));
        boolean walletCanUseAutopay = !isOperatorMccForClient && autopayService.canUseAutopay(
                calcClientAccessContainer.getGdClientInfo().getShard(),
                calcClientAccessContainer.getGdClientInfo().getChiefUserId(), clientId) && socialPaymentCondition;
        BooleanSupplier walletAllowedEdit = () ->
                !isOperatorMccForClient && isAllowedEditCamps(operator, firstWallet.get(), superSubclient) && socialPaymentCondition;
        boolean walletReadOnly = isOperatorMccForClient || isReadOnlyWallet(operator) && socialPaymentCondition;
        BooleanSupplier showBalanceLink = () -> showBalanceLink(operator, firstWallet.get()) && socialPaymentCondition;

        final ClientsRelation internalAdProductRelation;
        if (isInternalAdManager(operator)) {
            internalAdProductRelation =
                    rbacClientsRelations.getInternalAdProductRelation(operator.getClientId(), clientId)
                            .orElse(null);
        } else {
            internalAdProductRelation = null;
        }
        Set<ClientId> unBindedAgencies = agencyClientRelationService.getUnbindedAgencies(clientId);


        boolean canWrite = rbacService.canWrite(operator.getUid(),
                calcClientAccessContainer.getGdClientInfo().getChiefUserId());
        boolean canRead = operatorCanRead(operator.getUid(),
                calcClientAccessContainer.getGdClientInfo().getChiefUserId());
        boolean userAgencyIsNoPay = userService.getUserAgencyIsNoPay(operator.getUid());
        boolean canBindClientToAgency = isAgency(operator) && allowableToBindClients.contains(clientId);
        boolean agencyClientUnarchived = isAgency(operator) && unarchivedAgencyClients.contains(clientId.asLong());
        boolean canSeeRepresentativeClientsLink = isAgency(operator) && agencyClientId != null;
        boolean isAgencyServiceStopped =
                agencyClientId != null && unBindedAgencies.contains(ClientId.fromLong(agencyClientId));
        boolean canAgencyRepresentativeBeSet =
                canAgencyRepresentativeBeSet(calcClientAccessContainer.getGdClientInfo());

        boolean showManagerControls = RbacRole.MANAGER.anyOf(operator.getRole(),
                calcClientAccessContainer.getSubjectUser().getRole())
                && !RbacRole.AGENCY.equals(calcClientAccessContainer.getSubjectUser().getRole());
        boolean showAgencyControls =
                RbacRole.AGENCY.anyOf(operator.getRole(), calcClientAccessContainer.getSubjectUser().getRole()) &&
                        !RbacRole.MANAGER.equals(calcClientAccessContainer.getSubjectUser().getRole());
        BooleanSupplier hasMaximalNumberOfPromoExtensions = () -> promoExtensionRepository
                .getAllClientPromoExtensionIds(calcClientAccessContainer.getGdClientInfo().getShard(), clientId)
                .size() <= clientLimitsService.getClientLimits(clientId).getPromoExtensionCountLimitOrDefault();

        BooleanSupplier isCreativeFreeInterfaceEnabled = () -> featureService
                .isEnabledForClientId(clientId, FeatureName.CREATIVE_FREE_INTERFACE);

        return new OperatorClientRelations(operator, operatorFeatures, calcClientAccessContainer,
                superSubclient, canUseXLS,
                walletReadOnly,
                walletAllowedEdit,
                walletCanEnable,
                walletAllowPay,
                walletCanUseOverdraftRest,
                walletIdLinkToBalanceDisabled,
                walletCanUseAutopay,
                showBalanceLink,
                operatorDisallowMoneyTransfer,
                isOperatorManagerOfAgency, operatorIsFreelancer, isOperatorMccControlClient, isOperatorMccForClient, isSubclientManagedByRelatedClient,
                internalAdProductRelation, canWrite, canRead, isApiEnabledFeature,
                userAgencyIsNoPay,
                canBindClientToAgency, agencyClientUnarchived,
                canSeeRepresentativeClientsLink, isAgencyServiceStopped, canAgencyRepresentativeBeSet,
                showManagerControls, showAgencyControls, isSocialAdvertising, socialPaymentCondition,
                canUseAutobudgetWeekBundle, hasMaximalNumberOfPromoExtensions, isCreativeFreeInterfaceEnabled);
    }

    protected boolean operatorCanRead(Long operatorUid, Long clientUid) {
        return rbacService.canRead(operatorUid, clientUid);
    }

    private boolean canUseXLS(long operatorUid, CalcClientAccessContainer container) {
        return rbacService.canImportXLSIntoNewCampaign(operatorUid,
                container.getGdClientInfo().getChiefUserId(), ClientId.fromLong(container.getGdClientInfo().getId())) ||
                hasCampaignWithAllowImportXLS(operatorUid, container.getClientNonWalletCampaigns());
    }

    private boolean canEnableWallet(User operator, @Nullable GdWallet wallet,
                                    Collection<GdiBaseCampaign> campaigns) {
        if (wallet != null && wallet.getStatus().getEnabled()) {
            return false;
        }

        boolean isReadOnlyWallet = isReadOnlyWallet(operator);
        boolean existsCampaignCanBeUnderWallet = campaigns.stream()
                .map(GdiBaseCampaign::getType)
                .anyMatch(UNDER_WALLET::contains);

        return existsCampaignCanBeUnderWallet && !isReadOnlyWallet;
    }

    private boolean isOperatorAllowPayWallet(User operator, @Nullable GdWallet wallet, ClientId clientId) {
        if (wallet == null) {
            return false;
        }

        if (rbacService.isOperatorMccForClient(operator.getUid(), clientId.asLong())) {
            return false;
        }

        boolean walletIsEnabled = wallet.getStatus().getEnabled();
        boolean isReadOnlyWallet = isReadOnlyWallet(operator);
        boolean isAllowedPay = wallet.getActions().contains(GdWalletAction.PAY);

        return isAllowedPay && !isReadOnlyWallet && walletIsEnabled;
    }

    private boolean canUseOverdraftRestInWallet(GdClientInfo clientInfo, boolean superSubclient,
                                                @Nullable GdWallet wallet) {
        if (wallet == null || !wallet.getStatus().getEnabled()) {
            return false;
        }

        GdClientAutoOverdraftInfo autoOverdraftInfo = clientInfo.getAutoOverdraftInfo();
        BigDecimal overdraftRest = autoOverdraftInfo.getOverdraftRest();

        return NumberUtils.greaterThanZero(overdraftRest) &&
                (superSubclient || clientInfo.getAgencyInfo() == null);
    }

    private boolean isWalletIdLinkDisabled(User operator, @Nullable GdWallet wallet) {
        if (wallet == null || !wallet.getStatus().getEnabled()) {
            return true;
        }

        return hasOneOfRoles(operator, RbacRole.MANAGER) ||
                hasOneOfRoles(operator, RbacRole.CLIENT) && wallet.getIsAgencyWallet();
    }

    private boolean showBalanceLink(User operator, @Nullable GdWallet wallet) {
        if (wallet == null || !wallet.getStatus().getEnabled()) {
            return false;
        }

        return hasOneOfRoles(operator, RbacRole.SUPER, RbacRole.MANAGER, RbacRole.SUPPORT, RbacRole.LIMITED_SUPPORT,
                RbacRole.SUPERREADER);
    }

    private boolean isReadOnlyWallet(User operator) {
        return operator.getIsReadonlyRep() || hasOneOfRoles(operator, RbacRole.PLACER, RbacRole.MEDIA, RbacRole.SUPERREADER);
    }

    private boolean isAllowedEditCamps(User operator, @Nullable GdWallet wallet, boolean superSubclient) {
        if (wallet == null) {
            return false;
        }

        if (operator.getIsReadonlyRep()) {
            return false;
        }

        boolean isSuperAccess = isSuperAccess(operator);
        return !wallet.getIsAgencyWallet() || superSubclient || isSuperAccess;
    }

    private boolean isSuperAccess(User operator) {
        return hasOneOfRoles(operator, AGENCY, RbacRole.SUPER, RbacRole.MANAGER,
                RbacRole.SUPPORT) || isDeveloper(operator) && !hasOneOfRoles(operator, RbacRole.SUPERREADER);
    }

    private boolean hasCampaignWithAllowImportXLS(long operatorUid, Collection<GdiBaseCampaign> campaigns) {
        Set<Long> campaignsWithXLSType = campaigns.stream()
                .filter(c -> CampaignTypeKinds.XLS.contains(c.getType()))
                .map(GdiBaseCampaign::getId)
                .collect(toSet());

        return !rbacService.getCampaignsAllowImportXLS(operatorUid, campaignsWithXLSType).isEmpty();
    }

    private boolean canAgencyRepresentativeBeSet(GdClientInfo clientInfo) {
        return clientInfo.getAgencyUserId() != null &&
                rbacService.getChief(clientInfo.getAgencyUserId()) == clientInfo.getAgencyUserId();
    }

    private static boolean hasNotArchivedAutobudgetWeekBundle(Collection<GdiBaseCampaign> campaigns) {
        return StreamEx.of(campaigns)
                .remove(GdiBaseCampaign::getArchived)
                .map(GdiBaseCampaign::getStrategyName)
                .anyMatch(GdiCampaignStrategyName.AUTOBUDGET_WEEK_BUNDLE::equals);
    }
}
