package ru.yandex.direct.grid.processing.service.operator;

import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import one.util.streamex.EntryStream;

import ru.yandex.direct.core.entity.campaign.service.validation.CampaignWithPricePackagePermissionUtils;
import ru.yandex.direct.core.entity.pricepackage.service.PricePackagePermissionUtils;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.env.Environment;
import ru.yandex.direct.grid.core.entity.campaign.service.GridCampaignAccessServiceUtils;
import ru.yandex.direct.grid.processing.model.client.GdOperatorAction;

import static ru.yandex.direct.core.entity.user.utils.UserUtil.hasOneOfRoles;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isClient;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isDeveloper;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isSuper;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isSuperPlacer;
import static ru.yandex.direct.rbac.RbacRole.AGENCY;
import static ru.yandex.direct.rbac.RbacRole.CLIENT;
import static ru.yandex.direct.rbac.RbacRole.INTERNAL_AD_ADMIN;
import static ru.yandex.direct.rbac.RbacRole.INTERNAL_AD_MANAGER;
import static ru.yandex.direct.rbac.RbacRole.INTERNAL_AD_SUPERREADER;
import static ru.yandex.direct.rbac.RbacRole.LIMITED_SUPPORT;
import static ru.yandex.direct.rbac.RbacRole.MANAGER;
import static ru.yandex.direct.rbac.RbacRole.PLACER;
import static ru.yandex.direct.rbac.RbacRole.SUPER;
import static ru.yandex.direct.rbac.RbacRole.SUPERREADER;
import static ru.yandex.direct.rbac.RbacRole.SUPPORT;

@ParametersAreNonnullByDefault
public class OperatorAllowedActionsUtils {

    /**
     * Хеш, в котором каждому из расчитываемых здесь действий оператора ставится в соответствие предикат,
     * который должен выполняться для того, чтобы мы добавили действие в набор доступных
     */
    private static final Map<GdOperatorAction, Predicate<User>> ACTION_PREDICATES =
            ImmutableMap.<GdOperatorAction, Predicate<User>>builder()
                    .put(GdOperatorAction.REMODERATE_CAMPAIGN, GridCampaignAccessServiceUtils::getCanRemoderateCampaign)
                    .put(GdOperatorAction.COPY, OperatorAllowedActionsUtils::getCanCopy)
                    .put(GdOperatorAction.SEND_TO_BS, OperatorAllowedActionsUtils::getCanSendToBS)
                    .put(GdOperatorAction.SEND_TO_MODERATION, OperatorAllowedActionsUtils::getCanSendToModeration)
                    .put(GdOperatorAction.SEND_TO_MODERATION_BY_CLIENT,
                            OperatorAllowedActionsUtils::getCanSendToModerationByClient)
                    .put(GdOperatorAction.SEND_TO_REMODERATION, OperatorAllowedActionsUtils::getCanSendToRemoderation)
                    .put(GdOperatorAction.ACCEPT_MODERATION, OperatorAllowedActionsUtils::getCanAcceptModeration)
                    .put(GdOperatorAction.REMODERATE_ADS_CALLOUTS,
                            OperatorAllowedActionsUtils::getCanRemoderateAdsCallouts)
                    .put(GdOperatorAction.ACCEPT_ADS_CALLOUTS_MODERATION,
                            OperatorAllowedActionsUtils::getCanAcceptAdsCalloutsModeration)
                    .put(GdOperatorAction.CAN_OPEN_PRICE_PACKAGE_GRID,
                            PricePackagePermissionUtils::canViewPricePackages)
                    .put(GdOperatorAction.MANAGE_PRICE_PACKAGES, PricePackagePermissionUtils::canManagePricePackages)
                    .put(GdOperatorAction.MANAGE_PRICE_PACKAGE_CLIENTS,
                            PricePackagePermissionUtils::canManagePricePackageClients)
                    .put(GdOperatorAction.APPROVE_PRICE_PACKAGES, PricePackagePermissionUtils::canApprovePricePackages)
                    .put(GdOperatorAction.APPROVE_PRICE_SALES_CAMPAIGNS,
                            CampaignWithPricePackagePermissionUtils::canApprovePriceSalesCampaigns)
                    .put(GdOperatorAction.MODIFY_DB, OperatorAllowedActionsUtils::canModifyDb)
                    .put(GdOperatorAction.CAN_VIEW_CLIENT_COUNTERS, OperatorAllowedActionsUtils::canViewClientCounters)
                    .put(GdOperatorAction.CAN_VIEW_ADD_COUNTERS_CONTROL,
                            OperatorAllowedActionsUtils::canViewAddCountersControl)
                    .put(GdOperatorAction.CAN_IMPORT_EXCEL_FOR_INTERNAL_ADS,
                            OperatorAllowedActionsUtils::canImportExcelForInternalAds)
                    .build();

    public static Set<GdOperatorAction> getActions(User operator) {
        return EntryStream.of(ACTION_PREDICATES)
                .filterValues(p -> p.test(operator))
                .keys()
                .toSet();
    }

    public static boolean hasAction(GdOperatorAction action, User operator) {
        if (ACTION_PREDICATES.containsKey(action)) {
            return ACTION_PREDICATES.get(action).test(operator);
        }

        return false;
    }

    private static boolean getCanCopy(User operator) {
        return hasOneOfRoles(operator, SUPER, PLACER, MANAGER, AGENCY, CLIENT, INTERNAL_AD_ADMIN, INTERNAL_AD_MANAGER)
                && !operator.getIsReadonlyRep();
    }

    private static boolean getCanSendToBS(User operator) {
        return hasOneOfRoles(operator, SUPER, SUPERREADER, PLACER, SUPPORT, LIMITED_SUPPORT) || isDeveloper(operator);
    }

    private static boolean getCanSendToRemoderation(User operator) {
        return hasOneOfRoles(operator, SUPER, PLACER, SUPPORT, MANAGER, LIMITED_SUPPORT) || isDeveloper(operator);
    }

    private static boolean getCanSendToModeration(User operator) {
        return hasOneOfRoles(operator, SUPER, PLACER, SUPPORT, MANAGER, AGENCY) || isDeveloper(operator);
    }

    private static boolean getCanSendToModerationByClient(User operator) {
        return isClient(operator) && !operator.getIsReadonlyRep();
    }

    private static boolean getCanAcceptModeration(User operator) {
        return hasOneOfRoles(operator, SUPER, SUPPORT) || isSuperPlacer(operator) || isDeveloper(operator);
    }

    private static boolean getCanRemoderateAdsCallouts(User operator) {
        return hasOneOfRoles(operator, SUPER, PLACER, SUPPORT, LIMITED_SUPPORT) || isDeveloper(operator);
    }

    private static boolean getCanAcceptAdsCalloutsModeration(User operator) {
        //доступ аналогичен доступу для групп: DIRECT-94104
        return getCanAcceptModeration(operator);
    }

    private static boolean canModifyDb(User operator) {
        return isSuper(operator) && !Environment.getCached().isProductionOrPrestable();
    }

    public static boolean canBlockUser(User operator) {
        return hasOneOfRoles(operator, SUPER, PLACER, SUPPORT) || isDeveloper(operator);
    }

    private static boolean canViewClientCounters(User operator) {
        return hasOneOfRoles(operator, LIMITED_SUPPORT, AGENCY) || operator.getRole().isInternal();
    }

    private static boolean canViewAddCountersControl(User operator) {
        return hasOneOfRoles(operator, LIMITED_SUPPORT, AGENCY);
    }

    public static boolean canImportExcelForInternalAds(User operator) {
        return hasOneOfRoles(operator, SUPER, INTERNAL_AD_ADMIN, INTERNAL_AD_MANAGER, INTERNAL_AD_SUPERREADER);
    }

}
