package ru.yandex.direct.grid.processing.service.operator;

import java.util.Collections;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.BooleanSupplier;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import io.leangen.graphql.annotations.GraphQLNonNull;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.common.enums.YandexDomain;
import ru.yandex.direct.common.util.HostUtils;
import ru.yandex.direct.core.entity.banner.service.validation.ModerateBannerValidationService;
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusModerate;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignWithPricePackagePermissionUtils;
import ru.yandex.direct.core.entity.region.RegionConstants;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.utils.UserUtil;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.grid.model.campaign.GdCampaignAccess;
import ru.yandex.direct.grid.model.campaign.GdCampaignAction;
import ru.yandex.direct.grid.model.campaign.GdCampaignType;
import ru.yandex.direct.grid.model.campaign.GdiBaseCampaign;
import ru.yandex.direct.grid.model.campaign.GdiCampaignAction;
import ru.yandex.direct.grid.model.entity.campaign.strategy.GdStrategyExtractorFacade;
import ru.yandex.direct.grid.processing.model.campaign.GdWallet;
import ru.yandex.direct.grid.processing.model.client.GdClientAccess;
import ru.yandex.direct.grid.processing.model.client.GdUserInfo;
import ru.yandex.direct.grid.processing.service.campaign.CampaignAccessHelper;
import ru.yandex.direct.rbac.RbacRepType;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.rbac.model.ClientsRelation;
import ru.yandex.direct.rbac.model.ClientsRelationType;

import static ru.yandex.direct.core.entity.user.utils.UserUtil.hasOneOfRoles;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isAgency;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isAnyTeamLeader;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isClient;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isInternalAdManager;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isInternalAdRole;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isLimitedSupport;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isManager;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isMediaPlanner;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isSuper;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isSuperReader;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isSupport;
import static ru.yandex.direct.grid.model.entity.campaign.strategy.GdStrategyExtractorHelper.STRATEGIES_EXTRACTORS_BY_TYPES;
import static ru.yandex.direct.grid.processing.service.operator.OperatorAllowedActionsUtils.canBlockUser;
import static ru.yandex.direct.rbac.RbacRole.AGENCY;
import static ru.yandex.direct.rbac.RbacRole.CLIENT;
import static ru.yandex.direct.rbac.RbacRole.EMPTY;
import static ru.yandex.direct.rbac.RbacRole.INTERNAL_AD_ADMIN;
import static ru.yandex.direct.rbac.RbacRole.INTERNAL_AD_MANAGER;
import static ru.yandex.direct.rbac.RbacRole.INTERNAL_AD_SUPERREADER;
import static ru.yandex.direct.rbac.RbacRole.LIMITED_SUPPORT;
import static ru.yandex.direct.rbac.RbacRole.MANAGER;
import static ru.yandex.direct.rbac.RbacRole.MEDIA;
import static ru.yandex.direct.rbac.RbacRole.PLACER;
import static ru.yandex.direct.rbac.RbacRole.SUPER;
import static ru.yandex.direct.rbac.RbacRole.SUPERREADER;
import static ru.yandex.direct.rbac.RbacRole.SUPPORT;
import static ru.yandex.direct.regions.Region.BY_REGION_ID;
import static ru.yandex.direct.regions.Region.KAZAKHSTAN_REGION_ID;
import static ru.yandex.direct.regions.Region.RUSSIA_REGION_ID;
import static ru.yandex.direct.utils.CommonUtils.isValidId;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;

/**
 * Класс возвращающий флаги, рассказывающие о том, что оператор может сделать с клиентом
 */
@ParametersAreNonnullByDefault
//todo ssdmitriev DIRECT-107839: разбить на модель и объект управляющей ею.
public class OperatorClientRelations implements GdClientAccess {

    static final CampaignAccessHelper CAMPAIGN_ACCESS_HELPER =
            new CampaignAccessHelper(new GdStrategyExtractorFacade(STRATEGIES_EXTRACTORS_BY_TYPES));

    /**
     * GdCampaignType генерится, туда ничего не можем добавить, поэтому делаем wrapper для добавления методов
     * NB: большая часть видов кампаний недоступна для создания в продуктах внутренней рекламы
     */
    private enum CampaignTypes {
        TEXT(GdCampaignType.TEXT) {
            boolean canCreateThisType(OperatorClientRelations r) {
                return !r.clientIsInternalAdProduct.getAsBoolean();
            }
        },
        DYNAMIC(GdCampaignType.DYNAMIC) {
            boolean canCreateThisType(OperatorClientRelations r) {
                return !r.clientIsInternalAdProduct.getAsBoolean()
                        && !r.isSocialAdvertisement.getAsBoolean()
                        && (!r.clientIsTurkish.getAsBoolean() || isSuper(r.operator));
            }
        },
        MOBILE_CONTENT(GdCampaignType.MOBILE_CONTENT) {
            boolean canCreateThisType(OperatorClientRelations r) {
                return !r.clientIsInternalAdProduct.getAsBoolean() && !r.isSocialAdvertisement.getAsBoolean();
            }
        },
        PERFORMANCE(GdCampaignType.PERFORMANCE) {
            boolean canCreateThisType(OperatorClientRelations r) {
                return !r.clientIsInternalAdProduct.getAsBoolean()
                        && !r.isSocialAdvertisement.getAsBoolean();
            }
        },
        MCBANNER(GdCampaignType.MCBANNER) {
            boolean canCreateThisType(OperatorClientRelations r) {
                return !r.clientIsInternalAdProduct.getAsBoolean() && !r.isSocialAdvertisement.getAsBoolean();
            }
        },
        CPM_BANNER(GdCampaignType.CPM_BANNER) {
            boolean canCreateThisType(OperatorClientRelations r) {
                return !r.clientIsInternalAdProduct.getAsBoolean() && !r.isSocialAdvertisement.getAsBoolean();
            }
        },
        CPM_DEALS(GdCampaignType.CPM_DEALS) {
            boolean canCreateThisType(OperatorClientRelations r) {
                return !r.clientIsInternalAdProduct.getAsBoolean() && !r.isSocialAdvertisement.getAsBoolean() &&
                        (r.operatorHasCmpDealsFeature.getAsBoolean() || isSuper(r.operator));
            }
        },
        INTERNAL_AUTOBUDGET(GdCampaignType.INTERNAL_AUTOBUDGET) {
            boolean canCreateThisType(OperatorClientRelations r) {
                return r.clientIsInternalAdProduct.getAsBoolean() && !r.isSocialAdvertisement.getAsBoolean();
            }
        },
        INTERNAL_DISTRIB(GdCampaignType.INTERNAL_DISTRIB) {
            boolean canCreateThisType(OperatorClientRelations r) {
                return r.clientIsInternalAdProduct.getAsBoolean() && !r.isSocialAdvertisement.getAsBoolean();
            }
        },
        INTERNAL_FREE(GdCampaignType.INTERNAL_FREE) {
            boolean canCreateThisType(OperatorClientRelations r) {
                return r.clientIsInternalAdProduct.getAsBoolean() && !r.isSocialAdvertisement.getAsBoolean();
            }
        },
        CONTENT_PROMOTION(GdCampaignType.CONTENT_PROMOTION) {
            boolean canCreateThisType(OperatorClientRelations r) {
                return !r.clientIsInternalAdProduct.getAsBoolean() && !r.isSocialAdvertisement.getAsBoolean() &&
                        r.calcClientAccessContainer.getGdClientFeatures().getIsContentPromotionVideoEnabled();
            }
        },
        CPM_PRICE(GdCampaignType.CPM_PRICE) {
            boolean canCreateThisType(OperatorClientRelations r) {
                return !r.clientIsInternalAdProduct.getAsBoolean() &&
                        !r.clientCurrencyIsGBP.getAsBoolean() &&
                        r.calcClientAccessContainer.getGdClientFeatures().getIsCpmPriceCampaignEnabled();
            }
        },
        CPM_YNDX_FRONTPAGE(GdCampaignType.CPM_YNDX_FRONTPAGE) {
            @Override
            boolean canCreateThisType(OperatorClientRelations r) {
                return !r.clientIsInternalAdProduct.getAsBoolean() && !r.isSocialAdvertisement.getAsBoolean()
                        && !r.clientCurrencyIsGBP.getAsBoolean()
                        && (!r.clientIsTurkish.getAsBoolean() || isSuper(r.operator));
            }
        };


        GdCampaignType type;

        abstract boolean canCreateThisType(OperatorClientRelations r);

        CampaignTypes(GdCampaignType type) {
            this.type = type;
        }

        GdCampaignType getUnderlyingType() {
            return type;
        }

    }

    private final CalcClientAccessContainer calcClientAccessContainer;
    private final User operator;
    private final boolean operatorCanWrite;
    private final boolean operatorCanRead;

    private final BooleanSupplier socialPaymentCondition;

    private final BooleanSupplier operatorHasCmpDealsFeature;
    private final BooleanSupplier operatorHasCheckedSupportChatFeature;
    private final BooleanSupplier operatorHasDebugFeature;
    private final BooleanSupplier operatorIsManagerOfClientAgency;
    private final BooleanSupplier superSubclient;
    private final BooleanSupplier operatorDisallowMoneyTransfer;
    private final BooleanSupplier operatorIsClient;
    private final BooleanSupplier operatorIsClientsChief;
    private final BooleanSupplier operatorIsClientsManager;
    private final BooleanSupplier operatorIsClientsAgency;
    private final BooleanSupplier operatorIsFreelancer;
    private final BooleanSupplier operatorIsMccControlClient;
    private final BooleanSupplier operatorIsMccForClient;
    private final BooleanSupplier isSubclientManagedByOperator;
    private final BooleanSupplier operatorCanEditClient;
    private final BooleanSupplier operatorCanUseXLS;
    private final BooleanSupplier walletCanEnable;
    private final BooleanSupplier walletAllowPay;
    private final BooleanSupplier walletCanUseOverdraftRest;
    private final BooleanSupplier walletIdLinkToBalanceDisabled;
    private final BooleanSupplier walletCanUseAutopay;
    private final BooleanSupplier walletReadOnly;
    private final BooleanSupplier walletAllowedEdit;
    private final BooleanSupplier showBalanceLink;
    private final BooleanSupplier operatorIsManagerOfClientsCampaigns;
    private final BooleanSupplier operatorCanManageInternalAdCampaignsOfClient;
    private final BooleanSupplier operatorHasInternalAdRole;
    private final BooleanSupplier canBindClientToAgency;
    private final BooleanSupplier agencyClientUnarchived;
    private final BooleanSupplier isAgencyServiceStopped;
    private final BooleanSupplier canSeeRepresentativeClientsLink;
    private final BooleanSupplier canAgencyRepresentativeBeSet;
    private final BooleanSupplier canUseRoiStrategy;

    private final BooleanSupplier clientIsTurkish;
    private final BooleanSupplier clientCurrencyIsGBP;
    private final BooleanSupplier clientIsServiced;
    private final BooleanSupplier clientIsUnderAgency;
    private final BooleanSupplier canUseAutobudgetWeekBundle;
    private final BooleanSupplier isAgencySubclientAccessedByMcc;


    // в этом поле в GdClientFeatures по историческим причинам хранится обычно наличие internal_ad_product в
    // clients.perms
    private final BooleanSupplier clientIsInternalAdProduct;
    @Nullable
    private final ClientsRelation internalAdProductRelation;

    private final BooleanSupplier userAgencyIsNoPay;

    private final BooleanSupplier showManagerControls;
    private final BooleanSupplier showAgencyControls;
    private final BooleanSupplier isSocialAdvertisement;
    private final BooleanSupplier hasMaximalNumberOfPromoExtensions;

    private final BooleanSupplier isCreativeFreeInterfaceEnabled;

    /**
     * Мапа внешнего представления доступа оператора к кампаниям клиента
     * Нужно обращаться только через метод {@link #getGdCampaignAccessById()}
     * Инициализация ленивая, т.к. данные нужны для малого кол-во флагов.
     * CampaignId -> GdCampaignAccess
     */
    private Map<Long, GdCampaignAccess> gdCampaignAccessById;
    private final boolean isApiEnabledFeature;

    private static final Logger logger = LoggerFactory.getLogger(OperatorClientRelations.class);
    private static final Set<Long> SUPPORT_CHAT_REGION_IDS =
            Set.of(BY_REGION_ID, KAZAKHSTAN_REGION_ID, RUSSIA_REGION_ID);

    @SuppressWarnings("checkstyle:parameternumber")
    OperatorClientRelations(User operator, Set<String> operatorFeatures,
                            CalcClientAccessContainer calcClientAccessContainer,
                            boolean superSubclient,
                            BooleanSupplier operatorCanUseXLS,
                            boolean walletReadOnly,
                            BooleanSupplier walletAllowedEdit,
                            BooleanSupplier walletCanEnable, BooleanSupplier walletAllowPay,
                            BooleanSupplier walletCanUseOverdraftRest,
                            BooleanSupplier walletIdLinkToBalanceDisabled,
                            boolean walletCanUseAutopay,
                            BooleanSupplier showBalanceLink,
                            boolean operatorDisallowMoneyTransfer,
                            boolean operatorIsManagerOfClientAgency, boolean operatorIsFreelancer,
                            boolean operatorIsMccControlClient, boolean operatorIsMccForClient,
                            boolean isSubclientManagedByOperator,
                            @Nullable ClientsRelation internalAdProductRelation,
                            boolean operatorCanWrite, boolean operatorCanRead, boolean isApiEnabledFeature,
                            boolean userAgencyIsNoPay, boolean canBindClientToAgency,
                            boolean agencyClientUnarchived, boolean canSeeRepresentativeClientsLink,
                            boolean isAgencyServiceStopped, boolean canAgencyRepresentativeBeSet,
                            boolean showManagerControls, boolean showAgencyControls,
                            boolean isSocialAdvertising, boolean socialPaymentCondition,
                            BooleanSupplier canUseAutobudgetWeekBundle,
                            BooleanSupplier hasMaximalNumberOfPromoExtensions,
                            BooleanSupplier isCreativeFreeInterfaceEnabled) {
        this.operator = operator;
        this.calcClientAccessContainer = calcClientAccessContainer;
        this.walletReadOnly = () -> walletReadOnly;
        this.walletAllowedEdit = walletAllowedEdit;
        this.walletCanEnable = walletCanEnable;
        this.walletAllowPay = walletAllowPay;
        this.walletCanUseOverdraftRest = walletCanUseOverdraftRest;
        this.walletIdLinkToBalanceDisabled = walletIdLinkToBalanceDisabled;
        this.walletCanUseAutopay = () -> walletCanUseAutopay;
        this.showBalanceLink = showBalanceLink;

        this.operatorCanUseXLS = operatorCanUseXLS;
        this.superSubclient = () -> superSubclient;
        this.operatorDisallowMoneyTransfer = () -> operatorDisallowMoneyTransfer;
        this.operatorIsManagerOfClientAgency = () -> operatorIsManagerOfClientAgency;
        this.operatorIsFreelancer = () -> operatorIsFreelancer;
        this.operatorIsMccControlClient = () -> operatorIsMccControlClient;
        this.operatorIsMccForClient = () -> operatorIsMccForClient;
        this.isSubclientManagedByOperator = () -> isSubclientManagedByOperator;
        this.isAgencyServiceStopped = () -> isAgencyServiceStopped;
        this.internalAdProductRelation = internalAdProductRelation;
        this.operatorCanWrite = operatorCanWrite;
        this.operatorCanRead = operatorCanRead;
        this.userAgencyIsNoPay = () -> userAgencyIsNoPay;
        this.canUseRoiStrategy = () -> !calcClientAccessContainer.getGdClientFeatures().getIsCrrStrategyAllowed()
                || OperatorClientRelationsHelper.hasRoiStrategyInNotArchivedCampaigns(
                calcClientAccessContainer.getClientNonWalletCampaigns());

        this.isApiEnabledFeature = isApiEnabledFeature;
        this.canBindClientToAgency = () -> canBindClientToAgency;
        this.agencyClientUnarchived = () -> agencyClientUnarchived;

        this.operatorHasCmpDealsFeature = () -> operatorFeatures.contains(FeatureName.CPM_DEALS.getName());
        this.operatorHasCheckedSupportChatFeature =
                () -> operatorFeatures.contains(FeatureName.CHECKED_SUPPORT_CHAT.getName());
        this.operatorHasDebugFeature =
                () -> operatorFeatures.contains(FeatureName.DEBUG_OUTPUT_FEATURE1.getName());

        this.isSocialAdvertisement = () -> isSocialAdvertising;

        this.canSeeRepresentativeClientsLink = () -> canSeeRepresentativeClientsLink;
        this.canAgencyRepresentativeBeSet = () -> canAgencyRepresentativeBeSet;

        this.showManagerControls = () -> showManagerControls;
        this.showAgencyControls = () -> showAgencyControls;

        this.socialPaymentCondition = () -> socialPaymentCondition;
        this.canUseAutobudgetWeekBundle = canUseAutobudgetWeekBundle;


        operatorIsClient = () -> isClient(operator) && ((Long) operator.getClientId().asLong())
                .equals(calcClientAccessContainer.getGdClientInfo().getId());

        operatorIsClientsChief = () -> operatorIsClient.getAsBoolean() && operator.getUid()
                .equals(calcClientAccessContainer.getGdClientInfo().getChiefUserId());

        operatorIsClientsManager = () -> isManager(operator) &&
                calcClientAccessContainer.getGdClientInfo().getManagersInfo().stream()
                        .map(GdUserInfo::getUserId)
                        .anyMatch(operator.getUid()::equals);

        operatorIsManagerOfClientsCampaigns = () -> isManager(operator) &&
                calcClientAccessContainer.getGdClientFeatures().getCampaignManagers().contains(operator.getUid());

        operatorIsClientsAgency = () -> isAgency(operator) &&
                Objects.equals(
                        operator.getClientId().asLong(),
                        calcClientAccessContainer.getGdClientInfo().getAgencyClientId()
                );
        clientIsUnderAgency = () -> isValidId(calcClientAccessContainer.getGdClientInfo().getAgencyClientId());

        clientIsTurkish =
                () -> OperatorClientRelationsHelper.clientIsTurkish(calcClientAccessContainer.getGdClientInfo());
        clientCurrencyIsGBP =
                () -> OperatorClientRelationsHelper.clientCurrencyIsGBP(calcClientAccessContainer.getGdClientInfo());
        clientIsServiced = () -> isValidId(calcClientAccessContainer.getGdClientInfo().getManagerUserId());
        clientIsInternalAdProduct = () -> calcClientAccessContainer.getGdClientFeatures().getIsInternalAdsAllowed();

        operatorCanManageInternalAdCampaignsOfClient = () ->
                OperatorClientRelationsHelper.operatorCanManageInternalAdCampaignsOfClient(
                        operator, clientIsInternalAdProduct.getAsBoolean(), internalAdProductRelation);
        operatorCanEditClient = operatorCanEditClient();

        operatorHasInternalAdRole = () -> isInternalAdRole(operator);

        this.hasMaximalNumberOfPromoExtensions = hasMaximalNumberOfPromoExtensions;

        this.isCreativeFreeInterfaceEnabled = isCreativeFreeInterfaceEnabled;
        this.isAgencySubclientAccessedByMcc =
                () -> operatorIsMccForClient && clientIsUnderAgency.getAsBoolean() && !superSubclient;
    }

    private Map<Long, GdCampaignAccess> getGdCampaignAccessById() {
        if (gdCampaignAccessById == null) {
            gdCampaignAccessById = listToMap(calcClientAccessContainer.getClientNonWalletCampaigns(),
                    GdiBaseCampaign::getId,
                    campaign -> {
                        GdWallet wallet = isValidId(
                                campaign.getWalletId()) ? calcClientAccessContainer.getClientWallets().get(
                                campaign.getWalletId()) : null;
                        return CAMPAIGN_ACCESS_HELPER.getCampaignAccess(operator, campaign, wallet, false, false);
                    });
        }
        return gdCampaignAccessById;
    }

    /**
     * Вернуть истину, если оператор по грубой оценке может править клиента
     */
    private BooleanSupplier operatorCanEditClient() {
        return () -> {
            if (operatorCanWrite) {
                return true;
            }

            if (operator.getIsReadonlyRep()) {
                return false;
            }

            boolean clientCanEditItself = !clientIsUnderAgency.getAsBoolean() || superSubclient.getAsBoolean();
            // limited_support должен видеть интерфейс так, как клиент
            boolean operatorCanEditExactlyAsClient = isLimitedSupport(operator) && clientCanEditItself;
            return operatorCanEditExactlyAsClient ||
                    (operatorIsClient.getAsBoolean() && clientCanEditItself) ||
                    (isSubclientManagedByOperator.getAsBoolean() && clientCanEditItself) ||
                    operatorIsClientsManager.getAsBoolean() ||
                    operatorIsClientsAgency.getAsBoolean() || operatorIsManagerOfClientAgency.getAsBoolean() ||
                    operatorCanManageInternalAdCampaignsOfClient.getAsBoolean() ||
                    isSuper(operator) || isSupport(operator);
        };
    }

    @Override
    public @GraphQLNonNull
    Boolean getOperatorIsClient() {
        return operatorIsClient.getAsBoolean();
    }

    @Override
    public @GraphQLNonNull
    Boolean getOperatorIsClientChief() {
        return operatorIsClientsChief.getAsBoolean();
    }

    /**
     * Получить набор типов кампаний, которые оператор может создать у клиента
     */
    @Override
    public @GraphQLNonNull
    Set<@GraphQLNonNull GdCampaignType> getCanCreateCampaignTypes() {
        if (!getCanCreateCampaign()) {
            return Collections.emptySet();
        }

        return StreamEx.of(CampaignTypes.values())
                .filter(type -> type.canCreateThisType(this))
                .map(CampaignTypes::getUnderlyingType)
                .toSet();
    }

    @Override
    public @GraphQLNonNull
    Boolean getCanUseVideoConstructor() {
        return operatorCanEditClient.getAsBoolean() && calcClientAccessContainer.getGdClientFeatures().getIsVideoConstructorEnabled();
    }

    @Override
    public @GraphQLNonNull
    Boolean getCanUseVideoConstructorCreateFromScratch() {
        return operatorCanEditClient.getAsBoolean() &&
                calcClientAccessContainer.getGdClientFeatures().getIsVideoConstructorCreateFromScratchEnabled();
    }

    @Override
    public @GraphQLNonNull
    Boolean getCanUseVideoConstructorFeed() {
        return operatorCanEditClient.getAsBoolean() &&
                calcClientAccessContainer.getGdClientFeatures().getIsVideoConstructorFeedEnabled();
    }

    @Override
    public @GraphQLNonNull
    Boolean getCanUseVideoConstructorBeruruTemplate() {
        return operatorCanEditClient.getAsBoolean() &&
                calcClientAccessContainer.getGdClientFeatures().getIsVideoConstructorBeruruTemplateEnabled();
    }

    @Override
    public @GraphQLNonNull
    Boolean getCanCreateCampaign() {
        // для внутренней рекламы операторы имеют права создавать только для продукта (клиента внутренней рекламы)
        if (operatorHasInternalAdRole.getAsBoolean() && !clientIsInternalAdProduct.getAsBoolean()) {
            return false;
        }

        return operatorCanEditClient.getAsBoolean() || operatorIsManagerOfClientsCampaigns.getAsBoolean() ||
                // вещальщикам должна быть доступна массовая остановка кампаний
                hasOneOfRoles(operator, PLACER);
    }

    @Override
    public @GraphQLNonNull
    Boolean getCanCreatePromoExtension() {
        return operatorCanWrite && hasMaximalNumberOfPromoExtensions.getAsBoolean();
    }

    @Override
    public @GraphQLNonNull
    Boolean getCanCopyCampaign() {
        return hasOneOfRoles(operator, SUPER);
    }

    @Override
    public @GraphQLNonNull
    Boolean getCanMassCopyCampaign() {
        return !operator.getIsReadonlyRep() && hasOneOfRoles(operator, SUPER, MANAGER, SUPPORT, PLACER, AGENCY);
    }

    @Override
    public @GraphQLNonNull
    Boolean getCanManageFavoriteCampaigns() {
        return !operator.getIsReadonlyRep() && hasOneOfRoles(operator, CLIENT, SUPER, AGENCY, MANAGER);
    }

    @Override
    public @GraphQLNonNull
    Boolean getCanTransferMoney() {
        return socialPaymentCondition.getAsBoolean() && !operator.getIsReadonlyRep() && OperatorClientRelationsHelper
                .calculateCanTransferMoney(operator, operatorDisallowMoneyTransfer.getAsBoolean(),
                        calcClientAccessContainer.getClientNonWalletCampaigns(), getGdCampaignAccessById());
    }

    @Override
    public @GraphQLNonNull Boolean getCanPayCampaigns() {
        return socialPaymentCondition.getAsBoolean() && !operator.getIsReadonlyRep()
                && StreamEx.ofValues(getGdCampaignAccessById())
                .anyMatch(gdCampaignAccess -> gdCampaignAccess.getActions().contains(GdCampaignAction.PAY));
    }

    @Override
    public @GraphQLNonNull Boolean getHasPayableCampaigns() {
        return socialPaymentCondition.getAsBoolean() && !operator.getIsReadonlyRep()
                && StreamEx.of(calcClientAccessContainer.getClientNonWalletCampaigns())
                .anyMatch(c -> CAMPAIGN_ACCESS_HELPER.isCampaignPayable(calcClientAccessContainer.getSubjectUser(), c));
    }

    @Override
    public @GraphQLNonNull
    Boolean getCanPayByCash() {
        YandexDomain yandexDomain = HostUtils.getYandexDomain().orElse(null);
        boolean hasCampaignsSupportCashPay = getGdCampaignAccessById().values().stream()
                .anyMatch(access -> access.getActions().contains(GdCampaignAction.PAY_BY_CASH));

        return socialPaymentCondition.getAsBoolean() && !operator.getIsReadonlyRep() &&
                OperatorClientRelationsHelper.calculateCanPayByCash(yandexDomain,
                        hasCampaignsSupportCashPay, operator.getRole(), operatorIsClient.getAsBoolean(),
                        calcClientAccessContainer.getGdClientInfo().getWorkCurrency(), clientIsUnderAgency.getAsBoolean());
    }

    @Override
    public @GraphQLNonNull Boolean getCanPayByYandexMoney() {
        YandexDomain yandexDomain = HostUtils.getYandexDomain().orElse(null);
        return socialPaymentCondition.getAsBoolean() && !operator.getIsReadonlyRep()
                && OperatorClientRelationsHelper.calculateCanPayByYaMoney(yandexDomain,
                operatorIsClient.getAsBoolean(),
                calcClientAccessContainer.getClientNonWalletCampaigns(),
                calcClientAccessContainer.getGdClientInfo().getCountryRegionId(),
                calcClientAccessContainer.getGdClientInfo().getWorkCurrency(),
                clientIsUnderAgency.getAsBoolean()
        );
    }

    @Override
    public @GraphQLNonNull
    Boolean getCanViewOfflineReports() {
        if (operatorHasInternalAdRole.getAsBoolean() || clientIsInternalAdProduct.getAsBoolean()) {
            return false;
        }

        return hasOneOfRoles(operator, SUPER, SUPERREADER)
                || clientIsUnderAgency.getAsBoolean()
                || clientIsServiced.getAsBoolean()
                || calcClientAccessContainer.getGdClientFeatures().getHasCampaignsWithStats()
                || operatorIsClientsManager.getAsBoolean();
    }

    @Override
    public @GraphQLNonNull
    Boolean getCanManageVCards() {
        return calcClientAccessContainer.getGdClientFeatures().getCampaignIdForVCardsManagement() != null &&
                !(isSuperReader(operator) || isMediaPlanner(operator)) && !operator.getIsReadonlyRep();
    }

    @Override
    public @GraphQLNonNull
    Boolean getCanManageRetargertingConditions() {
        if (isAgencySubclientAccessedByMcc.getAsBoolean()) {
            return false;
        }
        if (operator.getIsReadonlyRep()) {
            return false;
        }
        return !hasOneOfRoles(operator, SUPERREADER, MEDIA,
                INTERNAL_AD_ADMIN, INTERNAL_AD_MANAGER, INTERNAL_AD_SUPERREADER);
    }

    /**
     * @deprecated Сейчас работа с турболендингами доступна всем (DIRECT-91489)
     */
    @Override
    @Deprecated
    public @GraphQLNonNull
    Boolean getCanManageTurboLandings() {
        return !operatorHasInternalAdRole.getAsBoolean() && !clientIsInternalAdProduct.getAsBoolean();
    }

    @Override
    public @GraphQLNonNull
    Boolean getCanManageCreatives() {
        if (isCreativeFreeInterfaceEnabled.getAsBoolean()) {
            return false;
        }

        if (operatorHasInternalAdRole.getAsBoolean() || clientIsInternalAdProduct.getAsBoolean()) {
            return false;
        }

        if (operator.getIsReadonlyRep()) {
            return false;
        }

        if (isAgencySubclientAccessedByMcc.getAsBoolean()) {
            return false;
        }

        return true;
    }

    @Override
    public @GraphQLNonNull
    Boolean getCanManageFeeds() {
        if (operatorHasInternalAdRole.getAsBoolean() || clientIsInternalAdProduct.getAsBoolean()) {
            return false;
        }

        if (operator.getIsReadonlyRep()) {
            return false;
        }

        if (isAgencySubclientAccessedByMcc.getAsBoolean()) {
            return false;
        }

        return true;
    }

    @Override
    public @GraphQLNonNull
    Boolean getCanManageUsingXLS() {
        if (operator.getIsReadonlyRep()) {
            return false;
        }
        return hasOneOfRoles(operator, SUPER, SUPERREADER, MANAGER, PLACER, AGENCY)
                || !clientIsUnderAgency.getAsBoolean() || operatorCanUseXLS.getAsBoolean();
    }

    @Override
    public @GraphQLNonNull
    Boolean getCanManageUsingApi() {
        return isApiEnabledFeature && hasOneOfRoles(operator, SUPER, SUPPORT, CLIENT, LIMITED_SUPPORT)
                && !operator.getIsReadonlyRep();
    }

    @Override
    public @GraphQLNonNull
    Boolean getCanManageUsingCommander() {
        boolean hasEditableCampaigns = calcClientAccessContainer.getClientNonWalletCampaigns().stream()
                .anyMatch(c -> c.getActions().getCanEdit());

        return hasEditableCampaigns
                && !hasOneOfRoles(operator, SUPERREADER, MEDIA);
    }

    @Override
    public @GraphQLNonNull
    Boolean getCanCheckCampaignsStats() {
        if (operatorHasInternalAdRole.getAsBoolean() || clientIsInternalAdProduct.getAsBoolean()) {
            return false;
        }

        if (operatorIsMccControlClient.getAsBoolean()) {
            return true;
        }

        return calcClientAccessContainer.getClientNonWalletCampaigns().stream()
                .anyMatch(c -> c.getActions().getActions().contains(GdiCampaignAction.SHOW_CAMP_STAT))
                && calcClientAccessContainer.getGdClientFeatures().getHasCampaignsWithStats();
    }

    @Override
    public @GraphQLNonNull
    Boolean getCanMonitorPhrasesPositions() {
        if (operatorHasInternalAdRole.getAsBoolean() || clientIsInternalAdProduct.getAsBoolean()) {
            return false;
        }

        return calcClientAccessContainer.getGdClientFeatures().getHasCampaignsWithShows() && !isMediaPlanner(operator);
    }

    @Override
    public @GraphQLNonNull
    Boolean getCanUploadDocuments() {
        if (operator.getIsReadonlyRep()) {
            return false;
        }
        boolean hasCampaignsForUpload = calcClientAccessContainer.getClientNonWalletCampaigns().stream()
                .anyMatch(c -> c.getActions().getCanEdit() && !c.getArchived()
                        && c.getStatusModerate() != CampaignStatusModerate.NEW);

        return !operatorIsClient.getAsBoolean() && hasCampaignsForUpload;
    }

    @Override
    public @GraphQLNonNull
    Boolean getCanSeeClientRepresentatives() {
        return operatorIsClientsChief.getAsBoolean()
                || hasOneOfRoles(operator, SUPER, SUPERREADER, MANAGER, SUPPORT, LIMITED_SUPPORT);
    }

    @Override
    public @GraphQLNonNull
    Boolean getCanViewUserLogs() {
        return hasOneOfRoles(operator, SUPER, SUPERREADER, MANAGER, SUPPORT, PLACER, MEDIA, CLIENT, AGENCY,
                LIMITED_SUPPORT);
    }

    @Override
    public @GraphQLNonNull
    Boolean getCanRequestFlService() {
        return operatorIsClientsChief.getAsBoolean() && !clientIsUnderAgency.getAsBoolean() &&
                !operatorIsFreelancer.getAsBoolean();
    }

    @Override
    public @GraphQLNonNull
    Boolean getCanViewMobileApps() {
        return !operatorHasInternalAdRole.getAsBoolean() && !clientIsInternalAdProduct.getAsBoolean();
    }

    @Override
    public @GraphQLNonNull
    Boolean getCanViewMobileAppDomain() {
        return !operatorHasInternalAdRole.getAsBoolean() && !clientIsInternalAdProduct.getAsBoolean()
                && getCanViewMobileApps() && operator.getRole().isInternal();
    }

    @Override
    public @GraphQLNonNull
    Boolean getCanEditMobileApps() {
        if (operatorHasInternalAdRole.getAsBoolean() || clientIsInternalAdProduct.getAsBoolean()) {
            return false;
        }

        if (operator.getIsReadonlyRep()) {
            return false;
        }

        return (isClient(operator) && (!clientIsUnderAgency.getAsBoolean() || superSubclient.getAsBoolean()))
                || (!isClient(operator) && !hasOneOfRoles(operator, SUPERREADER, PLACER, MEDIA));
    }

    @Override
    public @GraphQLNonNull
    Boolean getCanEditMobileAppDomain() {
        if (operator.getIsReadonlyRep()) {
            return false;
        }

        return !operatorHasInternalAdRole.getAsBoolean() && !clientIsInternalAdProduct.getAsBoolean() &&
                getCanEditMobileApps() && operator.getRole().isInternal();
    }

    @Override
    public @GraphQLNonNull Boolean getShowIosPopup() {
        return StreamEx.of(calcClientAccessContainer.getClientNonWalletCampaigns())
                .anyMatch(CampaignAccessHelper::isActiveIosMobileCampaign);
    }

    @Override
    public @GraphQLNonNull
    Boolean getCanEditInternalAds() {
        if (!getCanViewInternalAds()) {
            return false;
        }

        if (hasOneOfRoles(operator, SUPER, INTERNAL_AD_ADMIN)) {
            return true;
        }

        if (isInternalAdManager(operator)) {
            return internalAdProductRelation != null &&
                    internalAdProductRelation.getRelationType() == ClientsRelationType.INTERNAL_AD_PUBLISHER;
        }

        // TODO поменять на return false при отрывании клиентов с фичей; при этом упадёт
        // ru.yandex.direct.grid.processing.service.operator.OperatorAccessServiceCreateCampTypeTest
        return operatorIsClient.getAsBoolean();
    }

    @Override
    public @GraphQLNonNull
    Boolean getCanViewInternalAds() {
        return calcClientAccessContainer.getGdClientFeatures().getIsInternalAdsAllowed();
    }

    @Override
    public @GraphQLNonNull
    Boolean getCanExportToExcel() {
        // экпорт пока доступен только для внутренней рекламы
        return getCanViewInternalAds();
    }

    @Override
    public @GraphQLNonNull
    Boolean getCanManageInternalAdAccessRights() {
        return hasOneOfRoles(operator, SUPER, INTERNAL_AD_ADMIN);
    }

    @Override
    public @GraphQLNonNull
    Boolean getCanOnlyManageInternalAds() {
        return hasOneOfRoles(operator, INTERNAL_AD_ADMIN, INTERNAL_AD_MANAGER,
                INTERNAL_AD_SUPERREADER);
    }

    @Override
    public @GraphQLNonNull
    Boolean getCanSeeDealsLinkEverywhere() {
        return UserUtil.isAgency(operator);
    }

    @Override
    public @GraphQLNonNull
    Boolean getCanSeeAgencyClients() {
        return clientIsUnderAgency.getAsBoolean() &&
                (hasOneOfRoles(operator, SUPER, SUPERREADER, SUPPORT, MANAGER, AGENCY) || isAnyTeamLeader(operator));
    }

    @Override
    @GraphQLNonNull
    public Boolean getCanSeeManagerClients() {
        return clientIsServiced.getAsBoolean() && hasOneOfRoles(operator, SUPER, PLACER);
    }

    @Override
    @GraphQLNonNull
    public Boolean getCanSeeCampaignsServicedState() {
        return (clientIsServiced.getAsBoolean() && !clientIsUnderAgency.getAsBoolean()) || hasOneOfRoles(operator,
                SUPER, MANAGER);
    }

    @Override
    public @GraphQLNonNull Boolean getCanSeeRepresentativeClientsLink() {
        return canSeeRepresentativeClientsLink.getAsBoolean();
    }

    @Override
    public @GraphQLNonNull
    Boolean getCanViewLibMinusWords() {
        return !operatorHasInternalAdRole.getAsBoolean() && !clientIsInternalAdProduct.getAsBoolean() && operatorCanRead;
    }

    @Override
    public @GraphQLNonNull
    Boolean getCanManageLibMinusWords() {
        return getCanViewLibMinusWords() && operatorCanEditClient.getAsBoolean();
    }

    @Override
    public @GraphQLNonNull Boolean getCanMassEditAdGroupRegions() {
        return operatorCanEditClient.getAsBoolean();
    }

    @Override
    public @GraphQLNonNull Boolean getCanUseRoiStrategy() {
        return canUseRoiStrategy.getAsBoolean();
    }

    @Override
    public @GraphQLNonNull
    Boolean getOperatorCanRead() {
        return operatorCanRead;
    }

    @Override
    public @GraphQLNonNull Boolean getOperatorCanWrite() {
        return operatorCanWrite;
    }

    @Override
    public @GraphQLNonNull
    Boolean getOperatorCanViewBulkOperationsPopups() {
        return operatorCanWrite || hasOneOfRoles(operator, SUPERREADER, SUPPORT, LIMITED_SUPPORT);
    }

    @Override
    public @GraphQLNonNull Boolean getOperatorCanEditClientSettings() {
        return !operatorHasInternalAdRole.getAsBoolean() &&
                !clientIsInternalAdProduct.getAsBoolean() &&
                !hasOneOfRoles(operator, MEDIA) &&
                !operatorIsClient.getAsBoolean() &&
                !operatorIsMccForClient.getAsBoolean() &&
                (operatorCanWrite ||
                        clientIsUnderAgency.getAsBoolean() ||
                        hasOneOfRoles(operator, SUPERREADER, LIMITED_SUPPORT, PLACER)
                );
    }

    @Override
    public @GraphQLNonNull Boolean getInfoblockEnabled() {
        return hasOneOfRoles(operator, CLIENT, EMPTY, AGENCY, SUPER, SUPERREADER);
    }

    @Override
    public @GraphQLNonNull Boolean getIsFirstCampaignHelpAllowed() {
        return !clientIsUnderAgency.getAsBoolean() && !operatorIsFreelancer.getAsBoolean() &&
                !calcClientAccessContainer.getGdClientInfo().getIsUnderFreelancer() &&
                RegionConstants.FIRST_CAMPAIGN_HELP_COUNTRIES.contains(
                        calcClientAccessContainer.getGdClientInfo().getCountryRegionId());
    }

    @Override
    public @GraphQLNonNull Boolean getOperatorCanEditUserSettings() {
        return !operatorHasInternalAdRole.getAsBoolean() &&
                !clientIsInternalAdProduct.getAsBoolean() &&
                !hasOneOfRoles(operator, MEDIA) &&
                !operatorIsMccForClient.getAsBoolean() &&
                (operatorCanWrite ||
                        clientIsUnderAgency.getAsBoolean() ||
                        hasOneOfRoles(operator, SUPERREADER, LIMITED_SUPPORT)
                );
    }

    @Override
    public @GraphQLNonNull Boolean getCanEditUserSettings() {
        return OperatorClientRelationsHelper.calculateCanEditUserSettings(operator,
                calcClientAccessContainer.getGdClientInfo().getId(),
                operatorIsFreelancer.getAsBoolean() || operatorIsMccForClient.getAsBoolean());
    }

    @Override
    public @GraphQLNonNull Boolean getAutopaymentDisabled() {
        return operator.getRole() == RbacRole.AGENCY &&
                operator.getRepType() == RbacRepType.LIMITED && userAgencyIsNoPay.getAsBoolean();
    }

    @Override
    public @GraphQLNonNull Boolean getWalletIsWalletIdLinkToBalanceDisabled() {
        return walletIdLinkToBalanceDisabled.getAsBoolean();
    }

    @Override
    public @GraphQLNonNull Boolean getWalletAllowPay() {
        return socialPaymentCondition.getAsBoolean() && walletAllowPay.getAsBoolean();
    }

    @Override
    public @GraphQLNonNull Boolean getWalletCanEnable() {
        return socialPaymentCondition.getAsBoolean() && walletCanEnable.getAsBoolean();
    }

    @Override
    public @GraphQLNonNull Boolean getWalletCanUseOverdraftRest() {
        return socialPaymentCondition.getAsBoolean() && walletCanUseOverdraftRest.getAsBoolean();
    }

    @Override
    public @GraphQLNonNull Boolean getWalletCanUseAutopay() {
        return socialPaymentCondition.getAsBoolean() && walletCanUseAutopay.getAsBoolean();
    }

    @Override
    public @GraphQLNonNull Boolean getShowBalanceLink() {
        return socialPaymentCondition.getAsBoolean() && showBalanceLink.getAsBoolean();
    }

    @Override
    public @GraphQLNonNull Boolean getCanResetFlightStatusApprove() {
        return calcClientAccessContainer.getGdClientFeatures().getIsCpmPriceCampaignEnabled()
                && CampaignWithPricePackagePermissionUtils.canResetFlightStatusApprove(operator);
    }

    @Override
    public @GraphQLNonNull Boolean getCanIncreaseDailyBudget() {
        return isSocialAdvertisement.getAsBoolean() && getCanEditWalletDayBudget();
    }

    @Override
    public @GraphQLNonNull Boolean getCanEditWalletDayBudget() {
        return (getCanTransferMoney() || walletAllowedEdit.getAsBoolean()) && !walletReadOnly.getAsBoolean();
    }

    @Override
    public @GraphQLNonNull Boolean getCanUseCampDescription() {
        return operatorIsClientsAgency.getAsBoolean() || calcClientAccessContainer.getSubjectUser().getUseCampDescription();
    }

    @Override
    public @GraphQLNonNull Boolean getCanShowAccountScore() {
        return !clientIsTurkish.getAsBoolean();
    }

    @Override
    public @GraphQLNonNull Boolean getCanCreateMetrikaGoalsByCounter() {
        return operatorCanWrite;
    }

    @Override
    public @GraphQLNonNull Boolean getCanViewDocumentsAndPayments() {
        return (operatorIsClient.getAsBoolean() || hasOneOfRoles(operator, SUPPORT, LIMITED_SUPPORT))
                && !clientIsUnderAgency.getAsBoolean()
                && !calcClientAccessContainer.getClientNonWalletCampaigns().isEmpty()
                && socialPaymentCondition.getAsBoolean();
    }

    @Override
    public @GraphQLNonNull Boolean getCanBlockUser() {
        return hasOneOfRoles(calcClientAccessContainer.getSubjectUser(), CLIENT, AGENCY) && canBlockUser(operator);
    }

    @Override
    public @GraphQLNonNull Boolean getCanBindClientToAgency() {
        return canBindClientToAgency.getAsBoolean() && agencyClientUnarchived.getAsBoolean();
    }

    @Override
    public @GraphQLNonNull Boolean getIsAgencyServiceStopped() {
        return isAgencyServiceStopped.getAsBoolean();
    }

    @Override
    public @GraphQLNonNull Boolean getOperatorCanSetAgencyRepresentative() {
        boolean operatorIsClientsAgencyChief =
                operatorIsClientsAgency.getAsBoolean() && (isAgency(operator) && operator.getChiefUid().equals(
                        operator.getUid()));
        return clientIsUnderAgency.getAsBoolean() && canAgencyRepresentativeBeSet.getAsBoolean() &&
                (operatorIsClientsAgencyChief || isSuper(operator));
    }

    @Override
    public @GraphQLNonNull Boolean getOperatorCanUseSupportChat() {
        if (operatorHasDebugFeature.getAsBoolean()) {
            logger.info(
                    "ClientId: {}, operator: {}, hasFeature: {}, isClient: {}, " +
                            "isUnderAgency: {}, regionId: {}, verdict: {}",
                    calcClientAccessContainer.getGdClientInfo().getId(),
                    operator.getLogin(),
                    operatorHasCheckedSupportChatFeature.getAsBoolean(),
                    operatorIsClient.getAsBoolean(),
                    clientIsUnderAgency.getAsBoolean(),
                    calcClientAccessContainer.getGdClientInfo().getCountryRegionId(),
                    operatorHasCheckedSupportChatFeature.getAsBoolean()
                            && operatorIsClient.getAsBoolean() && !clientIsUnderAgency.getAsBoolean()
                            && SUPPORT_CHAT_REGION_IDS.contains(
                            calcClientAccessContainer.getGdClientInfo().getCountryRegionId()));
        }
        if (!operatorHasCheckedSupportChatFeature.getAsBoolean()) {
            return false;
        }
        if (operatorIsFreelancer.getAsBoolean()) {
            return true;
        }

        // считаем, что самоходный или менеджерский клиент (+не фрилансер) может смотреть только себя
        return operatorIsClient.getAsBoolean() && !clientIsUnderAgency.getAsBoolean()
                && SUPPORT_CHAT_REGION_IDS.contains(calcClientAccessContainer.getGdClientInfo().getCountryRegionId());
    }

    @Override
    public @GraphQLNonNull Boolean getShowManagerControls() {
        return showManagerControls.getAsBoolean();
    }

    @Override
    public @GraphQLNonNull Boolean getShowAgencyControls() {
        return showAgencyControls.getAsBoolean();
    }

    @Override
    public @GraphQLNonNull Boolean getCanSelfSendAdsOnRemoderation() {
        return ModerateBannerValidationService.remoderationAllowed(operator.getRole(),
                calcClientAccessContainer.getSubjectUser().getRole(),
                calcClientAccessContainer.getGdClientFeatures().getIsClientAllowedToRemoderate()) && operatorCanWrite;
    }

    @Override
    public @GraphQLNonNull Boolean getCanViewConversionSources() {
        return !clientIsInternalAdProduct.getAsBoolean()
                && calcClientAccessContainer.getGdClientFeatures().getIsConversionCenterEnabled() ;
    }

    @Override
    public @GraphQLNonNull Boolean getCanEditConversionSources() {
        return getCanViewConversionSources() && operatorCanWrite;
    }

    @Override
    @GraphQLNonNull
    public Boolean getCanUseAutobudgetWeekBundle() {
        return canUseAutobudgetWeekBundle.getAsBoolean();
    }

    @Override
    @GraphQLNonNull
    public Boolean getCanCreateFeedFromSite() {
        return !clientIsTurkish.getAsBoolean() || hasOneOfRoles(operator, SUPER, MANAGER);
    }

}
