package ru.yandex.direct.grid.processing.service.operator;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.Collection;
import java.util.Map;

import javax.annotation.Nullable;

import one.util.streamex.StreamEx;

import ru.yandex.direct.common.enums.YandexDomain;
import ru.yandex.direct.common.util.HostUtils;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.grid.model.campaign.GdCampaignAccess;
import ru.yandex.direct.grid.model.campaign.GdCampaignAction;
import ru.yandex.direct.grid.model.campaign.GdiBaseCampaign;
import ru.yandex.direct.grid.model.campaign.GdiCampaignStrategyName;
import ru.yandex.direct.grid.processing.model.client.GdClientInfo;
import ru.yandex.direct.rbac.RbacRepType;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.rbac.model.ClientsRelation;
import ru.yandex.direct.rbac.model.ClientsRelationType;
import ru.yandex.direct.regions.Region;

import static ru.yandex.direct.core.entity.user.utils.UserUtil.hasOneOfRoles;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isAgency;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isClient;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isInternalAdManager;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isManager;
import static ru.yandex.direct.rbac.RbacRole.AGENCY;
import static ru.yandex.direct.rbac.RbacRole.CLIENT;
import static ru.yandex.direct.rbac.RbacRole.INTERNAL_AD_ADMIN;
import static ru.yandex.direct.rbac.RbacRole.LIMITED_SUPPORT;
import static ru.yandex.direct.rbac.RbacRole.MANAGER;
import static ru.yandex.direct.rbac.RbacRole.SUPER;
import static ru.yandex.direct.rbac.RbacRole.SUPPORT;
import static ru.yandex.direct.regions.Region.TURKEY_REGION_ID;
import static ru.yandex.direct.utils.CommonUtils.isValidId;

//todo ssdmitriev: сделать компонентой
public class OperatorClientRelationsHelper {
    static final int DAYS_TO_COUNT_SHOWS_FOR_PAY_YA_MONEY_CHECK = 270;

    /**
     * Вычисляется возможность оператора переносить деньги. Нужно для показа соответствующей ссылки в подвале интерфейса
     * Логика вычисления была перенесена из перла. Точка входа p-campaigns__aux-actions.bemtree.js:
     * {@code canTransferMoney = auxParams.transfer_money && auxParams.camp_count_for_transfer > 1 && auxParams
     * .camp_count_with_sum > 0}
     */
    static boolean calculateCanTransferMoney(User operator, boolean operatorDisallowMoneyTransfer,
                                             Collection<GdiBaseCampaign> campaigns,
                                             Map<Long, GdCampaignAccess> gdCampaignAccessById) {
        boolean allowPay = hasOneOfRoles(operator, CLIENT, AGENCY, SUPER, MANAGER, SUPPORT, LIMITED_SUPPORT);
        boolean agencyDisallowMoneyTransfer = (isAgency(operator)
                && operator.getRepType() == RbacRepType.LIMITED) && operatorDisallowMoneyTransfer;

        boolean allowTransferMoneyFromToCamp = false;
        boolean canTransferMoney = false;
        int campaignsWithTransferableSum = 0;
        int campaignsForTransferNum = 0;

        for (GdiBaseCampaign campaign : campaigns) {
            GdCampaignAccess campaignAccess = gdCampaignAccessById.get(campaign.getId());

            boolean allowTransferMoneySubclient =
                    campaignAccess.getPseudoActions().contains(GdCampaignAction.ALLOW_TRANSFER_MONEY_SUBCLIENT);
            boolean hasAgency = campaign.getActions().getHasAgency();
            allowTransferMoneyFromToCamp = allowTransferMoneyFromToCamp || allowPay && !agencyDisallowMoneyTransfer
                    && (!hasAgency || allowTransferMoneySubclient || !isClient(operator));

            boolean hasActionPay = campaignAccess.getActions().contains(GdCampaignAction.PAY);

            //Оригинальное условие: can.transfer_money || can.process_money_common && !camp.noaction && (actionPay ||
            // allow_transfer_money_from_to_camp) && (can.transfer_money = true);
            //process_money_common для грида не актуален, поэтому его нет в итоговом условии
            canTransferMoney = canTransferMoney || !campaignAccess.getNoActions()
                    && (hasActionPay || allowTransferMoneyFromToCamp);

            BigDecimal sumTotal = campaign.getSum().subtract(campaign.getSumSpent());
            boolean isTotalSumGreaterZero = sumTotal.compareTo(BigDecimal.ZERO) > 0;

            boolean agencyServiced = campaignAccess.getPseudoActions().contains(GdCampaignAction.AGENCY_SERVICED);
            boolean canTransfer = allowTransferMoneyFromToCamp
                    && (isTotalSumGreaterZero || hasActionPay || allowTransferMoneySubclient)
                    && !(isManager(operator) && agencyServiced);
            if (canTransfer) {
                campaignsForTransferNum++;
            }

            boolean moneyNotBlocked = !campaign.getMoneyBlocked() && !isValidId(campaign.getWalletId());
            if (isTotalSumGreaterZero && moneyNotBlocked && !(hasAgency && isManager(operator))) {
                campaignsWithTransferableSum++;
            }
        }

        return canTransferMoney && campaignsForTransferNum > 1 && campaignsWithTransferableSum > 0;
    }

    static boolean calculateCanPayByCash(@Nullable YandexDomain yandexDomain, boolean hasCampaignsSupportCashPay,
                                         RbacRole operatorRole, boolean operatorIsClient,
                                         CurrencyCode clientWorkCurrency, boolean clientIsUnderAgency) {
        // limited_support должен видеть интерфейс так, как клиент
        boolean operatorCanPayExactlyAsClient = operatorRole.anyOf(LIMITED_SUPPORT) && !clientIsUnderAgency;
        boolean cashPayIsAvailableForOperator = (operatorIsClient && !clientIsUnderAgency)
                || operatorCanPayExactlyAsClient
                || operatorRole.anyOf(SUPER, MANAGER, SUPPORT);
        boolean currencyIsSupported =
                clientWorkCurrency == CurrencyCode.RUB || clientWorkCurrency == CurrencyCode.YND_FIXED;

        return yandexDomain == YandexDomain.RU && hasCampaignsSupportCashPay
                && cashPayIsAvailableForOperator
                && currencyIsSupported;
    }

    static boolean calculateCanPayByYaMoney(@Nullable YandexDomain yandexDomain, boolean operatorIsClient,
                                            Collection<GdiBaseCampaign> campaigns,
                                            @Nullable Long countryRegionId, CurrencyCode clientWorkCurrency,
                                            boolean clientIsUnderAgency) {
        boolean yaMoneyPayIsAvailableForOperator = operatorIsClient && !clientIsUnderAgency;

        boolean currencyIsSupported =
                clientWorkCurrency == CurrencyCode.RUB || clientWorkCurrency == CurrencyCode.YND_FIXED;

        boolean domainSupported = yandexDomain == YandexDomain.RU;

        LocalDateTime dateForShowsCheck = LocalDateTime.now().minusDays(DAYS_TO_COUNT_SHOWS_FOR_PAY_YA_MONEY_CHECK);
        boolean hasShows = StreamEx.of(campaigns)
                .anyMatch(campaign -> campaign.getLastShowTime() != null && campaign.getLastShowTime()
                        .isAfter(dateForShowsCheck));

        boolean countrySupported = countryRegionId == null || countryRegionId == Region.RUSSIA_REGION_ID;

        return domainSupported
                && yaMoneyPayIsAvailableForOperator
                && currencyIsSupported
                && hasShows
                && countrySupported;
    }

    public static boolean calculateCanEditUserSettings(User operator, Long clientId, boolean operatorIsRelatedClient) {
        return hasOneOfRoles(operator, SUPER, MANAGER, AGENCY, CLIENT) &&
                !operator.getIsReadonlyRep() &&
                // связанным клиентам можно, только если они редактируют свои настройки
                (!operatorIsRelatedClient || operator.getClientId().asLong() == clientId);
    }

    public static boolean operatorCanManageInternalAdCampaignsOfClient(User operator,
                                                                       boolean clientIsInternalAdProduct,
                                                                       ClientsRelation internalAdProductRelation) {
        if (!clientIsInternalAdProduct) {
            return false;
        }

        if (hasOneOfRoles(operator, SUPER, INTERNAL_AD_ADMIN)) {
            return true;
        }

        if (isInternalAdManager(operator)) {
            return internalAdProductRelation != null &&
                    internalAdProductRelation.getRelationType() == ClientsRelationType.INTERNAL_AD_PUBLISHER;
        }

        return false;
    }

    public static boolean clientIsTurkish(GdClientInfo gdClientInfo) {
        if (gdClientInfo.getCountryRegionId().equals(TURKEY_REGION_ID)
                || gdClientInfo.getWorkCurrency() == CurrencyCode.TRY) {
            return true;
        }

        YandexDomain yandexDomain = HostUtils.getYandexDomain().orElse(null);
        return yandexDomain == YandexDomain.TR;
    }

    public static boolean clientCurrencyIsGBP(GdClientInfo gdClientInfo) {
        return gdClientInfo.getWorkCurrency() == CurrencyCode.GBP;
    }

    public static boolean hasRoiStrategyInNotArchivedCampaigns(Collection<GdiBaseCampaign> gdiCampaigns) {
        return StreamEx.of(gdiCampaigns)
                .remove(GdiBaseCampaign::getArchived)
                .map(GdiBaseCampaign::getStrategyName)
                .anyMatch(GdiCampaignStrategyName.AUTOBUDGET_ROI::equals);
    }
}
