package ru.yandex.direct.grid.processing.service.operator;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.abac.Attribute;
import ru.yandex.direct.common.TranslationService;
import ru.yandex.direct.common.enums.YandexDomain;
import ru.yandex.direct.common.util.HostUtils;
import ru.yandex.direct.common.util.HttpUtil;
import ru.yandex.direct.common.util.TuneSecretKey;
import ru.yandex.direct.core.entity.client.Constants;
import ru.yandex.direct.core.entity.client.mcc.ClientMccService;
import ru.yandex.direct.core.entity.client.model.office.CityContact;
import ru.yandex.direct.core.entity.client.model.office.OfficeContact;
import ru.yandex.direct.core.entity.client.model.office.RegionContact;
import ru.yandex.direct.core.entity.client.service.ClientOfficeService;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.freelancer.model.Freelancer;
import ru.yandex.direct.core.entity.freelancer.service.FreelancerService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.geobasehelper.GeoBaseHelper;
import ru.yandex.direct.grid.core.frontdb.jsonsettings.service.JsonSettingsService;
import ru.yandex.direct.grid.processing.model.client.GdAvailableFields;
import ru.yandex.direct.grid.processing.model.client.GdCampaignsAvailableFields;
import ru.yandex.direct.grid.processing.model.client.GdClientMccCommonInfo;
import ru.yandex.direct.grid.processing.model.client.GdCoreFeatureWithDescription;
import ru.yandex.direct.grid.processing.model.client.GdOfficeContactData;
import ru.yandex.direct.grid.processing.model.client.GdOfficeInfo;
import ru.yandex.direct.grid.processing.model.client.GdOperatorAccess;
import ru.yandex.direct.grid.processing.model.client.GdOperatorInfo;
import ru.yandex.direct.grid.processing.model.client.GdUserInfo;
import ru.yandex.direct.grid.processing.service.attributes.AttributeResolverService;
import ru.yandex.direct.grid.processing.service.client.converter.GdFeatureWithDescriptionConverterService;
import ru.yandex.direct.i18n.Language;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.rbac.UserPerminfo;
import ru.yandex.direct.utils.CommonUtils;

import static java.util.Collections.singleton;
import static ru.yandex.direct.common.util.HttpUtil.getCurrentLocale;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.hasOneOfRoles;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isAnyTeamLeader;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isChiefRep;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isClient;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isSuperPlacer;
import static ru.yandex.direct.grid.processing.service.campaign.copy.CampaignCopyValidatorProvider.ROLES_ALLOWED_TO_COPY_CONVERSION_STRATEGY;
import static ru.yandex.direct.grid.processing.service.campaign.copy.CampaignCopyValidatorProvider.ROLES_ALLOWED_TO_COPY_WITH_REPORT;
import static ru.yandex.direct.grid.processing.service.campaign.copy.CampaignCopyValidatorProvider.ROLES_ALLOWED_TO_STOP_COPIED_CAMPAIGNS;
import static ru.yandex.direct.grid.processing.service.operator.OperatorConverter.toGdOperatorFeatures;
import static ru.yandex.direct.rbac.RbacRole.AGENCY;
import static ru.yandex.direct.rbac.RbacRole.CLIENT;
import static ru.yandex.direct.rbac.RbacRole.INTERNAL_AD_ADMIN;
import static ru.yandex.direct.rbac.RbacRole.INTERNAL_AD_MANAGER;
import static ru.yandex.direct.rbac.RbacRole.INTERNAL_AD_SUPERREADER;
import static ru.yandex.direct.rbac.RbacRole.LIMITED_SUPPORT;
import static ru.yandex.direct.rbac.RbacRole.MANAGER;
import static ru.yandex.direct.rbac.RbacRole.MEDIA;
import static ru.yandex.direct.rbac.RbacRole.PLACER;
import static ru.yandex.direct.rbac.RbacRole.SUPER;
import static ru.yandex.direct.rbac.RbacRole.SUPERREADER;
import static ru.yandex.direct.rbac.RbacRole.SUPPORT;
import static ru.yandex.direct.regions.Region.GLOBAL_REGION_ID;

/**
 * Сервис, возвращающий данные об операторе
 */
@Service
@ParametersAreNonnullByDefault
public class OperatorDataService {
    private static final Map<GdCampaignsAvailableFields, Attribute> CAMPAIGNS_AVAILABLE_FIELDS_ATTRIBUTE_MAP = Map.of(
            GdCampaignsAvailableFields.CONTENT_LANGUAGE, Attribute.CAN_EDIT_CAMPAIGN_CONTENT_LANGUAGE_BLOCK,
            GdCampaignsAvailableFields.ATTRIBUTION_MODEL, Attribute.CAN_EDIT_CAMPAIGN_ATTRIBUTION_MODEL_BLOCK,
            GdCampaignsAvailableFields.ALLOWED_PAGE_IDS, Attribute.OPERATOR_HAS_SET_CAMPAIGN_ALLOWED_PAGE_IDS_FEATURE,
            GdCampaignsAvailableFields.DISALLOWED_PAGE_IDS,
            Attribute.OPERATOR_HAS_SET_CAMPAIGN_DISALLOWED_PAGE_IDS_FEATURE
    );
    private static final Logger LOGGER = LoggerFactory.getLogger(OperatorDataService.class);

    private final ClientService clientService;
    private final ClientOfficeService clientOfficeService;
    private final UserService userService;
    private final RbacService rbacService;
    private final TranslationService translationService;
    private final GeoBaseHelper geoBaseHelper;
    private final FreelancerService freelancerService;
    private final FeatureService featureService;
    private final AttributeResolverService attributeResolverService;
    private final GdFeatureWithDescriptionConverterService gdFeatureWithDescriptionConverter;
    private final JsonSettingsService jsonSettingsService;
    private final ClientMccService clientMccService;

    @Autowired
    @SuppressWarnings("checkstyle:parameternumber")
    public OperatorDataService(ClientService clientService, ClientOfficeService clientOfficeService,
                               UserService userService, RbacService rbacService, TranslationService translationService,
                               GeoBaseHelper geoBaseHelper,
                               FreelancerService freelancerService,
                               FeatureService featureService, AttributeResolverService attributeResolverService,
                               GdFeatureWithDescriptionConverterService gdFeatureWithDescriptionConverter,
                               JsonSettingsService jsonSettingsService,
                               ClientMccService clientMccService) {
        this.clientService = clientService;
        this.clientOfficeService = clientOfficeService;
        this.userService = userService;
        this.rbacService = rbacService;
        this.translationService = translationService;
        this.geoBaseHelper = geoBaseHelper;
        this.freelancerService = freelancerService;
        this.featureService = featureService;
        this.attributeResolverService = attributeResolverService;
        this.gdFeatureWithDescriptionConverter = gdFeatureWithDescriptionConverter;
        this.jsonSettingsService = jsonSettingsService;
        this.clientMccService = clientMccService;
    }

    GdOperatorInfo getOperatorInfo(User operator) {
        GdUserInfo managerInfo = null;
        GdUserInfo agencyInfo = null;
        GdUserInfo chiefInfo = null;
        boolean isSelfAgency = false;

        GdUserInfo operatorInfo = UserDataConverter.toGdUserInfo(operator);
        Currency workCurrency = clientService.getWorkCurrency(operator.getClientId());
        UserPerminfo operatorPerminfo = rbacService.getUserPermInfo(operator.getUid());
        List<Freelancer> freelancers = freelancerService.getFreelancers(singleton(operator.getClientId().asLong()));
        boolean isFreelancer = !freelancers.isEmpty();
        boolean isAgency = RbacRole.AGENCY == operatorPerminfo.role();
        boolean isOlderThanWeek = operator.getCreateTime() != null &&
                operator.getCreateTime().isBefore(LocalDateTime.now().minusDays(7));

        var mccClientIds = operatorPerminfo.mccClientIds().stream().map(ClientId::fromLong).collect(Collectors.toSet());
        if (!mccClientIds.isEmpty()) {
            mccClientIds.add(operator.getClientId());
        }
        var mccClientsChiefUids = clientService.massGetChiefUidsByClientIds(mccClientIds).values();

        List<Long> userIds =
                StreamEx.of(operatorPerminfo.managerUid(), operatorPerminfo.chiefUid())
                        .append(operatorPerminfo.agencyUids())
                        .append(mccClientsChiefUids)
                        .filter(CommonUtils::isValidId)
                        .collect(Collectors.toList());
        Map<Long, GdUserInfo> usersInfoByUid = getUsersByIds(userIds);

        var mccClientsInfo = mccClientsChiefUids.stream()
                .map(usersInfoByUid::get)
                .sorted(Comparator.comparing(GdUserInfo::getName))
                .collect(Collectors.toList());

        var hasControlRequests = clientMccService.hasControlRequests(operator.getClientId());
        var gdClientMccCommonInfo = new GdClientMccCommonInfo()
                .withHasManagedClients(operatorPerminfo.isMcc())
                .withHasControlRequests(hasControlRequests)
                .withCanManageControlRequests(hasControlRequests && isChiefRep(operator))
                .withCanUseClientMccCommonInfo(hasOneOfRoles(operator, SUPER, SUPERREADER, SUPPORT, LIMITED_SUPPORT,
                        MANAGER));

        if (operatorPerminfo.managerUid() != null) {
            managerInfo = usersInfoByUid.get(operatorPerminfo.managerUid());

            if (RbacRole.AGENCY == operatorPerminfo.role()) {
                String managerEmail = userService.getUserEmail(operatorPerminfo.managerUid());
                isSelfAgency = Constants.MANAGER_EMAIL_OF_SELF_AGENCY.equals(managerEmail);
            }
        }

        if (!operatorPerminfo.agencyUids().isEmpty()) {
            // Теперь когда представителей может быть несколько, берем случайного.
            // Нужно будет добавить новое поле, куда будет отдавать всех представителей
            agencyInfo = usersInfoByUid.get(operatorPerminfo.agencyUids().iterator().next());
        }

        if (operatorPerminfo.chiefUid() != null) {
            chiefInfo = usersInfoByUid.get(operatorPerminfo.chiefUid());
        }

        String tuneSecretKey = TuneSecretKey.generateSecretKey(operator.getUid());
        Long countryRegionId = clientService.getCountryRegionIdByClientId(operator.getClientId()).orElse(null);
        OfficeContact officeContact = getOfficeContact(countryRegionId);

        Set<String> availableFeatures = featureService.getEnabledForClientId(operator.getClientId());

        Set<GdCoreFeatureWithDescription> coreFeaturesWithDescription =
                gdFeatureWithDescriptionConverter.convertToCore(availableFeatures);

        return new GdOperatorInfo()
                .withInfo(operatorInfo)
                .withFeatures(toGdOperatorFeatures(availableFeatures, operatorPerminfo.role(),
                        coreFeaturesWithDescription))
                .withOfficeInfo(toGdOfficeInfo(officeContact))
                .withTuneSecretKey(tuneSecretKey)
                .withIsAgency(isAgency)
                .withIsSelfAgency(isSelfAgency)
                .withIsFreelancer(isFreelancer)
                .withIsOlderThanWeek(isOlderThanWeek)
                .withManagerInfo(managerInfo)
                .withAgencyInfo(agencyInfo)
                .withChiefInfo(chiefInfo)
                .withWorkCurrency(workCurrency.getCode())
                .withClientMccInfo(mccClientsInfo)
                .withClientMccCommonInfo(gdClientMccCommonInfo)
                .withAccess(getOperatorAccess(operator));
    }

    GdOperatorAccess getOperatorAccess(User operator) {
        boolean hasEditInternationalisationRight = rbacService.canOperatorEditInternationalisation(operator.getUid());

        Set<String> operatorFeatures = featureService.getEnabledForClientId(operator.getClientId());
        boolean isSocialAdvertising = operatorFeatures.contains(FeatureName.SOCIAL_ADVERTISING.getName());
        boolean socialPaymentCondition =
                !isSocialAdvertising || operatorFeatures.contains(FeatureName.SOCIAL_ADVERTISING_PAYABLE.getName());

        boolean isRoRep = operator.getIsReadonlyRep();
        return new GdOperatorAccess()
                .withActions(OperatorAllowedActionsUtils.getActions(operator))
                .withCanSearchPhrasesDuplicates(operator.getRole().isInternal()
                        || hasOneOfRoles(operator, LIMITED_SUPPORT))
                .withCanEditAdDomain(operator.getRole().isInternal())
                .withCanEditInternationalization(hasEditInternationalisationRight && !isClient(operator))
                .withWalletDontShowWalletLink(!socialPaymentCondition || hasOneOfRoles(operator, PLACER, MEDIA))
                .withCanViewInternalAdsInSomeClients(hasOneOfRoles(operator,
                        SUPER, SUPERREADER,
                        INTERNAL_AD_ADMIN, INTERNAL_AD_MANAGER, INTERNAL_AD_SUPERREADER))
                .withCanViewListOfInternalAdManagers(hasOneOfRoles(operator,
                        SUPER, SUPERREADER, INTERNAL_AD_ADMIN))
                .withCanEditInternalAdManagers(hasOneOfRoles(operator, SUPER, INTERNAL_AD_ADMIN))
                .withCanViewListOfInternalAdProducts(hasOneOfRoles(operator,
                        SUPER, SUPERREADER,
                        INTERNAL_AD_ADMIN, INTERNAL_AD_MANAGER, INTERNAL_AD_SUPERREADER))
                .withCanEditInternalAdProducts(hasOneOfRoles(operator, SUPER, INTERNAL_AD_ADMIN))
                .withCanResumeAutopay(!isRoRep && hasOneOfRoles(operator, SUPER, CLIENT))
                .withCanEditAutopay(!isRoRep && !hasOneOfRoles(operator, PLACER, MEDIA, SUPERREADER))
                .withCanPay(!isRoRep && hasOneOfRoles(operator, CLIENT, AGENCY, SUPER, MANAGER, SUPPORT,
                        LIMITED_SUPPORT))
                .withCanViewCrmLink(operator.getRole().isInternal()
                        || hasOneOfRoles(operator, LIMITED_SUPPORT))
                .withCanViewModerationLink(hasOneOfRoles(operator,
                        SUPER, SUPERREADER, SUPPORT, PLACER, MEDIA, MANAGER, LIMITED_SUPPORT))
                .withCanViewPayCampaigns(hasOneOfRoles(operator,
                        SUPERREADER, MANAGER, SUPPORT, LIMITED_SUPPORT))
                .withCanViewMailLogs(hasOneOfRoles(operator,
                        SUPER, SUPERREADER, SUPPORT, LIMITED_SUPPORT, MANAGER))
                .withCanViewManagerLogs(isSuperPlacer(operator) || hasOneOfRoles(operator,
                        SUPER, SUPERREADER, SUPPORT, LIMITED_SUPPORT, MANAGER, MEDIA))
                .withCanViewCmdLogs(isAnyTeamLeader(operator) || hasOneOfRoles(operator,
                        SUPER, SUPERREADER, SUPPORT, LIMITED_SUPPORT, MANAGER))
                .withCanViewModerationDocuments(operator.getRole().isInternal()
                        || hasOneOfRoles(operator, LIMITED_SUPPORT))
                .withCanCreateCampaignBalance(hasOneOfRoles(operator,
                        SUPER, SUPERREADER, MANAGER, LIMITED_SUPPORT))
                .withCanViewCampaignBalance(hasOneOfRoles(operator,
                        SUPER, SUPERREADER, MANAGER, PLACER, SUPPORT, LIMITED_SUPPORT))
                .withCanViewLogviewerLink(hasOneOfRoles(operator,
                        SUPER, SUPERREADER, SUPPORT, LIMITED_SUPPORT))
                .withCanSetCampaignDisallowedPageIds(!isRoRep && (hasOneOfRoles(operator,
                        SUPER, SUPPORT, LIMITED_SUPPORT, MANAGER)
                        || operatorFeatures.contains(FeatureName.SET_CAMPAIGN_DISALLOWED_PAGE_IDS.getName())))
                .withCanOnlyCopyWithinSubclient(operator.getRole() == AGENCY)
                .withCanCopyCampaignsWithReport(hasOneOfRoles(operator,
                        ROLES_ALLOWED_TO_COPY_WITH_REPORT.toArray(new RbacRole[0])))
                .withCanCopyCampaignConversionStrategy(hasOneOfRoles(operator,
                        ROLES_ALLOWED_TO_COPY_CONVERSION_STRATEGY.toArray(new RbacRole[0])))
                .withCanStopCopiedCampaigns(hasOneOfRoles(operator,
                        ROLES_ALLOWED_TO_STOP_COPIED_CAMPAIGNS.toArray(new RbacRole[0])))
                .withCanAddTechAppMetrikaGoals(hasOneOfRoles(operator, MANAGER, SUPER));
    }

    GdAvailableFields getAvailableFields() {
        return new GdAvailableFields()
                .withCampaignFields(getGdCampaignsAvailableFields());
    }

    private Set<GdCampaignsAvailableFields> getGdCampaignsAvailableFields() {
        return StreamEx.ofKeys(CAMPAIGNS_AVAILABLE_FIELDS_ATTRIBUTE_MAP)
                .filter(field -> attributeResolverService.resolve(CAMPAIGNS_AVAILABLE_FIELDS_ATTRIBUTE_MAP.get(field)))
                .toSet();
    }

    OfficeContact getOfficeContact(@Nullable Long countryRegionId) {
        Optional<String> remoteHost = HostUtils.getRemoteHost();
        Optional<YandexDomain> yandexDomain = HostUtils.getYandexDomain(remoteHost.orElse(null));
        Long currentGeoRegionId = HttpUtil.getCurrentGeoRegionId().orElse(null);
        Long regionId = geoBaseHelper.convertToDirectRegionId(currentGeoRegionId).orElse(GLOBAL_REGION_ID);

        Locale locale = getCurrentLocale()
                .orElse(translationService.getLocale());
        Language language = Language.fromLocale(locale);

        return clientOfficeService.getOfficeContactForFooter(regionId, language, yandexDomain.orElse(null),
                countryRegionId);
    }

    public Map<Long, GdUserInfo> getUsersByIds(Collection<Long> userIds) {
        if (userIds.isEmpty()) {
            return Collections.emptyMap();
        }

        return userService.massGetUser(userIds).stream()
                .map(UserDataConverter::toGdUserInfo)
                .collect(Collectors.toMap(GdUserInfo::getUserId, Function.identity()));
    }

    GdOfficeInfo toGdOfficeInfo(OfficeContact officeContact) {
        GdOfficeInfo gdOfficeInfo = new GdOfficeInfo();

        if (officeContact.getCityContact() != null) {
            CityContact cityContact = officeContact.getCityContact();
            gdOfficeInfo.setCityContact(new GdOfficeContactData()
                    .withPhone(cityContact.getPhone())
                    .withPhoneExtension(cityContact.getPhoneExtension())
                    .withCityName(translationService.translate(cityContact.getCity()))
                    .withDescription(translationService.translate(cityContact.getDescription()))
            );
        }

        if (officeContact.getRegionContact() != null) {
            RegionContact regionContact = officeContact.getRegionContact();
            gdOfficeInfo.setRegionContact(new GdOfficeContactData()
                    .withPhone(regionContact.getPhone())
                    .withPhoneExtension(regionContact.getPhoneExtension())
                    .withDescription(translationService.translate(regionContact.getDescription()))
            );
        }

        if (officeContact.getRegionContact2() != null) {
            RegionContact regionContact2 = officeContact.getRegionContact2();
            gdOfficeInfo.setRegionContact2(new GdOfficeContactData()
                    .withPhone(regionContact2.getPhone())
                    .withPhoneExtension(regionContact2.getPhoneExtension())
                    .withDescription(translationService.translate(regionContact2.getDescription()))
            );
        }

        return gdOfficeInfo;
    }

    boolean shouldOpenFeedbackForm(User operator) {
        var uid = operator.getUid();
        if (uid == null) {
            return false;
        }
        var settings = jsonSettingsService.getFeedbackFormSessionsData(uid);
        var currentDateTime = LocalDateTime.now();
        var currentDate = currentDateTime.toLocalDate();
        var sessions = new HashSet<>(List.of(currentDate));
        if (settings.getOldestSessionDate() != null) {
            sessions.add(settings.getOldestSessionDate());
        }
        if (settings.getSecondSessionDate() != null) {
            sessions.add(settings.getSecondSessionDate());
        }
        if ((sessions.size() > 2)
                && (Boolean.TRUE.equals(settings.getRateLater())
                || settings.getLastShowTime() == null
                || settings.getLastShowTime().isBefore(currentDateTime.minusWeeks(6L)))) {
            jsonSettingsService.feedbackMarkAsShown(uid, currentDateTime);
            LOGGER.info("Should open the feedback form");
            return true;
        }

        // проверяем, не новая ли текущая дата сессии
        if (sessions.size() == 2 && settings.getSecondSessionDate() == null) {
            // записываем дату первой сессии тоже на всякий невероятный случай гонки
            jsonSettingsService.feedbackWriteSessionsDates(uid, settings.getOldestSessionDate(), currentDate);
        } else if (sessions.size() == 1 && settings.getOldestSessionDate() == null) {
            jsonSettingsService.feedbackWriteSessionsDates(uid, currentDate, null);
        }

        LOGGER.info("Shouldn't open the feedback form");
        return false;
    }

    void feedbackRateLater(User operator) {
        if (operator.getUid() == null) {
            return;
        }
        jsonSettingsService.feedbackRateLater(operator.getUid());
    }

    void feedbackMarkAsShown(User operator) {
        if (operator.getUid() == null) {
            return;
        }
        jsonSettingsService.feedbackMarkAsShown(operator.getUid(), LocalDateTime.now());
    }
}
