package ru.yandex.direct.grid.processing.service.organizations;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.grid.model.GdTime;
import ru.yandex.direct.grid.processing.model.cliententity.vcard.GdAddress;
import ru.yandex.direct.grid.processing.model.cliententity.vcard.GdPhone;
import ru.yandex.direct.grid.processing.model.cliententity.vcard.GdPointOnMap;
import ru.yandex.direct.grid.processing.model.cliententity.vcard.GdWorkTime;
import ru.yandex.direct.grid.processing.model.organizations.GdOrganization;
import ru.yandex.direct.grid.processing.model.organizations.GdOrganizationAccess;
import ru.yandex.direct.grid.processing.model.organizations.GdOrganizationStatusPublish;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.organizations.swagger.OrganizationInfo;
import ru.yandex.direct.organizations.swagger.OrganizationInfoConverters;
import ru.yandex.direct.organizations.swagger.model.Address;
import ru.yandex.direct.organizations.swagger.model.AddressComponent;
import ru.yandex.direct.organizations.swagger.model.CompanyPhone;
import ru.yandex.direct.organizations.swagger.model.Coordinates;
import ru.yandex.direct.organizations.swagger.model.LocalizedString;
import ru.yandex.direct.organizations.swagger.model.Point2;
import ru.yandex.direct.organizations.swagger.model.PubApiExtraCompanyField;
import ru.yandex.direct.organizations.swagger.model.WorkInterval;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static ru.yandex.direct.organizations.swagger.model.AddressComponent.KindEnum.COUNTRY;
import static ru.yandex.direct.organizations.swagger.model.AddressComponent.KindEnum.HOUSE;
import static ru.yandex.direct.organizations.swagger.model.AddressComponent.KindEnum.LOCALITY;
import static ru.yandex.direct.organizations.swagger.model.AddressComponent.KindEnum.METRO;
import static ru.yandex.direct.organizations.swagger.model.AddressComponent.KindEnum.STREET;
import static ru.yandex.direct.utils.CollectionUtils.isEmpty;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapAndFilterList;

@ParametersAreNonnullByDefault
public class OrganizationsConverter {
    private static final Logger logger = LoggerFactory.getLogger(OrganizationsConverter.class);
    private static final Set<PubApiExtraCompanyField> MANDATORY_FIELDS_IN_REQUEST =
            Set.of(PubApiExtraCompanyField.URLS, PubApiExtraCompanyField.PUBLISHING_STATUS);
    private static final Map<ModelProperty<?, ?>, Set<PubApiExtraCompanyField>> GD_FIELD_TO_SPRAV_FIELDS =
            ImmutableMap.<ModelProperty<?, ?>, Set<PubApiExtraCompanyField>>builder()
                    .put(GdOrganization.PERMALINK_ID, Set.of())//всегда есть
                    .put(GdOrganization.STATUS_PUBLISH, Set.of())//есть в MANDATORY_FIELDS_IN_REQUEST
                    .put(GdOrganization.CABINET_URL, Set.of(PubApiExtraCompanyField.LK_LINK))
                    .put(GdOrganization.PROFILE_URL, Set.of(PubApiExtraCompanyField.BUSINESS_PROFILE_LINK))
                    .put(GdOrganization.PREVIEW_HREF,
                            Set.of(PubApiExtraCompanyField.BUSINESS_PROFILE_LINK/*, PubApiExtraCompanyField.LOGO DIRECT-134713*/)
                    )
                    .put(GdOrganization.COMPANY_NAME, Set.of(PubApiExtraCompanyField.NAMES))
                    .put(GdOrganization.CHAIN_ID, Set.of(PubApiExtraCompanyField.CHAIN))
                    .put(GdOrganization.EMAIL, Set.of(PubApiExtraCompanyField.EMAILS))
                    .put(GdOrganization.PHONE, Set.of(PubApiExtraCompanyField.PHONES))
                    .put(GdOrganization.PHONES, Set.of(PubApiExtraCompanyField.PHONES))
                    .put(GdOrganization.ADDRESS, Set.of(PubApiExtraCompanyField.ADDRESS))
                    .put(GdOrganization.RUBRIC, Set.of(PubApiExtraCompanyField.RUBRICS, PubApiExtraCompanyField.RUBRIC_DEFS))
                    .put(GdOrganization.URLS, Set.of())//есть в MANDATORY_FIELDS_IN_REQUEST
                    .put(GdOrganization.WORK_TIMES, Set.of(PubApiExtraCompanyField.WORK_INTERVALS))
                    .put(GdOrganization.IS_ONLINE, Set.of(PubApiExtraCompanyField.IS_ONLINE))
                    .put(GdOrganization.ACCESS, Set.of())//делается подзапросами, сам GdOrganizationAccess пока пустой
                    .put(GdOrganization.COUNTER_ID, Set.of(PubApiExtraCompanyField.METRIKA_DATA))
                    .build();

    static {
        checkState(GD_FIELD_TO_SPRAV_FIELDS.keySet().containsAll(GdOrganization.allModelProperties()),
                "has to contain all GdOrganization properties");
    }

    private OrganizationsConverter() {
    }

    /**
     *
     * @param fieldsToFetch по факту набор ModelProperty<GdOrganization, ?>
     * @return набор запрашиваемых у Справочника полей
     */
    public static Set<PubApiExtraCompanyField> getRequestedSpravFields(Set<ModelProperty<?, ?>> fieldsToFetch) {
        return StreamEx.of(fieldsToFetch)
                .map(GD_FIELD_TO_SPRAV_FIELDS::get)
                .nonNull()
                .flatMap(StreamEx::of)
                .append(MANDATORY_FIELDS_IN_REQUEST)
                .toSet();
    }

    /**
     * Конвертирует данные по организации из формата клиента к API справочника в формат интерфейса
     */
    public static GdOrganization toGdOrganization(OrganizationInfo organizationInfo) {
        Long counterId = OrganizationInfoConverters.getMetrikaCounterId(organizationInfo.getMetrikaData());
        return new GdOrganization()
                .withPermalinkId(organizationInfo.getPermalinkId())
                .withChainId(organizationInfo.getChainId())
                .withEmail(organizationInfo.getEmail())
                .withCompanyName(organizationInfo.getCompanyName())
                .withStatusPublish(GdOrganizationStatusPublish.fromSource(organizationInfo.getStatusPublish()))
                .withAddress(toGdAddress(organizationInfo.getAddress()))
                .withPhone(toGdPhone(organizationInfo.getPhone()))
                .withPhones(toGdPhones(organizationInfo.getPhones()))
                .withWorkTimes(toGdWorkTimes(organizationInfo.getWorkIntervals()))
                .withRubric(organizationInfo.getRubric())
                .withCabinetUrl(organizationInfo.getCabinetUrl())
                .withProfileUrl(organizationInfo.getProfileUrl())
                .withUrls(organizationInfo.getUrls())
                .withIsOnline(organizationInfo.getIsOnline())
                .withAccess(
                        new GdOrganizationAccess().withPermalinkId(organizationInfo.getPermalinkId())
                )
                .withPreviewHref(organizationInfo.getPreviewHref())
                .withCounterId(counterId);
    }

    /**
     * Собрать и частично заполнить (по мере необходимости) OrganizationInfo из гридовой модели
     * @param gdOrganization — гридовая модель организации
     * @return частично заполненная ядровая модель организации
     */
    public static OrganizationInfo fromGdOrganizationPartial(GdOrganization gdOrganization) {
        var org = new OrganizationInfo()
                .withCompanyName(gdOrganization.getCompanyName())
                .withPhones(fromGdPhones(gdOrganization.getPhones()))
                .withUrls(List.of())
                .withWorkIntervals(List.of());
        org.setPermalinkId(gdOrganization.getPermalinkId());
        return org;
    }

    static CompanyPhone fromGdPhone(@Nullable GdPhone gdPhone) {
        if (gdPhone == null) {
            return null;
        }
        var phone = new CompanyPhone();
        phone.setCountryCode(nvl(gdPhone.getCountryCode(), ""));
        phone.setRegionCode(nvl(gdPhone.getCityCode(), ""));
        phone.setNumber(gdPhone.getPhoneNumber());
        phone.setExt(gdPhone.getExtension());
        return phone;
    }

    static List<CompanyPhone> fromGdPhones(@Nullable List<GdPhone> gdPhones) {
        if (isEmpty(gdPhones)) {
            return List.of();
        }
        return mapAndFilterList(gdPhones, OrganizationsConverter::fromGdPhone, Objects::nonNull);
    }


    /**
     * Конвертирует данные по времени работы организации из формата клиента к API справочника в формат интерфейса
     */
    @Nonnull
    static List<GdWorkTime> toGdWorkTimes(@Nullable Collection<WorkInterval> workIntervals) {
        if (workIntervals == null || workIntervals.isEmpty()) {
            return emptyList();
        }

        return StreamEx.of(workIntervals)
                .map(i -> new GdWorkTime()
                        .withDaysOfWeek(toDayNumbers(i.getDay()))
                        .withStartTime(toGdTime(i.getTimeMinutesBegin()))
                        .withEndTime(toGdTime(i.getTimeMinutesEnd())))
                .nonNull()
                .toList();
    }

    /**
     * Конвертирует время в минутах от начала суток в формат интерфейса
     *
     * @param minutes время в минутах от начала суток
     */
    static GdTime toGdTime(@Nullable Long minutes) {
        if (minutes == null) {
            return null;
        }

        long hour = TimeUnit.HOURS.convert(minutes, TimeUnit.MINUTES);
        return new GdTime()
                .withHour((int) hour)
                .withMinute(minutes.intValue() - (int) hour * 60);
    }

    static List<Integer> toDayNumbers(@Nullable WorkInterval.DayEnum day) {
        if (day == null) {
            // Everyday by default
            return List.of(0, 1, 2, 3, 4, 5, 6);
        }

        switch (day) {
            case MONDAY:
                return singletonList(0);
            case TUESDAY:
                return singletonList(1);
            case WEDNESDAY:
                return singletonList(2);
            case THURSDAY:
                return singletonList(3);
            case FRIDAY:
                return singletonList(4);
            case SATURDAY:
                return singletonList(5);
            case SUNDAY:
                return singletonList(6);
            case WEEKDAYS:
                return List.of(0, 1, 2, 3, 4);
            case WEEKEND:
                return List.of(5, 6);
            case EVERYDAY:
                return List.of(0, 1, 2, 3, 4, 5, 6);
            default:
                throw new RuntimeException("Unknown day enum member, regenerate models: " + day.getValue());
        }
    }

    /**
     * Конвертирует данные по номеру телефона организации из формата клиента к API справочника в формат интерфейса
     */
    static GdPhone toGdPhone(@Nullable CompanyPhone phone) {
        if (phone == null) {
            return null;
        }
        return new GdPhone()
                .withCountryCode(nvl(phone.getCountryCode(), ""))
                .withCityCode(nvl(phone.getRegionCode(), ""))
                .withPhoneNumber(phone.getNumber())
                .withExtension(phone.getExt());
    }

    static List<GdPhone> toGdPhones(List<CompanyPhone> phones) {
        if (isEmpty(phones)) {
            return emptyList();
        }
        return mapAndFilterList(phones, OrganizationsConverter::toGdPhone, Objects::nonNull);
    }

    /**
     * Конвертирует данные по адресу организации из формата клиента к API справочника в формат интерфейса
     */
    static GdAddress toGdAddress(@Nullable Address address) {
        if (address == null) {
            return null;
        }

        var result = new GdAddress();
        var components = StreamEx.of(address.getComponents())
                .mapToEntry(AddressComponent::getKind, AddressComponent::getName)
                .mapValues(LocalizedString::getValue)
                .toMap((s, s2) -> s + ", " + s2);

        return result
                .withCountry(nvl(components.get(COUNTRY), ""))
                .withCity(nvl(components.get(LOCALITY), ""))
                .withStreet(components.get(STREET))
                .withHouseWithBuilding(components.get(HOUSE))
                .withMetroStationName(components.get(METRO))
                .withGeoId(Optional.ofNullable(address.getGeoId()).map(Long::intValue).orElse(null))
                .withPointOnMap(toGdPointOnMap(address.getPos()));
    }

    static GdPointOnMap toGdPointOnMap(@Nullable Point2 pos) {
        if (pos == null) {
            return null;
        }
        Coordinates coordinates = pos.getCoordinates();
        if (coordinates.size() != 2 || StreamEx.of(coordinates).nonNull().count() != 2) {
            logger.warn("Coordinates should have size = 2, got {}. {}", coordinates.size(), coordinates);
            return null;
        }

        return new GdPointOnMap()
                .withX(BigDecimal.valueOf(coordinates.get(0)))
                .withY(BigDecimal.valueOf(coordinates.get(1)));
    }
}
